/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * http://aipostyle.com/
 * 
 * Copyright(C) 2011 avanza Co.,Ltd. All rights reserved.
 * http://www.avnz.co.jp/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package com.aimluck.eip.account;

import java.util.Calendar;
import java.util.Date;
import java.util.List;

import org.apache.cayenne.access.DataContext;
import org.apache.cayenne.access.Transaction;
import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.account.util.AccountUtils;
import com.aimluck.eip.cayenne.om.account.AvzMGroupsend;
import com.aimluck.eip.cayenne.om.account.AvzMGroupsendPosition;
import com.aimluck.eip.cayenne.om.account.AvzMUserGroupsend;
import com.aimluck.eip.common.ALAbstractFormData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALEipManager;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.ResultList;
import com.aimluck.eip.orm.query.SelectQuery;

/**
 * グループ送信登録フォームデータクラスです。
 * 
 */
public class AccountGroupSendFormData extends ALAbstractFormData {

  /** logger */
  private static final JetspeedLogger logger =
    JetspeedLogFactoryService.getLogger(AccountGroupSendFormData.class
      .getName());

  /** グループ送信名 */
  private ALStringField groupsend_name;

  /** 部署グループか？ */
  private ALStringField is_category_post;

  /**
   * 初期化します。
   * 
   * @param action
   *            アクションクラス
   * @param rundata
   *            JetSpeedランデータ
   * @param context
   *            JetSpeedコンテキスト
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    super.init(action, rundata, context);
  }

  /**
   * 画面フィールド初期化
   * 
   * @see com.aimluck.eip.common.ALData#initField()
   */
  public void initField() {
    groupsend_name = new ALStringField();
    groupsend_name.setFieldName("グループ送信名");
    groupsend_name.setTrim(true);

    // 管理者権限を付与するか
    is_category_post = new ALStringField();
    is_category_post.setFieldName("部署グループとする");
    is_category_post.setTrim(true);
  }

  /**
   * バリデーション初期化
   * 
   * @see com.aimluck.eip.common.ALAbstractFormData#setValidator()
   */
  @Override
  protected void setValidator() {
    groupsend_name.setNotNull(true);
    groupsend_name.limitMaxLength(50);
  }

  /**
   * バリデーション
   * 
   * @param msgList
   *            エラーメッセージリスト
   * @return 処理結果(正常：true、異常:false)
   * 
   * @see com.aimluck.eip.common.ALAbstractFormData#validate(java.util.ArrayList)
   */
  @Override
  protected boolean validate(List<String> msgList) {
    groupsend_name.validate(msgList);
    return msgList.size() == 0;
  }

  /**
   * グループ送信情報編集画面表示処理
   * 
   * @param rundata
   *            JetSpeedランデータ
   * @param context
   *            JetSpeedコンテキスト
   * @param msgList
   *            エラーメッセージリスト
   * @return 処理結果(正常：true、異常:false)
   * 
   * @throws ALDBErrorException
   * @see com.aimluck.eip.common.ALAbstractFormData#loadFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean loadFormData(RunData rundata, Context context,
      List<String> msgList) throws ALDBErrorException {
    try {
      AvzMGroupsend entity = AccountUtils.getAvzMGroupsend(rundata, context);
      if (entity == null) {
        throw new Exception("指定されたグループ送信がデータベースにありません。");
      }
      groupsend_name.setValue(entity.getGroupsendName());
      if (AvzMGroupsend.GROUPSEND_CATEGORY_POST.equals(entity.getCategory())) {
        is_category_post.setValue("true");
      } else {
        is_category_post.setValue("false");
      }

    } catch (Exception ex) {
      String msg = "グループ送信情報の編集画面の表示に失敗しました。";
      logger.error(msg, ex);
      throw new ALDBErrorException();
    }
    return true;
  }

  /**
   * グループ送信情報登録処理
   * 
   * @param rundata
   *            JetSpeedランデータ
   * @param context
   *            JetSpeedコンテキスト
   * @param msgList
   *            エラーメッセージリスト
   * @return 処理結果(正常：true、異常:false)
   * 
   * @see com.aimluck.eip.common.ALAbstractFormData#insertFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean insertFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      Date now = Calendar.getInstance().getTime();

      // 新規オブジェクトモデル
      AvzMGroupsend entity = Database.create(AvzMGroupsend.class);
      entity.setGroupsendName(groupsend_name.getValue());

      if (is_category_post.getValue() != null) {
        if (is_category_post.getValue().equals("true")) {
          entity.setCategory(AvzMGroupsend.GROUPSEND_CATEGORY_POST);
        } else {
          entity.setCategory("");
        }
      }
      entity.setCreateDate(now);
      entity.setUpdateDate(now);

      // add by motegi start グループ送信順番に登録
      List<AvzMGroupsendPosition> posposlist =
        Database.query(AvzMGroupsendPosition.class).fetchList();
      int new_pos =
        (posposlist != null && posposlist.size() > 0)
          ? posposlist.size() + 1
          : 1;
      AvzMGroupsendPosition p = Database.create(AvzMGroupsendPosition.class);
      p.setToAvzMGroupsend(entity);
      p.setPosition(Integer.valueOf(new_pos));
      // add end

      Database.commit();
      ALEipManager.getInstance().reloadGroupSend();

    } catch (Exception ex) {
      String msg = "グループ送信情報の登録に失敗しました。";
      logger.error(msg, ex);
      msgList.add(msg);
      Database.rollback();
      return false;
    }
    return true;
  }

  /**
   * グループ送信情報更新処理
   * 
   * @param rundata
   *            JetSpeedランデータ
   * @param context
   *            JetSpeedコンテキスト
   * @param msgList
   *            エラーメッセージリスト
   * @return 処理結果(正常：true、異常:false)
   * 
   * @see com.aimluck.eip.common.ALAbstractFormData#updateFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean updateFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      AvzMGroupsend entity = AccountUtils.getAvzMGroupsend(rundata, context);
      if (entity == null) {
        throw new Exception("指定されたグループ送信がデータベースにありません。");
      }
      entity.setGroupsendName(groupsend_name.getValue());
      if (is_category_post.getValue() != null) {
        if (is_category_post.getValue().equals("true")) {
          entity.setCategory(AvzMGroupsend.GROUPSEND_CATEGORY_POST);
        } else {
          entity.setCategory("");
        }
      }
      entity.setUpdateDate(Calendar.getInstance().getTime());
      Database.commit();
      ALEipManager.getInstance().reloadGroupSend();

    } catch (Exception ex) {
      String msg = "グループ送信情報の更新に失敗しました。";
      logger.error(msg, ex);
      msgList.add(msg);
      Database.rollback();
      return false;
    }
    return true;
  }

  /**
   * グループ送信情報削除処理
   * 
   * @param rundata
   *            JetSpeedランデータ
   * @param context
   *            JetSpeedコンテキスト
   * @param msgList
   *            エラーメッセージリスト
   * @return 処理結果(正常：true、異常:false)
   * 
   * @see com.aimluck.eip.common.ALAbstractFormData#deleteFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean deleteFormData(RunData rundata, Context context,
      List<String> msgList) {

    // 標準のトランザクション（オートコミット）
    Transaction baseTx = Transaction.getThreadTransaction();

    DataContext dataContext = DataContext.getThreadDataContext();

    // 自己管理トランザクション
    Transaction tx =
      Transaction.internalTransaction(DataContext
        .getThreadDataContext()
        .getParentDataDomain()
        .getTransactionDelegate());

    // 標準のトランザクションを自己管理トランザクションに置き換えます。
    Transaction.bindThreadTransaction(tx);

    try {

      // トランザクション開始
      tx.begin();

      AvzMGroupsend entity = AccountUtils.getAvzMGroupsend(rundata, context);
      if (entity == null) {
        throw new Exception("指定されたグループ送信がデータベースにありません。");
      }

      SelectQuery<AvzMUserGroupsend> query =
        Database.query(AvzMUserGroupsend.class);
      Expression exp =
        ExpressionFactory.matchExp(
          AvzMUserGroupsend.GROUPSEND_ID_PROPERTY,
          entity.getGroupsendId());
      query.setQualifier(exp);
      ResultList<AvzMUserGroupsend> list = query.getResultList();
      if (list != null && list.size() != 0) {
        Database.deleteAll(list);
      }
      Database.delete(entity.getAvzMGroupsendPositionArray());
      Database.delete(entity);
      dataContext.commitChanges();

      // 他のグループ送信の順番を変更する．
      SelectQuery<AvzMGroupsendPosition> p_query =
        Database.query(AvzMGroupsendPosition.class);
      p_query.orderAscending(AvzMGroupsendPosition.POSITION_PROPERTY);
      List<AvzMGroupsendPosition> entityPositions = p_query.fetchList();

      if (entityPositions != null && entityPositions.size() > 0) {

        int size = entityPositions.size();
        for (int i = 0; i < size; i++) {
          AvzMGroupsendPosition entityPosition = entityPositions.get(i);
          entityPosition.setPosition(Integer.valueOf(i + 1));
        }
      }
      Database.commit();
      ALEipManager.getInstance().reloadGroupSend();
    } catch (Exception ex) {
      String msg = "グループ送信情報の削除処理に失敗しました。";
      logger.error(msg, ex);
      msgList.add(msg);
      Database.rollback();
      return false;
    } finally {
      // 自己管理トランザクションを標準のトランザクションに戻します。
      Transaction.bindThreadTransaction(baseTx);
    }
    return true;
  }

  /**
   * グループ送信名取得
   * 
   * @return グループ送信名
   */
  public ALStringField getGroupsendName() {
    return groupsend_name;
  }

  /**
   * グループ送信種別取得処理
   * 
   * @return true:部署グループ false:部署グループ以外
   */
  public ALStringField getIsCategoryPost() {
    return is_category_post;
  }

  /**
   * ※未使用
   */
  @Override
  public String getAclPortletFeature() {
    return null;
  }
}
