/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * http://aipostyle.com/
 *
 * Copyright(C) 2011 avanza Co.,Ltd. All rights reserved.
 * http://www.avnz.co.jp/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.webmail;

import java.io.File;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALDateField;
import com.aimluck.commons.field.ALIllegalDateException;
import com.aimluck.commons.field.ALNumberField;
import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.cayenne.om.portlet.AvzTMailBatch;
import com.aimluck.eip.cayenne.om.portlet.EipMMailAccount;
import com.aimluck.eip.cayenne.om.portlet.EipTMailFolder;
import com.aimluck.eip.common.ALAbstractFormData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.mail.util.ALMailUtils;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.util.ALEipUtils;
import com.aimluck.eip.webmail.util.WebMailUtils;

/**
 * メール一括処理のフォームデータを管理するクラスです。
 * 
 */
public class WebMailPackageProcessFormData extends ALAbstractFormData {

  /** logger */
  private static final JetspeedLogger logger =
    JetspeedLogFactoryService.getLogger(WebMailPackageProcessFormData.class
      .getName());

  /** 対象期間開始日 */
  private ALDateField start_date;

  /** 対象期間終了日 */
  private ALDateField end_date;

  /** 対象フォルダID */
  private ALNumberField target_folder_id;

  /** 移動先フォルダID */
  private ALNumberField move_folder_id;

  /** 操作種別タイプ */
  private ALNumberField operation_type;

  /** 期間指定なしフラグ */
  private ALStringField no_span;

  /** ログインユーザーのID * */
  private int user_id;

  /** ログインユーザーのアカウントID * */
  private int account_id;

  /** ログインユーザーのメールアカウント * */
  private EipMMailAccount mailAccount;

  /** ログインユーザー名 * */
  private String login_user_name;

  /** ページタイプ * */
  private String page_view_type;

  /** hiddenパラメータ名:ページ種別 * */
  private final String PAGE_VIEW_TYPE = "page_view_type";

  /** page_view_type(AIPO_MAIL_PC_008)用文字列 * */
  private final String SAVE_PROCESS = "save";

  /** page_view_type(AIPO_MAIL_PC_009)用文字列 * */
  private final String CANCEL_PROCESS = "cancel";

  /** page_view_type(AIPO_MAIL_PC_010)用文字列 * */
  private final String DOWNLOAD_PROCESS = "download";

  /** page_view_type* */
  private final String DOWNLOAD_FINISH = "downloadfinish";

  /** フォルダ選択プルダウン用リスト * */
  // public List<WebMailFolderResultData> folderList;
  public List<ExtFolderInfo> folderList;

  /** 退避対象ダウンロードフォルダ名 * */
  private String save_folder_name;

  /** ログ出力用メッセージ(一括処理ステータス更新) * */
  private final String ERROR_MESSAGE_PROCESS_STATUS_UPDATE =
    "一括処理ステータス更新に失敗しました";

  /** ログ出力用メッセージ(一括処理初期表示) * */
  private final String ERROR_MESSAGE_PROCESS_SELECT_FORM = "一括処理選択初期表示に失敗しました";

  /** ログ出力用メッセージ(一括処理初期表示) * */
  private final String ERROR_MESSAGE_PROCESS_START_CHECK = "一括処理開始チェックに失敗しました";

  /** ログ出力用メッセージ(退避ファイルダウンロード初期表示) * */
  private final String ERROR_MESSAGE_PROCESS_DOWNLOAD_FORM =
    "退避ファイルダウンロード初期表示に失敗しました";

  /** ログ出力用メッセージ(一括処理呼出) * */
  private final String ERROR_MESSAGE_ASYNCHRONOUS_PROCESS = "一括処理呼出に失敗しました";

  /**
   * 初期処理
   * 
   * @param action
   *            アクションクラス
   * @param rundata
   *            JetSpeedランデータ
   * @param context
   *            JetSpeedコンテキスト
   * @throws ALPageNotFoundException
   * @throws ALDBErrorException
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    super.init(action, rundata, context);
    // ログインユーザーID取得
    user_id = ALEipUtils.getUserId(rundata);
    // ログインユーザー名取得
    login_user_name = ALEipUtils.getUserFullName(user_id);

    try {
      // アカウントID取得
      String tmpAccoundId =
        ALEipUtils.getTemp(rundata, context, WebMailUtils.ACCOUNT_ID);
      if (tmpAccoundId == null || "".equals(tmpAccoundId)) {
        ALEipUtils.setTemp(
          rundata,
          context,
          WebMailUtils.ACCOUNT_ID,
          ALEipUtils
            .getPortlet(rundata, context)
            .getPortletConfig()
            .getInitParameter("p3a-accounts"));
      }
      // アカウントID取得
      account_id =
        Integer.parseInt(ALEipUtils.getTemp(
          rundata,
          context,
          WebMailUtils.ACCOUNT_ID));
      // メールアカウント取得
      mailAccount = ALMailUtils.getMailAccount(user_id, account_id);
      if (mailAccount == null) {
        return;
      }

      if ((null == (rundata.getParameters().get(PAGE_VIEW_TYPE)))
        || "".equals(rundata.getParameters().get(PAGE_VIEW_TYPE))) {
        // 「リクエストからPAGE_VIEW_TYPEが取得できない場合」
        // 受信トレイ画面(AIPO_MAIL_PC_001)からの遷移であり
        // 処理ステータスを取得し、ステータスごとにpage_view_typeを切り替える。
        loadPageViewType();
        page_view_type = getPageViewType();
        if (page_view_type.equals(SAVE_PROCESS)) {
          // フォルダ選択用プルダウン情報の読み込み
          loadMailFolderList(rundata, context);
        } else if (page_view_type.equals(DOWNLOAD_PROCESS)
        // add start
          // 受入障害対応No.157
          || page_view_type.equals(DOWNLOAD_FINISH)) {
          // add end
          // ダウンロードフォルダ情報の読み込み
          loadDownLoadMailFolderName(rundata, context);
        }
      }
    } catch (Exception ex) {
      logger.error(ERROR_MESSAGE_PROCESS_SELECT_FORM + login_user_name, ex);
      return;
    }
  }

  /**
   * 画面フィールド初期化
   * 
   * @see com.aimluck.eip.common.ALData#initField()
   */
  @Override
  public void initField() {
    // 対象期間開始日
    start_date = new ALDateField();
    start_date.setFieldName("開始日付");
    start_date.setValue(new Date());
    // 対象期間終了日
    end_date = new ALDateField();
    end_date.setFieldName("終了日付");
    end_date.setValue(new Date());
    // 対象フォルダID
    target_folder_id = new ALNumberField();
    target_folder_id.setFieldName("対象フォルダ");
    // 移動先フォルダID
    move_folder_id = new ALNumberField();
    move_folder_id.setFieldName("移動先");
    // 期間指定なしフラグ
    no_span = new ALStringField();
    no_span.setFieldName("期間指定なし");
    no_span.setValue("false");
    // 操作種別
    operation_type = new ALNumberField();
    operation_type.setFieldName("操作");
  }

  /**
   * バリデータ初期化
   * 
   * @see com.aimluck.eip.common.ALAbstractFormData#setValidator()
   */
  @Override
  protected void setValidator() {

  }

  /**
   * バリデーション
   * 
   * @param msgList
   *            エラーメッセージリスト
   * @return 処理結果(正常：true、異常:false)
   * @see com.aimluck.eip.common.ALAbstractFormData#validate(java.util.ArrayList)
   */
  @Override
  protected boolean validate(List<String> msgList) {
    // 対象期間チェックを行う
    if (null != no_span.getValue()) {
      try {
        if (start_date
          .getValue()
          .getDate()
          .after(end_date.getValue().getDate())) {
          msgList.add("『 <span class='em'>"
            + end_date.getFieldName()
            + "</span> 』は『 <span class='em'>"
            + start_date.getFieldName()
            + "</span> 』以降の日付を指定してください。");
        }
      } catch (ALIllegalDateException ad) {
        logger.error(ERROR_MESSAGE_PROCESS_START_CHECK + login_user_name, ad);
      }
    }
    // 対象フォルダ選択必須チェックを行う
    if (null == target_folder_id
      || target_folder_id.getValueAsString().length() <= 0) {
      msgList.add("『 <span class='em'>"
        + target_folder_id.getFieldName()
        + "</span> 』を選択してください。");
    }
    // 移動先フォルダ選択必須チェックを行う(操作が一括フォルダ移動時)
    if (WebMailConsts.BATCH_TYPE_MOVE.equals(operation_type.getValueAsString())) {
      if (null == move_folder_id
        || move_folder_id.getValueAsString().length() <= 0) {
        msgList.add("『 <span class='em'>"
          + move_folder_id.getFieldName()
          + "</span> 』を選択してください。");
      } else {
        try {
          // 移動先フォルダチェックを行う
          if (target_folder_id.getValue() == move_folder_id.getValue()) {
            msgList.add("<span class='em'>移動先は対象フォルダ以外を選択してください。</span>");
          }
          if (!isMoveFoLder(account_id, target_folder_id, move_folder_id)) {
            msgList.add("<span class='em'>受信トレイと送信トレイを越える移動はできません。</span>");
          }
        } catch (Exception ex) {
          logger.error(ERROR_MESSAGE_PROCESS_START_CHECK + login_user_name, ex);
        }
      }
    }
    // 他の一括処理が行われているかチェックを行う
    try {
      if (isOtherProcess()) {
        msgList.add("<span class='em'>他の一括処理が終了するまでお待ちください。</span>");
      }
    } catch (Exception ex) {
      logger.error(ERROR_MESSAGE_PROCESS_START_CHECK + login_user_name, ex);
    }
    return (msgList.size() == 0);
  }

  /**
   * 抽象クラスのloadFormDataメソッドをオーバライドする必要があるため、 処理がなくても記述する必要がある。
   * 
   * @param rundata
   *            JetSpeedランデータ
   * @param context
   *            JetSpeedコンテキスト
   * @param msgList
   *            エラーメッセージリスト
   * @return 処理結果(正常：true、異常:false)
   * @see com.aimluck.eip.common.ALAbstractFormData#loadFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean loadFormData(RunData rundata, Context context,
      List<String> msgList) {
    return true;
  }

  /**
   * 
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean deleteFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {

    } catch (Throwable t) {
      Database.rollback();
      logger.error(t);
      return false;
    }
    return true;
  }

  /**
   * Webメール一括処理レコード作成処理
   * 
   * @param rundata
   *            JetSpeedランデータ
   * @param context
   *            JetSpeedコンテキスト
   * @param msgList
   *            エラーメッセージリスト
   * @return 処理結果(正常：true、異常:false)
   * @see com.aimluck.eip.common.ALAbstractFormData#insertFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean insertFormData(RunData rundata, Context context,
      List<String> msgList) throws ALDBErrorException {
    try {
      // 新規オブジェクトモデル
      AvzTMailBatch mailBatch = Database.create(AvzTMailBatch.class);

      Date now = Calendar.getInstance().getTime();
      String folder_kind =
        WebMailUtils.getEipTMailFolder(
          account_id,
          (int) target_folder_id.getValue()).getFolderKind();
      // ユーザーID
      mailBatch.setUserId(user_id);
      // アカウントID
      mailBatch.setAccountId(account_id);
      // フォルダ種別
      mailBatch.setFolderType(folder_kind);
      // 処理種別
      mailBatch.setType(operation_type.getValueAsString());
      // 処理ステータス
      mailBatch.setStatus(WebMailConsts.BATCH_PROCESSING);
      if ("false".equals(no_span.getValue())) {
        // 対象期間開始日付
        mailBatch.setFromDate(start_date.getValue().getDate());
        // 対象期間終了日付
        mailBatch.setToDate(end_date.getValue().getDate());
      }
      // 対象フォルダID
      mailBatch.setFolderId((int) (target_folder_id.getValue()));
      if (WebMailConsts.BATCH_TYPE_MOVE.equals(operation_type
        .getValueAsString())) {
        // 移動先フォルダID
        mailBatch.setToFolderId((int) (move_folder_id.getValue()));
      }
      // 作成日
      mailBatch.setCreateDate(now);
      // 更新日
      mailBatch.setUpdateDate(now);

      // add start by motegi 処理件数を先にチェック
      if (noFoundRecord(
        account_id,
        user_id,
        (int) target_folder_id.getValue(),
        mailBatch,
        folder_kind)) {
        msgList.add("指定された条件のメールは０件です。条件を変更下さい。");
        Database.rollback();
        return false;
      }
      // add end

      // 一括処理レコードを登録
      Database.commit();

    } catch (Exception t) {
      Database.rollback();
      logger.error(ERROR_MESSAGE_PROCESS_STATUS_UPDATE + login_user_name, t);
      return false;
    }
    return true;
  }

  /**
   * Webメール一括処理レコード更新処理
   * 
   * @param rundata
   *            JetSpeedランデータ
   * @param context
   *            JetSpeedコンテキスト
   * @param msgList
   *            エラーメッセージリスト
   * @return 処理結果(正常：true、異常:false)
   * @see com.aimluck.eip.common.ALAbstractFormData#updateFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean updateFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      AvzTMailBatch mailBatch =
        WebMailUtils.getAvzTMailBatchResultList(user_id);
      if (mailBatch == null) {
        // レコードが存在していない場合は
        // 一括処理レコードを新規作成
        this.insertFormData(rundata, context, msgList);
      } else {
        Date now = Calendar.getInstance().getTime();
        String folder_kind =
          WebMailUtils.getEipTMailFolder(
            account_id,
            (int) target_folder_id.getValue()).getFolderKind();
        // アカウントID
        mailBatch.setAccountId(account_id);
        // フォルダ種別
        mailBatch.setFolderType(folder_kind);
        // 処理種別
        mailBatch.setType(operation_type.getValueAsString());
        // 処理ステータス
        mailBatch.setStatus(WebMailConsts.BATCH_PROCESSING);

        mailBatch.setFromDate(null);
        mailBatch.setToDate(null);
        if ("false".equals(no_span.getValue())) {
          // 対象期間開始日付
          mailBatch.setFromDate(start_date.getValue().getDate());
          // 対象期間終了日付
          mailBatch.setToDate(end_date.getValue().getDate());
        }

        // 対象フォルダID
        mailBatch.setFolderId((int) (target_folder_id.getValue()));
        if (WebMailConsts.BATCH_TYPE_MOVE.equals(operation_type
          .getValueAsString())) {
          // 移動先フォルダID
          mailBatch.setToFolderId((int) (move_folder_id.getValue()));
        }
        // 更新日
        mailBatch.setUpdateDate(now);

        // add start by motegi 処理件数を先にチェック
        if (noFoundRecord(account_id, user_id, (int) target_folder_id
          .getValue(), mailBatch, folder_kind)) {
          msgList.add("指定された条件のメールは０件です。条件を変更下さい。");
          Database.rollback();
          return false;
        }
        // add end

        // 一括処理レコードを更新
        Database.commit();
      }
    } catch (Throwable t) {
      Database.rollback();
      logger.error(ERROR_MESSAGE_PROCESS_STATUS_UPDATE + login_user_name, t);
      return false;
    }
    // 非同期処理実行
    doAsynchronousProcess();
    return true;
  }

  // add start 20111109
  /**
   * 一括処理対象件数をチェック。
   * 
   * @param int
   *            account_id アカウントID
   * @param int
   *            user_id ユーザーID
   * @param int
   *            folder_id フォルダID
   * @param AvzTMailBatch
   *            mailBatch mailBatchオブジェクト
   * @param folder_kind
   *            フォルダ種別
   * @return 処理件数0件：true
   */
  private boolean noFoundRecord(int account_id, int user_id, int folder_id,
      AvzTMailBatch mailBatch, String folder_kind) {

    int count = 0;
    if (WebMailConsts.RECEIVE_FOLDER.equals(folder_kind)) {
      // フォルダ種別 = R:受信トレイの場合
      count =
        WebMailUtils.getEipTMailCount(
          account_id,
          user_id,
          (int) target_folder_id.getValue(),
          mailBatch);
    } else {
      // フォルダ種別 = S:送信トレイの場合
      count =
        WebMailUtils.getAvzTMailSendCount(
          account_id,
          user_id,
          (int) target_folder_id.getValue(),
          mailBatch);
    }

    logger.info("一括処理対象件数:"
      + count
      + " ["
      + ALEipUtils.getUserFullName(user_id)
      + "]");

    if (count == 0) {
      return true;
    }
    return false;
  }

  // add end

  /**
   * 期間終了日を取得します。
   * 
   * @return 期間終了日
   */
  public ALDateField getEndDate() {
    return end_date;
  }

  /**
   * 期間開始日を取得します。
   * 
   * @return 期間開始日
   */
  public ALDateField getStartDate() {
    return start_date;
  }

  /**
   * 処理対象フォルダIDを取得します。
   * 
   * @return 処理対象フォルダID
   */
  public ALNumberField getTargetFolderId() {
    return target_folder_id;
  }

  /**
   * 移動先フォルダIDを取得します。
   * 
   * @return 移動先フォルダID
   */
  public ALNumberField getMoveFolderId() {
    return move_folder_id;
  }

  /**
   * 
   * ログインユーザーのメールフォルダ一覧表示処理。
   * 
   * @param rundata
   * @param context
   */
  public void loadMailFolderList(RunData rundata, Context context) {
    if (mailAccount == null) {
      return;
    }
    try {
      // folderList = new ArrayList<WebMailFolderResultData>();
      //
      // // フォルダ一覧を取得する
      // List<EipTMailFolder> mailFolders =
      // ALMailUtils.getEipTMailFolderAll(mailAccount);
      // for (EipTMailFolder folder : mailFolders) {
      // WebMailFolderResultData rd = new WebMailFolderResultData();
      // // メールボックス（親フォルダID=0）は表示しない
      // if (folder.getParentFolderId() != 0) {
      // rd.initField();
      // rd.setFolderId(folder.getFolderId().intValue());
      // rd.setFolderName(folder.getFolderName());
      // folderList.add(rd);
      // }
      // }

      folderList = new ArrayList<ExtFolderInfo>();

      List<ExtFolderInfo> tmp =
        WebMailUtils.getFolderList(
          WebMailUtils.ROOT_PARENT_FODLER_ID,
          mailAccount.getAccountId(),
          WebMailConsts.RECEIVE_FOLDER);

      folderList.addAll(tmp);

      tmp =
        WebMailUtils.getFolderList(
          WebMailUtils.ROOT_PARENT_FODLER_ID,
          mailAccount.getAccountId(),
          WebMailConsts.SEND_FOLDER);

      folderList.addAll(tmp);

    } catch (Exception ex) {
      logger.error("Exception", ex);
    }
  }

  /**
   * 
   * ログインユーザーの退避対象メールフォルダ名表示処理。
   * 
   * @param rundata
   * @param context
   */
  public void loadDownLoadMailFolderName(RunData rundata, Context context) {

    try {
      // 退避パラメータの取得
      AvzTMailBatch mailBatch =
        WebMailUtils.getAvzTMailBatchResultList(user_id);
      // 退避対象アカウント情報取得
      EipMMailAccount account =
        WebMailUtils.getMailAccount(mailBatch.getAccountId().intValue());
      // 退避対象アカウント、退避対象フォルダIDをキーに退避対象フォルダオブジェクトを取得。
      EipTMailFolder eipTMailFolder =
        WebMailUtils.getEipTMailFolder(account, mailBatch
          .getFolderId()
          .toString());
      save_folder_name = eipTMailFolder.getFolderName();

    } catch (Exception ex) {
      logger.error(ERROR_MESSAGE_PROCESS_DOWNLOAD_FORM + login_user_name, ex);
    }
  }

  /**
   * ログインユーザーの退避対象メールフォルダ名取得
   * 
   * @return 退避対象メールフォルダ名
   */
  public String getDownLoadMailFolderName() {
    return save_folder_name;
  }

  /**
   * フォルダ選択一覧を取得
   * 
   * @return ログインユーザーのメールフォルダ選択一覧
   */
  // public List<WebMailFolderResultData> getMailFolderList() {
  public List<ExtFolderInfo> getMailFolderList() {
    return folderList;
  }

  /**
   * 表示画面切り替え用のpage_view_typeを取得
   * 
   * @return string page_view_type
   */
  private void loadPageViewType() {
    AvzTMailBatch mailBatch = WebMailUtils.getAvzTMailBatchResultList(user_id);

    if (mailBatch == null) {
      // レコードがない場合は「一括処理選択」画面(AIPO_MAIL_PC_008)名を渡す
      page_view_type = SAVE_PROCESS;
    } else {
      // 処理ステータス
      String status = mailBatch.getStatus();
      // 処理種別
      String type = mailBatch.getType();

      if (WebMailConsts.BATCH_PROCESSING.equals(status)) {
        // <処理ステータス> = 0:一括処理中 の場合
        // 「一括処理中表示」画面(AIPO_MAIL_PC_009)名を渡す
        page_view_type = CANCEL_PROCESS;
      } else if (WebMailConsts.BATCH_PROCESSING_COMPLETE.equals(status)
        && WebMailConsts.BATCH_TYPE_SAVE.equals(type)) {
        // <処理ステータス> = 1:一括処理完了 かつ
        // <処理種別> = 0:ダウンロード の場合
        // 「退避ファイルダウンロード」画面(AIPO_MAIL_PC_010)名を渡す
        String filePath = WebMailUtils.getDownloadFilePath(user_id);
        if ((new File(filePath)).exists()) {
          page_view_type = DOWNLOAD_PROCESS;
        } else {
          // ダウンロード用のZIPファイルが存在しない場合（状態不整合）は、状態を初期化
          page_view_type = SAVE_PROCESS;
          WebMailUtils.updateMailBatchData(
            mailBatch,
            WebMailConsts.BATCH_PROCESSING_CANCEL);
        }
        // add start
        // 受入障害対応No.157
      } else if (WebMailConsts.FINISH_DOWNLOAD.equals(status)
        && WebMailConsts.BATCH_TYPE_SAVE.equals(type)) {
        // <処理ステータス> = 1:一括処理完了 かつ
        // <処理種別> = 0:ダウンロード の場合
        // 「退避ファイルダウンロード」画面(AIPO_MAIL_PC_010)名を渡す
        String filePath = WebMailUtils.getDownloadFilePath(user_id);
        if ((new File(filePath)).exists()) {
          page_view_type = DOWNLOAD_FINISH;
        } else {
          // ダウンロード用のZIPファイルが存在しない場合（状態不整合）は、状態を初期化
          page_view_type = SAVE_PROCESS;
          WebMailUtils.updateMailBatchData(
            mailBatch,
            WebMailConsts.BATCH_PROCESSING_CANCEL);
        }
        // add end
      } else {
        // それ以外の場合
        // 「一括処理選択」画面(AIPO_MAIL_PC_008)名を渡す
        page_view_type = SAVE_PROCESS;
      }
    }
    // return page_view_type;
  }

  /**
   * 表示画面切り替え用のpage_view_typeを取得
   * 
   * @return string page_view_type
   */
  public String getPageViewType() {
    return page_view_type;
  }

  /**
   * 一括処理中かどうかの判定処理
   * 
   * @return boolean true:一括処理中,false:それ以外
   */
  private boolean isOtherProcess() {
    AvzTMailBatch mailBatch = WebMailUtils.getAvzTMailBatchResultList(user_id);
    if (mailBatch == null) {
    } else {
      // 処理ステータス取得
      String status = mailBatch.getStatus();
      // 処理ステータス判定
      if (WebMailConsts.BATCH_PROCESSING.equals(status)) {
        return true;
      }
    }
    return false;
  }

  /**
   * 移動可能なフォルダかどうかの判定処理
   * 
   * @param string
   *            account_id, string target_folder_id,string move_folder_id
   * @return true：可能, false：不可
   */
  public boolean isMoveFoLder(int account_id, ALNumberField target_folder_id,
      ALNumberField move_folder_id) {
    // 対象フォルダのフォルダ種別を取得します
    String target_folder_kind =
      WebMailUtils.getEipTMailFolder(
        account_id,
        (int) target_folder_id.getValue()).getFolderKind();
    // 移動先フォルダのフォルダ種別を取得します
    String move_folder_kind =
      WebMailUtils.getEipTMailFolder(
        account_id,
        (int) move_folder_id.getValue()).getFolderKind();
    // 対象フォルダと移動先フォルダのフォルダ種別の比較
    if (target_folder_kind.equals(move_folder_kind)) {
      return true;
    }
    return false;
  }

  /**
   * 操作種別ごとに非同期処理を行います
   */
  private void doAsynchronousProcess() {
    Runnable runnable = null;
    // 非同期処理選択
    try {
      if (WebMailConsts.BATCH_TYPE_SAVE.equals(operation_type
        .getValueAsString())) {
        // 操作種別：「フォルダ内退避」
        runnable =
          new WebMailPackageProcessFolderSaveThread(Database
            .createDataContext(Database.getDomainName()), user_id);
      } else if (WebMailConsts.BATCH_TYPE_MOVE.equals(operation_type
        .getValueAsString())) {
        // 操作種別：「フォルダ移動」
        runnable =
          new WebMailPackageProcessFolderMoveThread(Database
            .createDataContext(Database.getDomainName()), user_id);
      } else if (WebMailConsts.BATCH_TYPE_DELETE.equals(operation_type
        .getValueAsString())) {
        // 操作種別：「フォルダ削除」
        runnable =
          new WebMailPackageProcessFolderDeleteThread(Database
            .createDataContext(Database.getDomainName()), user_id, false);
      }
    } catch (Exception ex) {
      logger.error(ERROR_MESSAGE_ASYNCHRONOUS_PROCESS + login_user_name, ex);
      return;
    }
    Thread thread = new Thread(runnable);
    thread.start();
  }
}
