/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * Copyright (C) 2004-2011 Aimluck,Inc.
 * http://www.aipo.com
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package aipo.batch.utils;

import java.io.File;
import java.io.FileInputStream;
import java.util.List;
import java.util.Properties;

import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.jetspeed.services.resources.JetspeedResources;

import com.aimluck.eip.cayenne.om.account.AvzMGroupsend;
import com.aimluck.eip.cayenne.om.account.AvzMRole;
import com.aimluck.eip.cayenne.om.account.EipMPosition;
import com.aimluck.eip.cayenne.om.account.EipMPost;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.SelectQuery;

/**
 * <HR>
 * バッチ用ユーティリティクラス
 * <p>
 * 
 * バッチのユーティリティクラス<br>
 * <P>
 * <HR>
 * <P>
 */
public class BatchUtils {

  /** ロガー */
  private static final JetspeedLogger logger =
    JetspeedLogFactoryService.getLogger(BatchUtils.class.getName());

  /** プロパティファイル */
  private static final String PROPERTY_FILE =
    JetspeedResources.getString("aipo.conf", "")
      + File.separator
      + "Batch.properties";

  /** ファイルパス */
  public static final String IMPORT_CSV_FILE_PATH =
    (String) getProperties(PROPERTY_FILE).get("import.csv.file.path");

  /** 区切り文字 */
  public static final String SEPARATOR =
    (String) getProperties(PROPERTY_FILE).get("separator");

  /** システムユーザーID：1,2,3 */
  public static String[] SystemUserId =
    ((String) getProperties(PROPERTY_FILE).get("system.user.id")).split(",");

  /** インポートファイルエンコーディング */
  public static final String FILE_ENCODING =
    (String) getProperties(PROPERTY_FILE).get("import.csv.file.encoding");

  /**
   * プロパティファイル取得
   * <p>
   * プロパティファイルを読み込みます。<br>
   * 
   * @param path
   *          プロパティファイルのパス
   * @return Properties プロパティクラスのインスタンス
   */
  public static Properties getProperties(String path) {
    Properties properties = new Properties();
    try {
      properties.load(new FileInputStream(path));
    } catch (Exception e) {
      logger.error("プロパティファイル[" + path + "]が見つかりません。", e);
    }
    return properties;

  }

  /**
   * ダブルクォーテーション除去
   * <p>
   * 文字列から先頭と最後のダブルクォーテーションを除去します。<br>
   * 
   * @param s
   *          ダブルクォーテーションで括られた文字列
   * @return String ダブルクォーテーションを除去した文字列
   */
  public static String trimDoubleQuotes(String s) {
    int first = s.indexOf('\"');
    int last = s.lastIndexOf('\"');
    if (first == 0 && last == (s.length() - 1)) {
      s = s.substring(first + 1, last);
    }
    s = s.replace("\"\"", "\"");
    return s;
  }

  /**
   * 部署情報取得(部署コード)
   * <p>
   * パラメータで渡された値(部署コード)を条件に部署情報を検索する。<br>
   * 
   * @param postCode
   *          部署コード
   * @throws Exception
   *           DBアクセス異常などで発生
   * @return List<EipMPost> 部署情報のインスタンス配列
   */

  public static List<EipMPost> getEipMPostFromPostCode(String postCode)
      throws Exception {
    try {
      SelectQuery<EipMPost> query = Database.query(EipMPost.class);
      // 部署コードをキーに検索
      Expression exp =
        ExpressionFactory.matchExp(EipMPost.POST_CODE_PROPERTY, postCode);
      query.setQualifier(exp);
      List<EipMPost> resultList = query.fetchList();
      return resultList;
    } catch (Exception e) {
      String errMsg = "部署情報検索処理(部署コード)で予期せぬ例外が発生しました。";
      logger.error(errMsg, e);
      throw new Exception(errMsg + e.toString());
    }
  }

  /**
   * 部署情報取得(部署名)
   * <p>
   * パラメータで渡された値(部署名)を条件に部署情報を検索し、件数を返す。<br>
   * 
   * @param postName
   *          部署名
   * @throws Exception
   *           DBアクセス異常などで発生
   * @return int 部署情報の件数
   */

  public static int getEipMPostCount(String postName) throws Exception {
    try {
      SelectQuery<EipMPost> query = Database.query(EipMPost.class);
      // 部署名をキーに検索
      Expression exp =
        ExpressionFactory.matchExp(EipMPost.POST_NAME_PROPERTY, postName);
      query.setQualifier(exp);
      return query.getCount();
    } catch (Exception e) {
      String errMsg = "部署情報検索処理(部署名)で予期せぬ例外が発生しました。";
      logger.error(errMsg, e);
      throw new Exception(errMsg + e.toString());
    }
  }

  /**
   * 役職情報取得(役職名)
   * <p>
   * パラメータで渡された値(役職名)を条件に役職情報を検索し、件数を返す。<br>
   * 
   * @param positionName
   *          役職名
   * @throws Exception
   *           DBアクセス異常などで発生
   * @return int 役職情報の件数
   */

  public static int getEipMPositionCount(String positionName) throws Exception {
    try {
      SelectQuery<EipMPosition> query = Database.query(EipMPosition.class);
      // 役職名をキーに検索
      Expression exp =
        ExpressionFactory.matchExp(
          EipMPosition.POSITION_NAME_PROPERTY,
          positionName);
      query.setQualifier(exp);
      return query.getCount();
    } catch (Exception e) {
      String errMsg = "役職情報検索処理(役職名)で予期せぬ例外が発生しました。";
      logger.error(errMsg, e);
      throw new Exception(errMsg + e.toString());
    }
  }

  /**
   * 役割情報取得(役割名)
   * <p>
   * パラメータで渡された値(役割名)を条件に役割情報を検索し、件数を返す。<br>
   * 
   * @param positionName
   *          役割名
   * @throws Exception
   *           DBアクセス異常などで発生
   * @return int 役割情報の件数
   */

  public static int getAvzMRoleCount(String roleName) throws Exception {
    try {
      SelectQuery<AvzMRole> query = Database.query(AvzMRole.class);
      // 役割名をキーに検索
      Expression exp =
        ExpressionFactory.matchExp(AvzMRole.ROLE_NAME_PROPERTY, roleName);
      query.setQualifier(exp);
      return query.getCount();
    } catch (Exception e) {
      String errMsg = "役割情報検索処理(役割名)で予期せぬ例外が発生しました。";
      logger.error(errMsg, e);
      throw new Exception(errMsg + e.toString());
    }
  }

  /**
   * グループ送信情報取得(役割名)
   * <p>
   * パラメータで渡された値(グループ送信名)を条件にグループ送信情報を検索し、件数を返す。<br>
   * 
   * @param groupSendName
   *          グループ送信名
   * @throws Exception
   *           DBアクセス異常などで発生
   * @return int グループ送信情報の件数
   */

  public static int getAvzMGroupsendCount(String groupSendName)
      throws Exception {
    try {
      SelectQuery<AvzMGroupsend> query = Database.query(AvzMGroupsend.class);
      // グループ送信名をキーに検索
      Expression exp =
        ExpressionFactory.matchExp(
          AvzMGroupsend.GROUPSEND_NAME_PROPERTY,
          groupSendName);
      query.setQualifier(exp);
      return query.getCount();
    } catch (Exception e) {
      String errMsg = "グループ送信情報検索処理(グループ送信名)で予期せぬ例外が発生しました。";
      logger.error(errMsg, e);
      throw new Exception(errMsg + e.toString());
    }
  }

}
