/*
 * Copyright (C) 2012 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.ksk.contacts.tel.util;

import java.util.List;
import android.content.Context;
import android.text.ClipboardManager;
import android.view.LayoutInflater;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.View.OnLongClickListener;
import android.view.ViewGroup.LayoutParams;
import android.widget.CheckBox;
import android.widget.CompoundButton;
import android.widget.ImageView;
import android.widget.TableRow;
import android.widget.TextView;
import android.widget.Toast;
import android.widget.CompoundButton.OnCheckedChangeListener;
import com.ksk.contacts.R;
import com.ksk.contacts.util.SendAction;

public class ContactInfoList implements OnCheckedChangeListener, OnClickListener {

	//優先番号種別
	//音声発信
	public final Integer TYPE_TEL         = 0;

	//メール
	public final Integer TYPE_MAIL        = 1;

	//カスタム(音声発信)
	public final Integer TYPE_CUSTOM_TEL  = 2;

	//カスタム(メール)
	public final Integer TYPE_CUSTOM_MAIL = 3;

	//Context情報
	private Context   mContext;

	//リストRowView
	private View      mView;

	//リストデータ情報
	private List<ContactInfoList> listdata;

	//View：チェックボックス
	private CheckBox  check;

	//View：発信種別文言：Voiceに発信、Eメール送信
	private TextView  hint;

	//View：発信先(電話/メール)番号
	private TextView  number;

	//View：音声発信ボタン
	private ImageView action1;

	//View：メール送信ボタン
	private ImageView action2;

	//電話帳RawコンタクトID
	private long      rawcontact_id;

	//発信種別(電話/メール)
	private Integer   type;

	//発信先(電話/メール)番号
	private String    data;

	//優先番号フラグ(True：優先対象、False：優先対象外)
	private boolean   main_flg;

	//優先番号DB
	private PriorityNumberDB PN_DB;

	private final ClipboardManager mClipboardManager;

	public ContactInfoList(Context context, List<ContactInfoList> list){

		//情報の退避
		mContext = context;
		listdata = list;

		//カスタムレイアウトの取得
		LayoutInflater inflater = (LayoutInflater)context.getSystemService(Context.LAYOUT_INFLATER_SERVICE);
		mView = inflater.inflate(R.layout.contactinfo_list_row, null);

		//カスタムレイアウトの横幅がWrapになってしまうため、強制的にプロパティ情報を変更する。
		//LayoutParamsとweightの両方の設定が必要
		TableRow.LayoutParams text_layout_params = new TableRow.LayoutParams(LayoutParams.FILL_PARENT, LayoutParams.WRAP_CONTENT);
		text_layout_params.weight = 1;
		mView.setLayoutParams(text_layout_params);

		//リソースの取得
		check    = (CheckBox)  mView.findViewById(R.id.checkBox);
		hint     = (TextView)  mView.findViewById(R.id.kind);
		number   = (TextView)  mView.findViewById(R.id.number);
		action1  = (ImageView) mView.findViewById(R.id.action1);
		action2  = (ImageView) mView.findViewById(R.id.action2);

		//イベント登録
		check.setOnCheckedChangeListener(this);
		action1.setOnClickListener(this);
		action2.setOnClickListener(this);

		//設定済みの優先番号を取得
		PN_DB = new PriorityNumberDB(context);

		//ClipManagerの取得
		mClipboardManager = (ClipboardManager) context.getSystemService(Context.CLIPBOARD_SERVICE);
	}

	public View getListView(){
		return mView;
	}

	public void setRawContactID(long id){
		rawcontact_id = id;
	}

	public void setType(Integer t){
		type = t;

		//発信種別に応じた文言、ボタン表示の決定
		if((type != TYPE_MAIL) && (type != TYPE_CUSTOM_MAIL)){
			//音声系の場合
			hint.setText("VOICEに発信");
			action1.setVisibility(View.VISIBLE);
			action2.setVisibility(View.VISIBLE);
		}else{
			//メール系の場合
			hint.setText("EMAILに送信");
			action1.setVisibility(View.INVISIBLE);
			action2.setVisibility(View.VISIBLE);
		}
	}

	public void setNumber(String n){
		data = n;
		if(data != null){
			number.setText(data);

			//番号Viewの長押しにて番号コピーを行う
			number.setTag(data);
			number.setOnLongClickListener(new OnLongClickListener() {
				@Override
				public boolean onLongClick(View v) {
					mClipboardManager.setText((String) v.getTag());
					Toast.makeText(mContext.getApplicationContext(), "クリップボードにコピーしました。\n" + (String)v.getTag(), Toast.LENGTH_SHORT).show();
					return false;
				}
			});
		}
	}

	public void setChecked(boolean f){
		main_flg = f;
		if(check != null){
			check.setChecked(main_flg);
		}
	}

	@Override
	public void onClick(View arg0) {
		doAction(arg0, listdata);
	}

	@Override
	public void onCheckedChanged(CompoundButton buttonView, boolean isChecked) {
		doChecked(buttonView, isChecked, listdata);
	}

	private void doChecked(View checkView, boolean flag, List<ContactInfoList> list){
		int pos_old = -1;
		int pos_new = -1;

		for(int i=0; i<list.size(); i++){
			ContactInfoList info = list.get(i);

			//旧チェック箇所の場合
			if(info.main_flg == true){
				pos_old = i;
			}

			//新チェック箇所の場合
			if(info.check.equals(checkView)){
				pos_new = i;
			}
		}

		//古いチェックボックスを剥がす
		if(pos_old != -1){
			//チェックボックスの状態更新(チェック有→無)
			ContactInfoList info = list.get(pos_old);
			info.main_flg = false;
			info.check.setChecked(false);

			//優先番号DBの更新(設定→未設定)
			if(info.type == info.TYPE_TEL){
				//電話番号
				PN_DB.setPhoneNumber(info.rawcontact_id, "");

			}else if(info.type == info.TYPE_MAIL){
				//メール
				PN_DB.setMailNumber(info.rawcontact_id, "");

			}
		}

		//新しいチェックボックスの状態に設定する
		if((flag == true) && (pos_new != -1)){
			//チェックボックスの状態更新(最新の状態)
			ContactInfoList info = list.get(pos_new);
			info.main_flg = flag;
			info.check.setChecked(flag);

			//優先番号DBの更新(未設定→設定)
			if(info.type == info.TYPE_TEL){
				//電話番号
				PN_DB.setPhoneNumber(info.rawcontact_id, info.data);

			}else if(info.type == info.TYPE_MAIL){
				//メール
				PN_DB.setMailNumber(info.rawcontact_id, info.data);

			}

			Toast.makeText(mContext, "優先番号が更新されました", Toast.LENGTH_SHORT).show();
		}
	}

	private void doAction(View actionView, List<ContactInfoList> list){
		SendAction action = new SendAction(mContext);

		for(int i=0; i<list.size(); i++){

			ContactInfoList info = list.get(i);

			//押下されたボタンに応じた処理を実施。
			if( info.action1.equals(actionView) ){
				//音声発信
				action.sendCall(info.data);
				break;

			} else if( info.action2.equals(actionView) ){
				//メール送信
				action.sendMail(info.data);
				break;
			}
		}
	}
}
