/*
 * Copyright (C) 2012 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.ksk.contacts.tel.util;

import java.text.SimpleDateFormat;
import java.util.List;

import com.ksk.contacts.R;
import com.ksk.contacts.util.CallLogMgr.CallLogData;

import android.content.Context;
import android.graphics.Color;
import android.provider.CallLog;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.BaseAdapter;
import android.widget.ImageView;
import android.widget.TextView;

public class CallLogListAdapter extends BaseAdapter {

	//メンバーリスト情報
	private List<CallLogData> mListData;

	//Context情報
	private Context        mContext;

	/* メンバーリスト表示用Holder */
	private class Holder {
		int         contact_id;     //ContactID
		ImageView   image;          //ユーザー画像用View
		TextView    name;           //ユーザー名用View
		TextView    number;         //電話番号用View
		TextView    date;           //日付情報用View
		ImageView   status;         //結果画像用View
		TextView    response;       //結果情報用View
	}

	//コンストラクタ
    public CallLogListAdapter(Context context) {
    	mContext  = context;
	}

    /**
     * @param dataList
     */
    public void setDataList(List<CallLogData> dataList) {
    	mListData = dataList;
	}

	/* (非 Javadoc)
	 * @see android.widget.Adapter#getCount()
	 */
	@Override
    public int getCount() {
    	return mListData.size();
    }

    /* (非 Javadoc)
     * @see android.widget.Adapter#getItem(int)
     */
    @Override
    public Object getItem(int position) {
    	return mListData.get(position);
    }

    /* (非 Javadoc)
     * @see android.widget.Adapter#getItemId(int)
     */
    @Override
    public long getItemId(int position) {
    	return position;
    }

    /* (非 Javadoc)
     * @see android.widget.Adapter#getView(int, android.view.View, android.view.ViewGroup)
     */
    public View getView(int position, View convertView, ViewGroup parent) {
    	Holder holder = null;
		View   v = convertView;

		//viewが既に作成済みかをチェック
		if(v == null){
			//未作成の場合は、レイアウトを元にViewを新規作成する
			LayoutInflater inflater = (LayoutInflater)mContext.getSystemService(Context.LAYOUT_INFLATER_SERVICE);
			v = inflater.inflate(R.layout.call_log_row, null);

			//Holderを作成し、各パラメータを設定する。
			holder = new Holder();
			holder.image    = (ImageView) v.findViewById(R.id.log_user);     //ユーザー画像
			holder.name     = (TextView)  v.findViewById(R.id.log_name);     //ユーザー名(フルネーム)
			holder.number   = (TextView)  v.findViewById(R.id.log_number);   //電話番号
			holder.date     = (TextView)  v.findViewById(R.id.log_date);     //日付
			holder.status   = (ImageView) v.findViewById(R.id.log_status);   //状態アイコン
			holder.response = (TextView)  v.findViewById(R.id.log_response); //状態
			//ViewにHolderを登録する。
			v.setTag(holder);

		}else{
			//View作成済みのため、Viewに登録されているTagからHolder情報を取得する。
			holder = (Holder) v.getTag();
		}

		//選択位置を元にリスト情報を取得する。
		CallLogData item = (CallLogData) getItem(position);

		//リスト情報がNULLではない、かつHolderがNULLでない場合は、各情報の設定を行う。
		if((item != null) && (holder != null) ){

			//ContactIDの保存
			holder.contact_id = item.contact_id;
			v.setTag(holder);

			//ユーザー画像の描画
			//TODO

			//ユーザー名の設定
			if(item.name != null){
				holder.name.setText(item.name);

			}else{
				holder.name.setText("未登録");
			}

			//電話番号の設定
			holder.number.setText(item.number);

			//日付の設定
			SimpleDateFormat sdf = new SimpleDateFormat("MM'/'dd kk':'mm");
			String date =sdf.format(item.date);
			holder.date.setText(date);

			//状態の設定
			if(item.type == CallLog.Calls.MISSED_TYPE){
				holder.status.setImageResource(android.R.drawable.sym_call_missed);

				//通話時間の設定
				holder.response.setText("不在");
				holder.response.setTextColor(Color.RED);

			}else{
				if(item.type == CallLog.Calls.INCOMING_TYPE){
					holder.status.setImageResource(android.R.drawable.sym_call_incoming);

				}else if(item.type == CallLog.Calls.OUTGOING_TYPE){
					holder.status.setImageResource(android.R.drawable.sym_call_outgoing);
				}

				//通話時間の設定
				if(item.duration == 0){
					holder.response.setText("不在");
					holder.response.setTextColor(Color.RED);

				}else{
					if(item.duration < 60){
						holder.response.setText(item.duration + "秒");

					}else if(item.duration < 60 * 60){
						holder.response.setText((item.duration / 60) + "分" + (item.duration % 60) + "秒");

					}else if(item.duration < 60 * 60 * 24){
						holder.response.setText((item.duration / 3600) + "時" + (item.duration / 60) + "分");

					}else{
						holder.response.setText((item.duration / 3600) + "時");
					}
					holder.response.setTextColor(Color.BLUE);
				}
			}
		}

		return v;
    }
}
