﻿using System;
using System.Drawing;
using System.Diagnostics;
using System.Windows.Forms;
using SharpDX;
using SharpDX.Direct2D1;
using SharpDX.DirectWrite;

using D2D = SharpDX.Direct2D1;
using DW = SharpDX.DirectWrite;
using Drawing = System.Drawing;

namespace D2DBench
{
    class Direct2DBitmap : IDisposable,IBitmap
    {
        internal D2D.Bitmap bmp;
        public Direct2DBitmap(D2D.Bitmap bmp)
        {
            this.bmp = bmp;
        }

        public void Dispose()
        {
            this.bmp.Dispose();
        }

        public int Width
        {
            get { return this.bmp.PixelSize.Width; }
        }

        public int Height
        {
            get { return this.bmp.PixelSize.Height; }
        }
    }

    class Direct2D : IDisposable, IRender
    {
        D2D.Factory Factory2D;
        DW.Factory FactoryDWrite;
        WindowRenderTarget render;
        TextFormat format;
        TextLayout layout;
        ColorTableD2D colors;
        const string showStr = "D2DBenchMark";

        public Direct2D(Control ctrl, Drawing.Size ClientSize, bool antialias)
        {
            Factory2D = new SharpDX.Direct2D1.Factory();
            FactoryDWrite = new SharpDX.DirectWrite.Factory();
            HwndRenderTargetProperties hwndprops = new HwndRenderTargetProperties { Hwnd = ctrl.Handle, PixelSize = ClientSize, PresentOptions = PresentOptions.Immediately };
            render = new WindowRenderTarget(Factory2D, new RenderTargetProperties(), hwndprops);
            format = new TextFormat(FactoryDWrite, ctrl.Font.Name, ctrl.Font.Size * render.DotsPerInch.Width / 72.0f);
            colors = new ColorTableD2D(render);
            layout = new TextLayout(FactoryDWrite, showStr, format, ClientSize.Width, ClientSize.Height);
            render.AntialiasMode = antialias ? AntialiasMode.PerPrimitive : AntialiasMode.Aliased;
        }

        public IBitmap Load(Drawing.Bitmap srcBitmap)
        {
            var bitmapData = srcBitmap.LockBits(
                new Drawing.Rectangle(0, 0, srcBitmap.Width, srcBitmap.Height),
                System.Drawing.Imaging.ImageLockMode.ReadOnly,
                System.Drawing.Imaging.PixelFormat.Format32bppPArgb);

            var stream = new DataStream(bitmapData.Scan0, bitmapData.Stride * bitmapData.Height, true, false);

            var properties = new BitmapProperties()
            {
                PixelFormat = new D2D.PixelFormat(SharpDX.DXGI.Format.B8G8R8A8_UNorm, AlphaMode.Premultiplied)
            };
            var bitmap = new D2D.Bitmap(this.render, srcBitmap.Size, stream, bitmapData.Stride, properties);

            srcBitmap.UnlockBits(bitmapData);

            return new Direct2DBitmap(bitmap);
        }

        public void UnLoad(IBitmap bmp)
        {
            Direct2DBitmap dbmp = (Direct2DBitmap)bmp;
            dbmp.Dispose();
        }

        public void BeginDraw()
        {
            render.BeginDraw();
        }

        public int ColorCount { get { return this.colors.Count; } }

        public void DrawBitmap(IBitmap bmp,int left,int top,int right,int bottom)
        {
            Direct2DBitmap localBmp = (Direct2DBitmap)bmp;
            SharpDX.RectangleF rect = new SharpDX.RectangleF();
            rect.Left = left;
            rect.Top = top;
            rect.Right = right;
            rect.Bottom = bottom;
            render.DrawBitmap(localBmp.bmp, rect, 1.0f, BitmapInterpolationMode.NearestNeighbor);
        }

        public void DrawString(int x, int y,int colorno)
        {
            render.DrawTextLayout(new Point(x,y), layout, colors[colorno]);
        }

        public void DrawEllipseByBrush(int x, int y, int radiusX, int radiusY, int colorno, int backcolorno)
        {
            D2D.Ellipse ellipse = new Ellipse();
            ellipse.Point = new PointF(x, y);
            ellipse.RadiusX = radiusX;
            ellipse.RadiusY = radiusY;
            EllipseGeometry geo = new EllipseGeometry(this.Factory2D, ellipse);
            render.FillGeometry(geo,colors[backcolorno]);
            render.DrawGeometry(geo,colors[colorno]);
        }

        public void DrawRectangleByBrush(int left, int top, int right, int bottom, int colorno, int backcolorno)
        {
            SharpDX.RectangleF rect = new SharpDX.RectangleF();
            rect.Left = left;
            rect.Top = top;
            rect.Right = right;
            rect.Bottom = bottom;
            RectangleGeometry geo = new RectangleGeometry(this.Factory2D, rect);
            render.FillGeometry(geo, colors[backcolorno]);
            render.DrawGeometry(geo, colors[colorno]);
        }

        public void DrawEllipse(int x, int y, int radiusX, int radiusY, int colorno)
        {
            var fore = colors[colorno];
            D2D.Ellipse ellipse = new Ellipse();
            ellipse.Point = new PointF(x, y);
            ellipse.RadiusX = radiusX;
            ellipse.RadiusY = radiusY;
            render.DrawEllipse(ellipse, fore);
        }

        public void DrawRectangle(int left, int top, int right, int bottom, int colorno)
        {
            var fore = colors[colorno];
            SharpDX.RectangleF rect = new SharpDX.RectangleF();
            rect.Left = left;
            rect.Top = top;
            rect.Right = right;
            rect.Bottom = bottom;
            render.DrawRectangle(rect, fore);
        }

        public void FillRectangle(int left, int top, int right, int bottom, int colorno)
        {
            var fore = colors[colorno];
            SharpDX.RectangleF rect = new SharpDX.RectangleF();
            rect.Left = left;
            rect.Top = top;
            rect.Right = right;
            rect.Bottom = bottom;
            render.FillRectangle(rect, fore);
        }

        public void EndDraw()
        {
            render.EndDraw();
        }

        public void Dispose()
        {
            colors.Clear();
            layout.Dispose();
            format.Dispose();
            render.Dispose();
            Factory2D.Dispose();
            FactoryDWrite.Dispose();
        }
    }
}
