package daruma.client.darumaviewer.views;

import java.net.URL;

import org.eclipse.core.runtime.Path;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DropTarget;
import org.eclipse.swt.dnd.DropTargetEvent;
import org.eclipse.swt.dnd.DropTargetListener;
import org.eclipse.swt.dnd.FileTransfer;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Device;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IFolderLayout;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;

import daruma.client.darumaviewer.DaRuMaViewerPlugin;
import daruma.client.darumaviewer.Messages;

/**
 * SWTユーティリティ
 * 
 * @author shumi
 * 
 */
public class SwtUtil {
	/**
	 * グリッドレイアウトを生成する
	 * 
	 * @param gridCol
	 *            カラム数
	 * @return グリッドレイアウト
	 */
	public static GridLayout makeGridLayout(int gridCol) {
		GridLayout layout = new GridLayout(gridCol, false);
		layout.verticalSpacing = 8;
		layout.horizontalSpacing = 8;
		return layout;
	}

	/**
	 * 固定サイズグリッドを生成する
	 * 
	 * @param width
	 *            幅
	 * @param height
	 *            高さ
	 * @param hSpan
	 *            水平SPAN
	 * @return グリッド
	 */
	public static GridData makeFixedGridData(int width, int height, int hSpan) {
		return (makeGridData(width, height, hSpan, -1,
				GridData.HORIZONTAL_ALIGN_BEGINNING));
	}

	/**
	 * 固定サイズグリッドを生成する
	 * 
	 * @param width
	 *            幅
	 * @param height
	 *            高さ
	 * @param hSpan
	 *            水平SPAN
	 * @param vSpan
	 *            垂直SPAN
	 * @return グリッド
	 */
	public static GridData makeFixedGridData(int width, int height, int hSpan,
			int vSpan) {
		return (makeGridData(width, height, hSpan, vSpan,
				GridData.HORIZONTAL_ALIGN_BEGINNING));
	}

	/**
	 * 可変サイズグリッドを生成する
	 * 
	 * @param width
	 *            幅
	 * @param height
	 *            最小高さ
	 * @param hSpan
	 *            水平SPAN
	 * @return グリッド
	 */
	public static GridData makeVarGridData(int width, int height, int hSpan) {
		return (makeGridData(width, height, hSpan, -1,
				GridData.HORIZONTAL_ALIGN_BEGINNING | GridData.FILL_BOTH));
	}

	/**
	 * 可変サイズグリッドを生成する
	 * 
	 * @param width
	 *            幅
	 * @param height
	 *            最小高さ
	 * @param hSpan
	 *            水平SPAN
	 * @param vSpan
	 *            垂直SPAN
	 * @return グリッド
	 */
	public static GridData makeVarGridData(int width, int height, int hSpan,
			int vSpan) {
		return (makeGridData(width, height, hSpan, vSpan,
				GridData.HORIZONTAL_ALIGN_BEGINNING | GridData.FILL_BOTH));
	}

	/**
	 * 最大サイズグリッドを生成する
	 * 
	 * @return グリッド
	 */
	public static GridData makeMaximumGridData() {
		return (makeGridData(-1, -1, -1, -1,
				GridData.HORIZONTAL_ALIGN_BEGINNING | GridData.FILL_BOTH));
	}

	/**
	 * 水平可変サイズグリッドを生成する
	 * 
	 * @param width
	 *            幅
	 * @param height
	 *            最小高さ
	 * @param hSpan
	 *            水平SPAN
	 * @return グリッド
	 */
	public static GridData makeHorVarGridData(int width, int height, int hSpan) {
		return (makeGridData(width, height, hSpan, -1,
				GridData.HORIZONTAL_ALIGN_BEGINNING | GridData.FILL_HORIZONTAL));
	}

	/**
	 * 水平可変サイズグリッドを生成する
	 * 
	 * @param width
	 *            幅
	 * @param height
	 *            最小高さ
	 * @param hSpan
	 *            水平SPAN
	 * @param vSpan
	 *            垂直SPAN
	 * @return グリッド
	 */
	public static GridData makeHorVarGridData(int width, int height, int hSpan,
			int vSpan) {
		return (makeGridData(width, height, hSpan, vSpan,
				GridData.HORIZONTAL_ALIGN_BEGINNING | GridData.FILL_HORIZONTAL));
	}

	/**
	 * 垂直可変サイズグリッドを生成する
	 * 
	 * @param width
	 *            幅
	 * @param height
	 *            最小高さ
	 * @param hSpan
	 *            水平SPAN
	 * @return グリッド
	 */
	public static GridData makeVerVarGridData(int width, int height, int hSpan) {
		return (makeGridData(width, height, hSpan, -1,
				GridData.HORIZONTAL_ALIGN_BEGINNING | GridData.FILL_VERTICAL));
	}

	/**
	 * 垂直可変サイズグリッドを生成する
	 * 
	 * @param width
	 *            幅
	 * @param height
	 *            最小高さ
	 * @param hSpan
	 *            水平SPAN
	 * @param vSpan
	 *            垂直SPAN
	 * @return グリッド
	 */
	public static GridData makeVerVarGridData(int width, int height, int hSpan,
			int vSpan) {
		return (makeGridData(width, height, hSpan, vSpan,
				GridData.HORIZONTAL_ALIGN_BEGINNING | GridData.FILL_VERTICAL));
	}

	/**
	 * グリッドを生成する
	 * 
	 * @param width
	 *            幅
	 * @param height
	 *            最小高さ
	 * @param hSpan
	 *            水平SPAN
	 * @param vSpan
	 *            垂直SPAN
	 * @param flg
	 *            フラグ
	 * @return グリッド
	 */
	private static GridData makeGridData(int width, int height, int hSpan,
			int vSpan, int flg) {
		GridData gridData = new GridData(flg);
		if (width > 0)
			gridData.widthHint = width;
		if (height > 0)
			gridData.heightHint = height;
		if (hSpan > 0)
			gridData.horizontalSpan = hSpan;
		if (vSpan > 0)
			gridData.verticalSpan = vSpan;
		return gridData;
	}

	/**
	 * ワークベンチページを取得する
	 * 
	 * @return ワークベンチページ
	 */
	public static IWorkbenchPage getPage() {
		IWorkbench workbench = PlatformUI.getWorkbench();
		IWorkbenchWindow window = workbench.getActiveWorkbenchWindow();
		return window.getActivePage();
	}

	/**
	 * レイアウトにビューを表示する 該当ビューが存在しない場合追加してから表示する
	 * 
	 * @param layout
	 *            レイアウト
	 * @param id
	 *            ビューID
	 * @return ビュー
	 */
	public static IViewPart showView(IFolderLayout layout, String id) {
		IViewPart view = null;
		IWorkbenchPage page = getPage();
		try {
			if (page != null) {
				view = page.findView(id);
				if (view == null) {
					if (layout != null)
						layout.addView(id);
				}

				view = page.showView(id, null, IWorkbenchPage.VIEW_ACTIVATE);
			}
		} catch (PartInitException e) {
			view = null;
			e.printStackTrace();
		}
		return view;
	}

	/**
	 * ビューを非表示にする
	 * 
	 * @param id
	 *            ビューID
	 */
	public static void hideView(String id) {
		try {
			IViewPart view = findView(id);
			IWorkbenchPage page = getPage();
			if ((page != null) && (view != null))
				page.hideView(view);
		} catch (Exception e) {
		}
	}

	/**
	 * ビューを非表示にする
	 * 
	 * @param view
	 *            ビュー
	 */
	public static void hideView(IViewPart view) {
		try {
			IWorkbenchPage page = view.getSite().getPage();
			if (page != null)
				page.hideView(view);

		} catch (Exception e) {
		}
	}

	/**
	 * ビューを検索する
	 * 
	 * @param id
	 *            ビューID
	 * @return 見つかった場合該当ビュー それ以外はNULLを返す
	 */
	public static IViewPart findView(String id) {
		IViewPart view = null;
		IWorkbenchPage page = getPage();
		if (page != null)
			view = getPage().findView(id);
		return view;
	}

	/**
	 * レイアウトにビューを表示する 該当ビューが存在しない場合追加してから表示する
	 * 
	 * @param layout
	 *            レイアウト
	 * @param id
	 *            ビューID
	 * @return ビュー
	 */
	public static IViewPart showView(IFolderLayout layout, String id,
			String secondaryId) {
		IViewPart view = null;
		IWorkbenchPage page = getPage();
		try {
			if (page != null) {
				view = page.showView(id, secondaryId,
						IWorkbenchPage.VIEW_ACTIVATE);
			}
		} catch (PartInitException e) {
			view = null;
			e.printStackTrace();
		}
		return view;
	}

	/**
	 * テーブルをセットアップする
	 * 
	 * @param table
	 *            テーブル
	 * @param cols
	 *            コラムタイトル配列
	 * @param widths
	 *            カラム幅配列
	 */
	public static void setupTable(Table table, String[] cols, int[] widths) {
		for (int i = 0; i < cols.length; i++) {
			TableColumn col = new TableColumn(table, SWT.LEFT);
			col.setText(cols[i]);
			col.setWidth(widths[i]);
		}
		table.setHeaderVisible(true);
		table.setLinesVisible(true);
	}

	/**
	 * イメージを生成する
	 * 
	 * @param width
	 * @param height
	 * @return 生成されたイメージ
	 */
	public static Image makeImg(int width, int height) {
		Device display = Display.getCurrent();
		try {
			Image img = new Image(display, width, height);
			return img;
		} catch (Exception e) {
			System.out.println("Fail to make image"); //$NON-NLS-1$
		}
		return null;
	}

	/**
	 * イメージをロードする
	 * 
	 * @param name
	 * @return ロードされたイメージ
	 */
	public static Image loadImg(String name) {
		Device display = Display.getCurrent();
		try {
			URL url = DaRuMaViewerPlugin.getDefault().find(new Path(name));
			Image img = new Image(display, url.openStream());
			return img;
		} catch (Exception e) {
			System.out.println("Fail to load image " + name); //$NON-NLS-1$
		}
		return null;
	}

	/**
	 * イメージをロードする
	 * 
	 * @param name
	 * @return ロードされたイメージ
	 */
	public static ImageDescriptor loadImageDescriptor(String name) {
		try {
			URL url = DaRuMaViewerPlugin.getDefault().find(new Path(name));
			ImageDescriptor img = ImageDescriptor.createFromURL(url);
			return img;
		} catch (Exception e) {
			System.out.println("Fail to load image " + name); //$NON-NLS-1$
		}
		return null;
	}

	public static void expandToLevel(TreeViewer tree, Object obj, int level) {
		tree.getTree().setRedraw(false);
		if (obj == null)
			tree.expandToLevel(level);
		else
			tree.expandToLevel(obj, level);
		tree.getTree().setRedraw(true);
	}

	public static boolean showQuestion(String msg) {
		return MessageDialog.openQuestion(null,
				Messages.getString("SwtUtil.0"), msg); //$NON-NLS-1$
	}

	public static void showInfo(String msg) {
		MessageDialog.openInformation(null,
				Messages.getString("SwtUtil.1"), msg); //$NON-NLS-1$
	}

	public static void showError(String msg) {
		MessageDialog.openError(null, Messages.getString("SwtUtil.2"), msg); //$NON-NLS-1$
	}

	public static class DirectoryButtonListener implements SelectionListener {
		private Text currText;

		public DirectoryButtonListener(Text txt) {
			currText = txt;
		}

		public void widgetSelected(SelectionEvent e) {
			DirectoryDialog dialog = new DirectoryDialog(currText.getShell());
			String dir = dialog.open();
			if (dir != null)
				if (currText.isEnabled())
					currText.setText(dir);

		}

		public void widgetDefaultSelected(SelectionEvent e) {
			// TODO Auto-generated method stub

		}

	}

	public static class FileButtonListener implements SelectionListener {
		private Text currText;

		public FileButtonListener(Text txt) {
			currText = txt;
		}

		public void widgetSelected(SelectionEvent e) {
			FileDialog dialog = new FileDialog(currText.getShell());
			String dir = dialog.open();
			if (dir != null)
				if (currText.isEnabled())
					currText.setText(dir);

		}

		public void widgetDefaultSelected(SelectionEvent e) {
			// TODO Auto-generated method stub

		}

	}

	public static class TextDropTargetListener implements DropTargetListener {
		private Text currText;

		private FileTransfer fileTransfer;

		public static TextDropTargetListener makeTextDropTargetListener(
				Text currText) {
			TextDropTargetListener listener = new TextDropTargetListener();
			FileTransfer fileTransfer = FileTransfer.getInstance();

			listener.currText = currText;
			listener.fileTransfer = fileTransfer;

			Transfer[] types = new Transfer[] { fileTransfer };
			int operations = DND.DROP_MOVE | DND.DROP_COPY | DND.DROP_DEFAULT;
			DropTarget source = new DropTarget(currText, operations);
			source.setTransfer(types);
			source.addDropListener(listener);
			return listener;
		}

		public void dragEnter(DropTargetEvent event) {
		}

		public void dragOperationChanged(DropTargetEvent event) {
		}

		public void dragOver(DropTargetEvent event) {
		}

		public void drop(DropTargetEvent event) {
			if (fileTransfer.isSupportedType(event.currentDataType)) {
				String[] files = (String[]) event.data;
				if (files.length > 0)
					if (currText.isEnabled())
						currText.setText(files[0]);
			}
		}

		public void dragLeave(DropTargetEvent event) {
		}

		public void dropAccept(DropTargetEvent event) {
		}

	}
}
