/*
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.dbunitng.annotations;

import static java.lang.annotation.ElementType.METHOD;
import static java.lang.annotation.RetentionPolicy.RUNTIME;

import java.lang.annotation.Retention;
import java.lang.annotation.Target;

/**
 * データベースへの接続情報などテストスイート全体での設定を表す。
 * 
 * @author jyukutyo
 * 
 */
@Retention(RUNTIME)
@Target(METHOD)
public @interface DbUnitNG {

	/**
	 * JDBCドライバクラスを返す。
	 * 
	 * <pre>
	 * e.g. &quot;com.mysql.jdbc.Driver&quot;
	 * </pre>
	 * 
	 * @return JDBCドライバクラス
	 */
	String driver();

	/**
	 * JDBCのURLを返す。
	 * 
	 * <pre>
	 * &quot;jdbc:mysql://127.0.0.1/database_name&quot;
	 * </pre>
	 * 
	 * @return JDBC URL
	 */
	String url();

	/**
	 * データベースのユーザー名を返す。
	 * 
	 * @return データベースのユーザー名
	 */
	String username();

	/**
	 * データベースのパスワードを返す。
	 * 
	 * @return パスワード
	 */
	String password();

	/**
	 * データベースへの接続方法を返す。 "DBCP"であればデータソースを利用する。それ以外であればDriverManagerを利用する。
	 * 
	 * @return 接続方法
	 */
	String datasource() default "";

	/**
	 * データベースのスキーマを返す。
	 * 
	 * @return スキーマ
	 */
	String schema() default "";

	/**
	 * スイート全体のデフォルトとするデータベース操作方法を返す
	 * 
	 * @return データベース操作方法
	 */
	DatabaseOperationType defaultOperation() default DatabaseOperationType.NONE;
}
