package org.dbunitng.listeners;

import java.lang.reflect.Method;

import org.testng.ITestResult;

/**
 * DbUnitNGのテストコンテキスト。
 * 
 * @author kiy0taka
 *
 */
public class DbUnitNGContext {

	/**
	 * コンテキスト格納用のスレッドローカル
	 */
	private static final ThreadLocal<DbUnitNGContext> threadLocal
		= new ThreadLocal<DbUnitNGContext>();

	/**
	 * 現在のテストコンテキストを取得する。
	 * @return 現在のテストコンテキスト
	 */
	public static DbUnitNGContext currentContext() {
		return threadLocal.get();
	}

	/**
	 * コンテキストの開始。
	 */
	static void start() {
		threadLocal.set(new DbUnitNGContext());
	}

	/**
	 * コンテキストの終了
	 */
	static void end() {
		threadLocal.remove();
	}

	/**
	 * テストメソッドの開始。
	 * @param testResult テスト結果
	 */
	static void startTest(ITestResult testResult) {
		DbUnitNGContext ctx = currentContext();
		ctx.testResult = testResult;
		if (testResult.getMethod().getMethod().equals(ctx.lastTest)) {
			ctx.invocationCount++;
		} else {
			ctx.invocationCount = 1;
		}
	}

	/**
	 * テストメソッドの終了
	 */
	static void endTest() {
		DbUnitNGContext ctx = currentContext();
		ctx.lastTest = ctx.testResult.getMethod().getMethod();
	}

	/**
	 * テスト結果
	 */
	private ITestResult testResult;

	/**
	 * 前回実行テストメソッド
	 */
	private Method lastTest;

	/**
	 * テストメソッドの実行回数
	 */
	private int invocationCount;

	/**
	 * 新しいコンテキストを構築。
	 */
	protected DbUnitNGContext() {}

	/**
	 * テストメソッド名を返す。
	 * @return テストメソッド名
	 */
	public String getName() {
		return testResult.getName();
	}

	/**
	 * テストクラスを返す。
	 * @return テストクラス
	 */
	public Class<?> getTestClass() {
		return testResult.getTestClass().getRealClass();
	}

	/**
	 * テストメソッドの実行回数を返す。
	 * @return テストメソッドの実行回数
	 */
	public int getInvocationCount() {
		return invocationCount;
	}
}
