//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Sep-30, 2012.
//
#ifndef DCCONTAINER_H
#define DCCONTAINER_H

#include "TKContainer.h"

class TKCell;
class TKCellCode;
class DCCell;
class DCCellCode;
class DCScene;
class DCContent;
class DCVComponent;
class DCUIGraphicsScene;

#include "dcvcomponent.h"

#include <QList>
#include <QString>

class DCContainer : public TKContainer
{
    DCContent       *d_content;
    DCScene         *d_scene;

    std::string     d_factoryCachedLocation;
    DCVCRefHolder   d_factoryCachedPageObject;

    QString         d_workDirRoot;
    QString         d_workDirCellRoot;
    QString         d_workDirCellCodeRoot;

public:
    DCContainer();
    virtual ~DCContainer();

    bool                    getIsModified() const;

    DCContent*              getContent() const { return d_content; }
    DCScene*                getScene() const { return d_scene; }
    QList<DCVComponent*>    getSelectedCellObjects() const;
    virtual float           getValue(std::string key) const { Q_UNUSED(key); return 0; }

    TKCellCode*             getEmptyCellCodeClass() const { return mEmptyCellClass; }
    bool                    getIsScriptableCell(const DCCell *cell) const;

    QString                 getWorkFilePathForCustomScript(const DCCell *cell) const;
    QString                 getWorkFilePathForCellCode(const DCCellCode *cellcode) const;

    virtual TKCell*         addCell(std::string theLocation, std::string theName, std::string type, std::string customScript);
    virtual TKCell*         addCell(std::string theLocation, std::string theName, TKCellCode *cellCode, std::string customScript);
    virtual TKCellCode*     addCellCode(std::string theName, std::string theAPIType, std::string code);

    virtual TKCell*         cellFactory(std::string location, std::string name, std::string type, bool canInterfaceIn, bool canInterfaceOut);
    virtual TKCell*         pluginCellFactory(std::string location, std::string fullName, std::string type, std::string pluginName, std::string pluginValue, bool canInterfaceIn, bool canInterfaceOut);
    virtual TKAxon*         axonFactory(TKCell *theOwner);
    virtual TKReceptor*     receptorFactory(TKCell *theOwner);
    virtual TKAxonTerminal* axonTerminalFactory(TKAxon *theOwner);
    virtual TKCellCode*     cellCodeFactory(std::string name, std::string cellapi, std::string code);

    /**
     * @brief remove cell from container and the page (DCVCPage) belonging.
     * @param cell
     *
     * All the axon terminals connected to this cell will also be removed after this.
     * This expected to only be called from command classes which are defined in dceditcommands.h
     *
     * @return true for success, false for failure
     */
    bool                    removeCell(DCCell *cell);

    /**
     * @brief remove cell code from container and the page (DCVCPage) belonging.
     * @param cellcode
     *
     * The cells which use the deleting cell code will have empty cell code after this.
     * This expected to only be called from command classes which are defined in dceditcommands.h
     *
     * @return true for success, false for failure
     */
    bool                    removeCellCode(DCCellCode *cellcode);

    /**
     * @brief movePage
     *
     * @param oldContainerBasedPathName
     * @param newContainerBasedPathName
     *
     * Change the name / path for a container file.
     * All cells, cell codes and connections are updated accordingly.
     * This expected to be called from command classes which are defined in dceditcommands.h
     *
     * @return pointer to a page object for the new file path or NULL when this failed.
     */
    DCVCPage*               movePage(const QString &oldContainerBasedPathName, const QString &newContainerBasedPathName);

    /**
     * @brief change the path information of a cell.
     * @param cell
     * @param pageNewContainerBasedPathName
     *
     * @note This will be called from DCVCCell::changePageBelonging().
     * This shouldn't be directly called from other methods.
     * @return
     */
    bool                    moveCell(DCCell*cell, const QString& pageNewContainerBasedPathName);

    /**
     * @brief change the path information of a cell code.
     * @param cellcode
     * @param pageNewContainerBasedPathName
     *
     * @note This will be called from DCVCCellCode::changePageBelonging().
     * This shouldn't be directly called from other methods.
     * @return
     */
    bool                    moveCellCodeClass(DCCellCode*cellcode, const QString& pageNewContainerBasedPathName);

    bool                    renameCell(DCCell *cell, const QString& newName);

    bool                    renameCellCodeClass(DCCellCode *cellcode, const QString& newName);

    void                    setContent(DCContent *content);

    void                    unselectCellObjectAll();

    virtual void            setValue(std::string key, float value) { Q_UNUSED(key); Q_UNUSED(value);}

    virtual void            beganParsePage(const char *docRoot, const char *path);
    virtual void            endedParsePage(const char *docRoot, const char *path);

    void                    beganBuildContainer();
    void                    endedBuildContainer();

    bool                    saveCustomScriptToWorkFile(const DCCell *cell, std::string customScript);
    bool                    saveClassScriptToWorkFile(const DCCellCode *cellCode, std::string code);

    QString                 sysFilePathToContainerBasedPath(const QString& sysFilePath);
    QString                 containerBasedPathToSysFilePath(const QString& containerBasedPath);

    QString                 readCustomScriptFromWorkFile(const DCCell *cell);
    QString                 readCellCodeScriptFromFile(const DCCellCode *cellcode);

    /***************************************************************************
     * Cell type rules
     **************************************************************************/

    /**
     * @brief return the avaiable cell types as the list of QString
     * @return
     */
    QList<QString>          getAvailableCellTypes() const;

    QList<QString>          getAvailableScriptableCellTypes() const;

    /**
     * @brief check if given cell type can have script
     * @param type
     * @return true if the given cell type can have script, false if no script avaiable for the cell type.
     */
    bool                    getIsScriptable(const QString& type) const;

    /**
     * @brief return if the cell type can accept receptors.
     *        The cell which inputs value from external can not receive value from internal cells.
     * @param type
     * @return
     */
    bool                    getIsReceptorAvailable(const QString& type) const;

    /***************************************************************************
     * plugin rules
     **************************************************************************/

    bool                    getIsPluginType(const QString& type) const;
    QList<QString>          getAvailablePluginLibraries() const;
    QString                 createPluginCellName(const QString& name, const QString& type, const QString& libraryName);
};

#endif // DCCONTAINER_H
