//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Sep-30, 2012.
//
#include "dcaddcellcodeclassdialog.h"

#include "dccontainer.h"
#include "dccreator.h"
#include "dcscene.h"
#include "dcvcpage.h"
#include "command/dccommand.h"
#include "utils/dcqtitemmodel.h"
#include "dcsinglecolumntableview.h"

#include <QLineEdit>
#include <QTableView>
#include <QHeaderView>

DCAddCellCodeClassDialog::DCAddCellCodeClassDialog(DCCreator *creator, const QString& path, QWidget *parent) :
    QDialog(parent), d_creator(creator)
{
    d_container = d_creator->getCurrentContainer();

    setWindowTitle(tr("Add cell code"));

    d_textField = new QLineEdit;

    d_table = new DCSingleColumnTableView();
    QStringList headers;
    headers << "page";
    d_tableModel = new DCQtItemModel(headers);
    d_tableModel->setReadOnly(0,true);

    d_comboBox = new QComboBox;
    d_comboBox->addItems(QStringList(d_container->getAvailableScriptableCellTypes()));

    DCScene *scene = d_container->getScene();
    const QMap<QString,DCVCPage*> *pages = scene->getPages();
    QMapIterator<QString, DCVCPage*> i(*pages);
    int row = 0;
    int selection = -1;
    while (i.hasNext())
    {
        i.next();
        QString item = i.key();
        d_tableModel->insertString(item);
        if (item == path) selection = row;
        row++;
    }
    d_table->setModel(d_tableModel);

    setLayout(new QVBoxLayout);
    d_table->horizontalHeader()->setResizeMode(QHeaderView::Stretch);

    d_okButton = new QPushButton(tr("OK"));
    d_cancelButton = new QPushButton(tr("Cancel"));
    QHBoxLayout *buttonLayout = new QHBoxLayout;
    buttonLayout->addItem(new QSpacerItem(10,10, QSizePolicy::Expanding));
    buttonLayout->addWidget(d_cancelButton);
    buttonLayout->addWidget(d_okButton);
    d_okButton->setEnabled(false);
    d_cancelButton->setAutoDefault(true);
    d_cancelButton->setDefault(true);

    QGridLayout *glayout = new QGridLayout;
    glayout->addWidget(new QLabel(tr("name")),0,0);
    glayout->addWidget(d_textField,0,1);
    glayout->addWidget(new QLabel(tr("type")),1,0);
    glayout->addWidget(d_comboBox);
    ((QVBoxLayout*)layout())->addLayout(glayout);
    layout()->addWidget(d_table);
    d_statusText = new QLabel;
    d_message = new QLabel;
    d_message->setStyleSheet("color : red;");
    layout()->addWidget(d_statusText);
    layout()->addWidget(d_message);
    ((QVBoxLayout*)layout())->addLayout(buttonLayout);

    connect(d_textField, SIGNAL(textChanged(QString)), this, SLOT(textInputChanged(QString)));
    connect(d_okButton, SIGNAL(clicked()), this, SLOT(okButtonClicked()));
    connect(d_cancelButton, SIGNAL(clicked()), this, SLOT(cancelButtonClicked()));
    connect(d_table,SIGNAL(selectionChangedSignal(QItemSelection,QItemSelection)), this, SLOT(listSelectionChanged(QItemSelection,QItemSelection)));
    connect(d_creator, SIGNAL(commandExecuted(const QUndoCommand*)), this, SLOT(commandExecuted(const QUndoCommand*)));

    if (selection >= 0)
    {
        d_table->selectRow(selection);
    }
}

DCAddCellCodeClassDialog::~DCAddCellCodeClassDialog()
{
    if (d_tableModel)
        d_tableModel->deleteLater();
}


bool DCAddCellCodeClassDialog::checkInput()
{
    d_message->setText("");
    d_statusText->setText("");
    if (d_selectedPagePath.length() == 0)
    {
        d_message->setText(tr("Select a page to place the class"));
    }
    else if (d_textField->text().length() > 0)
    {
        QString path = d_selectedPagePath;
        path.append("#");
        path.append(d_textField->text());
        TKCellCode *cellCode = d_container->getCellCode(path.toStdString());
        if (cellCode)
        {
            d_message->setText(tr("The class name exists"));
        }
        else
        {
            d_okButton->setEnabled(true);
            d_statusText->setText(path);
            return true;
        }
    }
    else
    {
        d_message->setText(tr("Input class name"));
    }
    d_okButton->setEnabled(false);
    return false;
}

void DCAddCellCodeClassDialog::textInputChanged(const QString &text)
{
    (void)text;

    checkInput();
}

void DCAddCellCodeClassDialog::okButtonClicked()
{
    if (checkInput())
    {
        QString path = d_selectedPagePath;
        path.append("#");
        path.append(d_textField->text());
        d_creator->doCommandAddCellCodeClass(this, d_container, path, d_comboBox->currentText());
    }

}

void DCAddCellCodeClassDialog::cancelButtonClicked()
{
    done(false);
}


void DCAddCellCodeClassDialog::listSelectionChanged(const QItemSelection &selected, const QItemSelection &deselected)
{
    (void)deselected;

    if (selected.count()>0)
    {
        d_selectedPagePath = d_tableModel->data(selected.at(0).indexes().at(0), Qt::DisplayRole).toString();
    }
    else
    {
        d_selectedPagePath = "";
    }
    checkInput();
}

QString DCAddCellCodeClassDialog::getAddedCellCodeClassName() const
{
    if (d_selectedPagePath.length() == 0)
    {
        return "";
    }
    else if (d_textField->text().length() > 0)
    {
        QString path = d_selectedPagePath;
        path.append("#");
        path.append(d_textField->text());
        return path;
    }
    return "";
}

void DCAddCellCodeClassDialog::commandExecuted(const QUndoCommand *executedCommand)
{
    const DCCommand *command = dynamic_cast<const DCCommand*>(executedCommand);
    if (command && command->getRequester() == this)
    {
        done(true);
    }
}
