//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Sep-30, 2012.
//
#include "dcvcpage.h"

#include "dcvcomponent.h"
#include "dcvpagecomponent.h"
#include "dcvccell.h"
#include "dcvccellcode.h"
#include "dccell.h"
#include "utils/dccomponentutil.h"
#include "dccontainersaver.h"

#include <QtOpenGL>

DCVCPage::DCVCPage(const QString &locationPath) :
    d_locationPath(locationPath), d_height(0), d_width(10), d_depth(6), d_shouldUpdateShape(true),
    d_colorR(0), d_colorG(0), d_colorB(0), d_colorA(0), d_editCursor(NULL)
{
}

DCVCPage::~DCVCPage()
{
    while (!d_cells.isEmpty())
    {
        DCVPageComponent *cell = d_cells.takeFirst();
        if (cell)
        {
            cell->unregisteredFromPage();
        }
    }
    while (!d_cellCodeClasses.isEmpty())
    {
        DCVPageComponent *cellCodeClass = d_cellCodeClasses.takeFirst();
        if (cellCodeClass)
        {
            cellCodeClass->unregisteredFromPage();
        }
    }
    if (d_editCursor)
    {
        d_editCursor->unregisteredFromPage();
    }
}

void DCVCPage::registerCell(DCVPageComponent *cell)
{
    if (cell)
    {
        if (cell->getPageBelonging())
        {
            cell->getPageBelonging()->unregisterCell(cell);
        }
        d_cells.append(cell);
        cell->regiesteredToPage(this);
    }
}

void DCVCPage::unregisterCell(DCVPageComponent *cell)
{
    if (cell && d_cells.indexOf(cell) >= 0)
    {
        d_cells.removeOne(cell);
        cell->unregisteredFromPage();
    }
}

void DCVCPage::registerCellCodeClass(DCVPageComponent *cellCodeClass)
{
    if (cellCodeClass)
    {
        if (cellCodeClass->getPageBelonging())
        {
            cellCodeClass->getPageBelonging()->unregisterCellCodeClass(cellCodeClass);
        }
        d_cellCodeClasses.append(cellCodeClass);
        cellCodeClass->regiesteredToPage(this);
    }
}

void DCVCPage::unregisterCellCodeClass(DCVPageComponent *cellCodeClass)
{
    if (cellCodeClass && d_cellCodeClasses.indexOf(cellCodeClass) >= 0)
    {
        d_cellCodeClasses.removeOne(cellCodeClass);
        cellCodeClass->unregisteredFromPage();
    }
}

void DCVCPage::registerEditCursor(DCVPageComponent *ecursor)
{
    if (ecursor)
    {
        if (ecursor->getPageBelonging())
        {
            ecursor->getPageBelonging()->unregisterEditCursor();
        }
    }
    d_editCursor = ecursor;
    d_editCursor->regiesteredToPage(this);
}

void DCVCPage::unregisterEditCursor()
{
    d_editCursor->unregisteredFromPage();
    d_editCursor = NULL;
}

bool DCVCPage::moveComponentsTo(DCVCPage *newPage)
{
    QList<DCVPageComponent*> cells = d_cells;
    for (int i = 0; i < cells.size(); i++)
    {
        cells.at(i)->changePageBelonging(newPage);
    }

    QList<DCVPageComponent*> cellCodes = d_cellCodeClasses;
    for (int i = 0; i < cellCodes.size(); i++)
    {
        cellCodes.at(i)->changePageBelonging(newPage);
    }

    if (d_editCursor)
    {
        d_editCursor->changePageBelonging(newPage);
    }
    return true;
}

void DCVCPage::updateVisiblity(bool linkedCellVisible, bool linkedCellRenderFull, bool linkedCellSelectable, bool otherCellVisible, bool otherCellRenderFull, bool otherCellSelectable)
{
    setVisible(linkedCellVisible || otherCellVisible ? DCV_VISIBLE_FULL : DCV_VISIBLE_NONE, linkedCellVisible || otherCellVisible ? DCV_VISIBLE_FULL : DCV_VISIBLE_NONE);

    for (int i = 0; i < d_cells.size(); ++i)
    {
        DCVComponent *vcell = d_cells.at(i);
        if (vcell->getIsSelected())
        {
            vcell->setVisible(DCV_VISIBLE_FULL,DCV_VISIBLE_FULL);
            vcell->setSelectable(true);
        }
        else
        {
            if (linkedCellVisible && otherCellVisible && linkedCellRenderFull == otherCellRenderFull && linkedCellSelectable == otherCellSelectable)
            {
                vcell->setVisible(linkedCellRenderFull ? DCV_VISIBLE_FULL : DCV_VISIBLE_WIREFRAME, linkedCellRenderFull ? DCV_VISIBLE_FULL : DCV_VISIBLE_NONE);
                vcell->setSelectable(linkedCellSelectable);
            }
            else
            {
                if(DCComponentUtil::isConnectedToSelectedCell(vcell))
                {
                    if (linkedCellVisible)
                        vcell->setVisible(linkedCellRenderFull ? DCV_VISIBLE_FULL : DCV_VISIBLE_WIREFRAME, linkedCellRenderFull ? DCV_VISIBLE_FULL : DCV_VISIBLE_NONE);
                    else
                        vcell->setVisible(DCV_VISIBLE_NONE, DCV_VISIBLE_NONE);
                    vcell->setSelectable(linkedCellSelectable);
                }
                else
                {
                    if (otherCellVisible)
                        vcell->setVisible(otherCellRenderFull ? DCV_VISIBLE_FULL : DCV_VISIBLE_WIREFRAME, otherCellRenderFull ? DCV_VISIBLE_FULL : DCV_VISIBLE_NONE);
                    else
                        vcell->setVisible(DCV_VISIBLE_NONE, DCV_VISIBLE_NONE);
                    vcell->setSelectable(otherCellSelectable);
                }
            }
        }
    }
}


void DCVCPage::prepareChildrenForDraw(bool isAnimationInterval)
{
    setWidth(10.0);
    setDepth(6.0);
    for (int i = 0; i < d_cells.size(); ++i)
    {
        d_cells.at(i)->prepareForDraw(isAnimationInterval);
    }
    if (d_editCursor)
    {
        d_editCursor->prepareForDraw(isAnimationInterval);
    }
}

void DCVCPage::drawChildren(bool isAnimationInterval)
{
    for (int i = 0; i < d_cells.size(); ++i)
    {
        d_cells.at(i)->draw(isAnimationInterval);
    }
    if (d_editCursor)
    {
        d_editCursor->draw(isAnimationInterval);
    }
}

void DCVCPage::drawChildrenForSelection(QList<DCVComponent *> *itemList)
{
    for (int i = 0; i < d_cells.size(); ++i)
    {
        d_cells.at(i)->drawForSelection(itemList);
    }
    if (d_editCursor)
    {
        d_editCursor->drawForSelection(itemList);
    }
}

void DCVCPage::translate()
{
    glTranslatef(0,d_height,0);
}

void DCVCPage::setSelected(bool isSelected, bool updateChildren)
{
    DCVComponent::setSelected(isSelected, updateChildren);

    d_shouldUpdateShape = true;

    if (updateChildren)
    {
        for (int i = 0; i < d_cells.size(); ++i)
        {
            d_cells.at(i)->setSelected(isSelected, true);
        }
    }
}

void DCVCPage::setVisible(DCVVisibility visibleSelf, DCVVisibility visibleChildren)
{
    DCVComponent::setVisible(visibleSelf, visibleChildren);
}

void DCVCPage::renderOwnShape(bool isAnimationInterval, bool renderAsWireframe)
{
    (void)isAnimationInterval;
    (void)renderAsWireframe;

    glDisable(GL_DEPTH_TEST);
    glEnable(GL_BLEND);
    glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
    glColor4f(d_colorR, d_colorG, d_colorB, d_colorA);
    glBegin(GL_QUADS);
    glVertex3f(-d_width/2.0f, -0.01f, -d_depth/2.0f);
    glVertex3d(-d_width/2.0f, -0.01f,  d_depth/2.0f);
    glVertex3d( d_width/2.0f, -0.01f,  d_depth/2.0f);
    glVertex3d( d_width/2.0f, -0.01f, -d_depth/2.0f);
    glEnd();
    glDisable(GL_BLEND);
    glEnable(GL_DEPTH_TEST);

    glEndList();
}


void DCVCPage::setHeight(float height)
{
    d_height = height;
}

void DCVCPage::setWidth(float width)
{
    if (d_width != width)
    {
        d_width = width;
        d_shouldUpdateShape = true;
    }
}

void DCVCPage::setDepth(float depth)
{
    if (d_depth != depth)
    {
        d_depth = depth;
        d_shouldUpdateShape = true;
    }
}

void DCVCPage::setColor(float r, float g, float b, float a)
{
    d_colorR = r;
    d_colorG = g;
    d_colorB = b;
    d_colorA = a;
}

bool DCVCPage::startDrag(float x, float y, float z, bool isResizingDrag)
{
    (void)x; (void)y; (void)z; (void)isResizingDrag;

    return false;
}

bool DCVCPage::dragging(float x, float y, float z, bool isResizingDrag)
{
    (void)x; (void)y; (void)z; (void)isResizingDrag;

    return false;
}

bool DCVCPage::endDrag(float x, float y, float z, bool isResizingDrag)
{
    (void)x; (void)y; (void)z; (void)isResizingDrag;

    return false;
}

void DCVCPage::saveAttributesToXML(QDomDocument *document, QDomElement* element) const
{
    QDomElement rootNode = document->createElement("page");
    rootNode.setAttribute("path", d_locationPath);
    for (int i = 0; i < d_cells.length(); i++)
    {
        QDomElement cellNode = document->createElement("cell");
        cellNode.setAttribute("name", QString::fromStdString(d_cells.at(i)->getOwnerCell()->getName()));
        d_cells.at(i)->saveAttributesToXML(document, &cellNode);
        rootNode.appendChild(cellNode);
    }
    element->appendChild(rootNode);
}

void DCVCPage::loadAttributesFromXML(QDomElement element)
{
    QDomNodeList children = element.childNodes();
    for (int i = 0; i < children.size(); i++)
    {
        if (!children.at(i).isElement())
            continue;

        QDomElement e = children.at(i).toElement();
        QString tName = e.tagName();
        if (tName == "page")
        {
            QString path = e.attribute("path", "");
            if (path == d_locationPath)
            {
                QDomNodeList cellNodes = e.childNodes();
                for (unsigned int j = 0; j < cellNodes.length(); j++)
                {
                    if (!cellNodes.at(j).isElement())
                        continue;

                    QDomElement cellNode = cellNodes.at(j).toElement();
                    std::string cellName = cellNode.attribute("name", "").toStdString();

                    for (int k = 0; k < d_cells.length(); k++)
                    {
                        if (cellName == d_cells.at(k)->getOwnerCell()->getName())
                        {
                            d_cells.at(k)->loadAttributesFromXML(cellNode);
                            break;
                        }
                    }
                }
            }
        }
    }
}

void DCVCPage::updateSavedState(DCContainer *container)
{
    DCContainerSaver saver(container);
    d_hash = saver.calculateHashForPage(this);
}

bool DCVCPage::getIsModified(DCContainer *container) const
{
    DCContainerSaver saver(container);
    return d_hash != saver.calculateHashForPage(this);
}
