//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Nov-3, 2012.
//
#include "dccellscriptseditorwindow.h"

#include <QSplitter>
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QGridLayout>
#include <QGroupBox>

#include "dccodeeditor.h"
#include "dccreator.h"
#include "dccontainer.h"
#include "dccell.h"
#include "dccellcode.h"
#include "dccellscriptseditortabwidget.h"
#include "mainwindow.h"
#include "utils/dcutil.h"
#include "dialog/dcassigncellcodeclassdialog.h"

static DCCellScriptsEditorWindow *s_instance = NULL;

DCCellScriptsEditorWindow::DCCellScriptsEditorWindow(DCCreator *creator, QWidget *parent) :
    QDialog(parent), d_creator(creator)
{
    //code editor
    d_tabWidget = new DCCellScriptsEditorTabWidget(this);
    d_tabWidget->setTabsClosable(true);

    //tool
    QWidget *toolWidget = new QWidget;
    QSplitter *toolWidgetSplitter = new QSplitter(Qt::Vertical);

    //cell info tool
    QWidget *cellInfoWidget = new QGroupBox(tr("Cell Information"));
    QGridLayout *cellInfoLayout = new QGridLayout;
    cellInfoLayout->addWidget(new QLabel(tr("Name:")),0,0);
    cellInfoLayout->addWidget(new QLabel(tr("Path")), 1,0);
    cellInfoLayout->addWidget(new QLabel(tr("Type")), 2,0);

    d_cellName = new QLineEdit();
    d_cellName->setReadOnly(true);
    d_cellPath = new QLineEdit();
    d_cellPath->setReadOnly(true);
    d_cellType = new QLineEdit();
    d_cellType->setReadOnly(true);
    cellInfoLayout->addWidget(d_cellName, 0, 1);
    cellInfoLayout->addWidget(d_cellPath, 1, 1);
    cellInfoLayout->addWidget(d_cellType, 2, 1);

    cellInfoWidget->setLayout(cellInfoLayout);

    //cell code info tool
    QWidget *cellCodeInfoWidget = new QWidget;
    d_cellCodePanel = new QStackedLayout;

    QWidget *assignedCellCodeInfoWidget = new QGroupBox(tr("Cell Code Information"));
    QGridLayout *cellCodeInfoLayout = new QGridLayout;
    cellCodeInfoLayout->addWidget(new QLabel(tr("Name:")),0,0);
    cellCodeInfoLayout->addWidget(new QLabel(tr("Path:")),1,0);
    d_cellCodeName = new QLineEdit();
    d_cellCodeName->setReadOnly(true);
    d_cellCodePath = new QLineEdit();
    d_cellCodePath->setReadOnly(true);
    d_unassignCellCodeButton = new QPushButton(tr("Unassign cell code"));
    cellCodeInfoLayout->addWidget(d_cellCodeName, 0,1);
    cellCodeInfoLayout->addWidget(d_cellCodePath, 1,1);
    cellCodeInfoLayout->addWidget(d_unassignCellCodeButton, 2,0,1,2);
    assignedCellCodeInfoWidget->setLayout(cellCodeInfoLayout);

    QWidget *nonCellCodeInfoWidget = new QGroupBox(tr("Cell code - not assigned"));
    QVBoxLayout *nonCellCodeInfoLayout = new QVBoxLayout;
    QLabel *msg = new QLabel(tr("This cell don't have a assinged cell code"));
    msg->setStyleSheet("color:red;");
    nonCellCodeInfoLayout->addWidget(msg);
    d_assignCellCodeButton = new QPushButton(tr("Assign a cell code to this cell..."));
    nonCellCodeInfoLayout->addWidget(d_assignCellCodeButton);
    nonCellCodeInfoWidget->setLayout(nonCellCodeInfoLayout);

    QWidget *cellCodeNotAvaiableInfoWidget = new QGroupBox(tr("Cell code - not avaiable"));
    QVBoxLayout *cellCodeNotAvaiableInforLayout = new QVBoxLayout;
    QLabel *msg2 = new QLabel(tr("This cell's cell type doesn't arrow to have cell code"));
    msg2->setStyleSheet("color:red;");
    cellCodeNotAvaiableInforLayout->addWidget(msg2);
    cellCodeNotAvaiableInfoWidget->setLayout(cellCodeNotAvaiableInforLayout);

    cellCodeInfoWidget->setLayout(d_cellCodePanel);
    d_cellCodePanel->addWidget(assignedCellCodeInfoWidget);
    d_cellCodePanel->addWidget(nonCellCodeInfoWidget);
    d_cellCodePanel->addWidget(cellCodeNotAvaiableInfoWidget);

    toolWidgetSplitter->addWidget(cellInfoWidget);
    toolWidgetSplitter->addWidget(cellCodeInfoWidget);

    QVBoxLayout *toolWidgetLayout = new QVBoxLayout;
    toolWidgetLayout->addWidget(toolWidgetSplitter);
    toolWidgetLayout->addStretch(1);
    toolWidget->setLayout(toolWidgetLayout);

    //buttons
    QHBoxLayout *buttonLayout = new QHBoxLayout;
    d_saveButton = new QPushButton(tr("Save"));
    d_closeButton = new QPushButton(tr("Close"));
    buttonLayout->addStretch();
    buttonLayout->addWidget(d_saveButton);
    buttonLayout->addWidget(d_closeButton);

    // put together
    QSplitter *hsplitter = new QSplitter(Qt::Horizontal, this);
    hsplitter->addWidget(d_tabWidget);
    hsplitter->addWidget(toolWidget);
    hsplitter->setStretchFactor(0,4);
    hsplitter->setStretchFactor(1,1);

    QVBoxLayout *rootLayout = new QVBoxLayout;
    rootLayout->addWidget(hsplitter);
    rootLayout->addLayout(buttonLayout);

    setLayout(rootLayout);

    connect(d_creator, SIGNAL(sceneChanged(const void*,DCScene*)), this, SLOT(sceneChanged(const void*,DCScene*)));
    connect(d_creator, SIGNAL(commandExecuted(const QUndoCommand*)), this, SLOT(creatorCommandExecuted(const QUndoCommand*)));
    connect(d_tabWidget, SIGNAL(activeEditTabChanged(DCCell*)), this, SLOT(editTabChanged(DCCell*)));
    connect(d_tabWidget, SIGNAL(editTabContentModifiedStatusChanged(DCCell*, bool)), this, SLOT(editTabContentModifiedStatusChanged(DCCell*, bool)));
    connect(d_tabWidget, SIGNAL(editTabRemoved(DCCell*)), this, SLOT(editTabRemoved(DCCell*)));
    connect(d_tabWidget, SIGNAL(tabCloseRequested(DCCell*)), this, SLOT(editTabCloseRequested(DCCell*)));

    connect(d_saveButton, SIGNAL(clicked()), this, SLOT(saveButtonClicked()));
    connect(d_closeButton, SIGNAL(clicked()), this, SLOT(close()));
    connect(d_assignCellCodeButton, SIGNAL(clicked()), this, SLOT(assignCellCode()));
    connect(d_unassignCellCodeButton, SIGNAL(clicked()), this, SLOT(unassignCellCode()));

    QByteArray gdata = MainWindow::readSettingsForCellScriptsEditorGeometory();
    if (gdata.length() > 0)
        restoreGeometry(gdata);
}


DCCellScriptsEditorWindow::~DCCellScriptsEditorWindow()
{

}

//static
DCCellScriptsEditorWindow* DCCellScriptsEditorWindow::construct(DCCreator *creator)
{
    if (s_instance)
    {
        delete s_instance;
    }
    s_instance = new DCCellScriptsEditorWindow(creator);
    return s_instance;
}

//static
bool DCCellScriptsEditorWindow::getIsVisible()
{
    if (s_instance)
    {
        return s_instance->isVisible();
    }
    return false;
}

//static
bool DCCellScriptsEditorWindow::startEditing(DCCell *cell, bool focusInCellCodeScriptFirst)
{
    if (s_instance)
    {
        return s_instance->startEditingPrivate(cell, focusInCellCodeScriptFirst);
    }
    return false;
}

//static
bool DCCellScriptsEditorWindow::saveScriptsBelongTo(DCVCPage *page, bool modifiedOnly)
{
    if (s_instance)
    {
        return s_instance->saveScriptsBelongToPrivate(page, modifiedOnly);
    }
    return false;
}

//static
QSet<DCVCPage*> DCCellScriptsEditorWindow::saveScriptsAll(bool modifiedOnly)
{
    QSet<DCVCPage*> empty;

    if (s_instance)
    {
        return s_instance->saveScriptsAllPrivate(modifiedOnly);
    }
    return empty;
}

//static
bool DCCellScriptsEditorWindow::closeEditor()
{
    if (s_instance && s_instance->isVisible())
    {
        return s_instance->close();
    }
    return true;
}

//static
void DCCellScriptsEditorWindow::destroyEditor()
{
    if (s_instance)
    {
        delete s_instance;
        s_instance = NULL;
    }
}


void DCCellScriptsEditorWindow::closeEvent(QCloseEvent *event)
{
    if (maybeSave())
    {
        if (isVisible())
            MainWindow::writeSettingsForCellScriptsEditorGeometory(saveGeometry());
        event->accept();
    }
    else
    {
        event->ignore();
    }
}


bool DCCellScriptsEditorWindow::maybeSave()
{
    bool r = true;
    for (int i = 0; r && i < d_tabWidget->count(); i++)
    {
        r = maybeSave(i);
    }

    if (r)
    {
        d_tabWidget->removeAllTab();
    }

    return r;
}

bool DCCellScriptsEditorWindow::maybeSave(DCCell *cell)
{
    int index = d_tabWidget->getIndexOf(cell);
    if (index < 0)
        return true;

    return maybeSave(index);
}

bool DCCellScriptsEditorWindow::maybeSave(int index)
{
    bool r = true;
    if (d_tabWidget->getIsModified(index))
    {
        d_tabWidget->setCurrentIndex(index);
        QMessageBox::StandardButton ret;
        ret = QMessageBox::warning(this, tr("Cell scripts editor"),
            tr("The cell scripts has been modified.\n"
            "Do you want to save your changes?"),
            QMessageBox::Save | QMessageBox::Discard | QMessageBox::Cancel);
        if (ret == QMessageBox::Save)
        {
            QSet<DCVCPage*> modifiedPages = d_tabWidget->saveContentToFile(index, true);
            d_creator->savePage(modifiedPages, false);
        }
        else if (ret == QMessageBox::Cancel)
        {
            r = false;
        }
    }
    return r;
}

void DCCellScriptsEditorWindow::updateWindowState()
{
    QString windowTitle = "";

    bool modified = false;
    if (d_creator->getScene())
    {
        DCCell *cell = d_tabWidget->getCurrentEditCell();
        if (cell)
        {
            windowTitle.append(QString::fromStdString(cell->getName()));
            if (d_tabWidget->getIsModified(cell))
            {
                windowTitle.append("*");
                modified = true;
            }
            windowTitle.append(" - ");

            d_cellName->setText(QString::fromStdString(cell->getName()));
            d_cellPath->setText(QString::fromStdString(cell->getLocation()));
            d_cellType->setText(cell->getType());
            DCContainer *container = d_creator->getCurrentContainer();

            if (container->getIsScriptableCell(cell))
            {
                d_tabWidget->setCurrentTabReadOnly(false);
                if (cell->getIsCellCodeAssgined())
                {
                    d_cellCodePanel->setCurrentIndex(0);
                    DCCellCode *cellCode = cell->getCellCode();
                    QString fqnName = QString::fromStdString(cellCode->getFQNName());
                    d_cellCodeName->setText(DCUtil::getNameFromFQNPath(fqnName));
                    d_cellCodePath->setText(DCUtil::getContainerBasedPathFromFQNPath(fqnName));
                }
                else
                {
                    d_cellCodePanel->setCurrentIndex(1);
                    d_cellCodeName->setText("");
                    d_cellCodePath->setText("");
                }
            }
            else
            {
                d_tabWidget->setCurrentTabReadOnly(true);
                d_cellCodePanel->setCurrentIndex(2);
                d_cellCodeName->setText("");
                d_cellCodePath->setText("");
            }
        }
    }
    windowTitle.append("Cell Scripts editor");
    setWindowTitle(windowTitle);

    if (modified)
    {
        d_saveButton->setEnabled(true);
        d_saveButton->setDefault(true);
    }
    else
    {
        d_saveButton->setEnabled(false);
        d_closeButton->setDefault(true);
    }
}

//slot
void DCCellScriptsEditorWindow::sceneChanged(const void *, DCScene *)
{
    close();
}

//slot
void DCCellScriptsEditorWindow::editTabChanged(DCCell*)
{
    updateWindowState();
}

//slot
void DCCellScriptsEditorWindow::editTabContentModifiedStatusChanged(DCCell *, bool)
{
    updateWindowState();
}

//slot
void DCCellScriptsEditorWindow::editTabRemoved(DCCell *cell)
{
    (void)cell;

    if (d_tabWidget->count() == 0)
    {
        close();
    }
}

//slot
void DCCellScriptsEditorWindow::editTabCloseRequested(DCCell *editTarget)
{
    if (maybeSave(editTarget))
    {
        d_tabWidget->removeTabForCell(editTarget);
    }
}

//slot
void DCCellScriptsEditorWindow::saveButtonClicked()
{
     QSet<DCVCPage*> modifiedPages = d_tabWidget->saveActiveTabContentToFile(false);
     d_creator->savePage(modifiedPages, false);
}

//slot
void DCCellScriptsEditorWindow::assignCellCode()
{
    DCCell *cell = d_tabWidget->getCurrentEditCell();
    if (!cell)
        return;

    DCAssignCellCodeClassDialog dialog(d_creator,this);
    dialog.exec();
    DCCellCode *cellCode = dialog.getSelectedCellCodeClass();
    if (cellCode)
    {
        d_creator->doCommandAssignCellCodeClassToCell(this, cell , cellCode);
    }
}

//slot
void DCCellScriptsEditorWindow::unassignCellCode()
{
    DCCell *cell = d_tabWidget->getCurrentEditCell();
    if (!cell)
        return;

    d_creator->doCommandUnassignCellCodeClassFromCell(this, cell);
}

void DCCellScriptsEditorWindow::creatorCommandExecuted(const QUndoCommand *command)
{
    (void)command;

    updateWindowState();
}

bool DCCellScriptsEditorWindow::startEditingPrivate(DCCell *cell, bool focusInCellCodeScriptFirst)
{
    d_tabWidget->openTab(cell, focusInCellCodeScriptFirst);

    if (!isVisible())
    {
        QByteArray gdata = MainWindow::readSettingsForCellScriptsEditorGeometory();
        if (gdata.length() > 0)
            restoreGeometry(gdata);
    }
    show();
    activateWindow();
    return true;
}

bool DCCellScriptsEditorWindow::saveScriptsBelongToPrivate(DCVCPage *page, bool modifiedOnly)
{
    bool result;
    for (int i = 0; i < d_tabWidget->count(); i++)
    {
        DCCell *cell = d_tabWidget->getCellForIndex(i);
        DCCellCode *cellCode = NULL;
        if (cell->getIsCellCodeAssgined())
            cellCode = cell->getCellCode();

        if (page == cell->getPageBelonging())
        {
            if (cellCode && page == cellCode->getPageBelonging())
            {
                if (d_tabWidget->saveContentToFile(i, modifiedOnly).count())
                {
                    result = false;
                }
            }
            else
            {
                if (!d_tabWidget->saveCustomScriptOnlyToFile(i, modifiedOnly))
                {
                    result = false;
                }
            }
        }
        else if (cellCode && page == cellCode->getPageBelonging())
        {
            if (!d_tabWidget->saveCellCodeScriptOnlyToFile(i, modifiedOnly))
            {
                result = false;
            }
        }
    }

    return result;
}

QSet<DCVCPage*> DCCellScriptsEditorWindow::saveScriptsAllPrivate(bool modifiedOnly)
{
    QSet<DCVCPage*> modifiedPages;
    for (int i = 0; i < d_tabWidget->count(); i++)
    {
        modifiedPages += d_tabWidget->saveContentToFile(i, modifiedOnly);
    }
    return modifiedPages;
}
