//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Sep-30, 2012.
//
#include "dccell.h"

#include "dcvpagecomponent.h"
#include "dccontainer.h"
#include "dcaxon.h"
#include "dcreceptor.h"
#include "dccellcode.h"
#include "dcaxonterminal.h"

#include "DNUtils.h"

const float DCCell::S_MAXSIZE = 5;
const float DCCell::S_MINSIZE = 0.5;

DCCell::DCCell(DCContainer *container, std::string location, std::string name, std::string type, bool canInputInterface, bool canOutputInterface) :
    TKCell(container, location, name, canInputInterface, canOutputInterface), d_vComponent(NULL), d_cellCode(NULL),
    d_viewSize(0.5), d_viewHeight(0.5)
{
    d_type = QString::fromStdString(type);
}

DCCell::~DCCell()
{
    removeAllConnections();
    if (d_vComponent)
    {
        delete d_vComponent;
        d_vComponent = NULL;
    }
}

void DCCell::bindComponent(DCVPageComponent *component)
{
    d_vComponent = component;
    d_vComponent->updateShape();
}

std::string DCCell::getReceptorName(DCReceptor *receptor) const
{
    for ( TKReceptorMap::const_iterator it = mReceptors.begin(); it != mReceptors.end(); ++it ) {
        if (it->second == receptor)
        {
            return it->first;
        }
    }
    return "";
}

DCReceptor* DCCell::getReceptor(const QString &receptorName)
{
    DCReceptor *receptor = NULL;
    const TKReceptorMap *receptors = getReceptors();
    TKReceptorMap::const_iterator it = receptors->begin();
    while( it != receptors->end())
    {
        QString aReceptorName = QString::fromStdString((*it).first);
        if (receptorName == aReceptorName)
        {
            receptor = dynamic_cast<DCReceptor*>((*it).second);
            break;
        }
        ++it;
    }
    return receptor;
}

bool DCCell::setCellCode(TKCellCode *code, const void *data)
{
    bool r = false;

    if (d_cellCode != code)
    {
        d_cellCode = dynamic_cast<DCCellCode*>(code);
        r = TKCell::setCellCode(code,data);

        if (d_vComponent)
            d_vComponent->updateShape();

        emit cellCodeChanged();
    }
    return r;
}

DCVCPage* DCCell::getPageBelonging() const
{
    return d_vComponent->getPageBelonging();
}

bool DCCell::getIsCellCodeAssgined() const
{
    DCContainer *container = dynamic_cast<DCContainer*>(getContainer());

    return d_cellCode && (container && d_cellCode != container->getEmptyCellCodeClass());
}

DCAxon* DCCell::getAxon() const
{
    return dynamic_cast<DCAxon*>(mAxon);
}

QString DCCell::getCustomScript() const
{
    return ((DCContainer*)mContainer)->readCustomScriptFromWorkFile(this);
}

bool DCCell::saveCustomScript(const QString &script)
{
    return ((DCContainer*)mContainer)->saveCustomScriptToWorkFile(this, script.toStdString());
}

QString DCCell::getType() const
{
    DCContainer *container = dynamic_cast<DCContainer*>(getContainer());

    if (d_cellCode && (container && d_cellCode != container->getEmptyCellCodeClass()))
    {
        return QString::fromStdString(d_cellCode->getCellAPIName());
    }
    else
    {
        return d_type;
    }
}

float DCCell::getViewPageX() const
{
    return d_vComponent->getPageX();
}

float DCCell::getViewPageY() const
{
    return d_vComponent->getPageY();
}

float DCCell::getViewSize() const
{
    return d_viewSize;
}

float DCCell::getViewHeight() const
{
    return d_viewHeight;
}

void DCCell::getViewHCrossPoint(float dx, float dz, float *outX, float *outZ) const
{
    float offset = d_viewSize * 0.5;

    *outX = 0;
    *outZ = 0;

    if (dx == 0)
    {
        *outZ = dz < 0 ? -offset : offset;
    }
    else
    {
        float a = fabs(dz / dx);
        if (a < 1)
        {
            *outX = offset * (dx < 0 ? -1 : 1);
            *outZ = offset * a * (dz < 0 ? -1 : 1);
        }
        else
        {
            *outX = offset / a * (dx < 0 ? -1 : 1);
            *outZ = offset * (dz < 0 ? -1 : 1);
        }
    }
}

void DCCell::getViewVCrossPoint(float dx, float dy, float *outX, float *outY) const
{
    (void)dx; (void)dy; (void)outX; (void)outY;
    //TODO
}

QString DCCell::getWorkFilePathForCustomScript() const
{
    DCContainer *container = dynamic_cast<DCContainer*>(getContainer());
    if (container)
    {
        return container->getWorkFilePathForCustomScript(this);
    }
    return "";
}

void DCCell::setViewPageX(float x)
{
    d_vComponent->setPageX(x);
}

void DCCell::setViewPageY(float y)
{
    d_vComponent->setPageY(y);
}

void DCCell::setViewSize(float size)
{
    if (size > S_MAXSIZE)
        size = S_MAXSIZE;
    else if (size < S_MINSIZE)
        size = S_MINSIZE;

    if (d_viewSize != size)
    {
        d_viewSize = size;
        if (d_vComponent)
            d_vComponent->updateShape();
    }
}

void DCCell::setViewHeight(float _height)
{
    if (d_viewHeight != _height)
    {
        d_viewHeight = _height;
        if (d_vComponent)
            d_vComponent->updateShape();
    }
}

bool DCCell::renameReceptor(const QString &oldName, const QString &newName)
{
    if (newName.length() == 0)
        return false;

    TKReceptorMap::iterator newNameIt = mReceptors.find(newName.toStdString());
    if (newNameIt != mReceptors.end())
        return false;

    TKReceptorMap::iterator it = mReceptors.find(oldName.toStdString());
    if (it != mReceptors.end())
    {
        TKReceptor *receptor = (*it).second;
        mReceptors.erase(it);
        mReceptors.insert( TKReceptorMap::value_type( newName.toStdString(), receptor ) );
        return true;
    }
    return false;
}

bool DCCell::removeReceptor(const QString &name)
{
    TKReceptorMap::iterator it = mReceptors.find(name.toStdString());
    if (it != mReceptors.end())
    {
        TKReceptor *receptor = (*it).second;
        mReceptors.erase(it);
        delete receptor;
        return true;
    }
    return false;
}

bool DCCell::removeReceptor(DCReceptor *receptor)
{
    for ( TKReceptorMap::iterator it = mReceptors.begin(); it != mReceptors.end(); ++it )
    {
        if (it->second == receptor)
        {
            mReceptors.erase(it);
            delete receptor;
            return true;
        }
    }
    return false;
}

DCReceptor* DCCell::createReceptor(const QString &receptorName)
{
    return dynamic_cast<DCReceptor*>(TKCell::createReceptor(receptorName.toStdString()));
}

bool DCCell::removeAllConnections()
{
    TKReceptorMap::iterator it = mReceptors.begin();
    while(it != mReceptors.end())
    {
        DCReceptor *receptor = dynamic_cast<DCReceptor*>(it->second);
        DCAxonTerminal *terminal = receptor->getTarget();
        if (terminal)
        {
            DCAxon *receptorTargetCellAxon = dynamic_cast<DCAxon*>(terminal->getOwner());
            if (receptorTargetCellAxon)
            {
                receptorTargetCellAxon->removeAxonTerminal(terminal);
            }
        }
        mReceptors.erase(it++);
        delete receptor;
    }

    DCAxon *axon = dynamic_cast<DCAxon*>(mAxon);

    for (int i = 0; i < axon->getNumberOfTerminals(); i++)
    {
        DCAxonTerminal *terminal = axon->getTerminalAt(i);
        DCReceptor *axonTargetReceptor = terminal->getTarget();
        DCCell *axonTargetCell = axonTargetReceptor->getOwnerCell();
        if (axonTargetCell)
        {
            axonTargetCell->removeReceptor(axonTargetReceptor);
        }
    }
    return true;
}

void DCCell::changeName(const QString &newName)
{
    mName = newName.toStdString();
    d_vComponent->updateShape();
}

void DCCell::changePath(const QString &newPath)
{
    mLocation = newPath.toStdString();
    d_vComponent->updateShape();
}

void DCCell::changeType(const QString &type)
{
    d_type = type;
    d_vComponent->updateShape();
}
