//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Sep-30, 2012.
//
#include "dcscene.h"

#include "dcvcpage.h"
#include "DNUtils.h"
#include "dccontainer.h"
#include "dccellcode.h"
#include "dcvceditmodecursor.h"

#include <QMapIterator>
#include <QMutex>
#include <QtOpenGL>

bool DCScene::sShouldContinueAnimation = false;

DCScene::DCScene(DCContainer *owner)
    : d_owner(owner),
      d_vpMode(DCV_PERSMODE_NAVIGATION), d_veMode(DCV_EDITMODE_LAYOUT), d_transition(DC_TRANS_NONE),
      d_scale(200), d_xAngle(45), d_yAngle(20),
      d_centerX(0), d_centerBrowsModeY(0), d_centerEditModeY(0),
      d_navigationModeSelectedPageAlpha(0.9f), d_navigationModeAboveAlpha(0.15f), d_navigationModeBelowAlpha(0.4f),
      d_pageEditModeSelectedPageAlpha(0.8f),
      d_navigationModeSelectedCellMovable(true), d_navigationModeOtherCellSelectable(true), d_navigationModeOtherCellVisible(true),
      d_navigationModeOtherCellRenderFull(true),
      d_pageEditModeOtherSamePageRenderFull(true),
      d_pageEditModeLinkedOtherPageVisible(true), d_pageEditModeLinkedOtherPageRenderFull(false),
      d_pageEditModeOtherOtherPageVisible(true), d_pageEditModeOtherOtherPageRenderFull(false),
      d_editAxon(NULL), d_editReceptor(NULL), d_editCellCodeCell(NULL)
{
    d_terminalEditCursor = new DCVCEditModeCursor;
    connect(d_terminalEditCursor, SIGNAL(pageBelongingChanged(DCVCPage*)), this, SLOT(slotTerminalEditCursorBelongingPageChagend(DCVCPage*)));
    connect(d_terminalEditCursor, SIGNAL(dropTargetChanged(DCVComponent*)), this, SLOT(slotTerminalEditCursorDropTargetChanged(DCVComponent*)));
}

DCScene::~DCScene()
{
    while (!d_pages.isEmpty())
    {
        QMap<QString, DCVCPage*>::iterator i = d_pages.begin();
        delete i.value();
        d_pages.take(i.key());
    }

    d_terminalEditCursor->disconnect(this);
    if (d_terminalEditCursor)
        delete d_terminalEditCursor;
}

void DCScene::initMode(DCVPersMode persMode, DCVEditMode editMode)
{
    d_vpMode = persMode;
    d_veMode = editMode;
    emit modeInitialized();
}

bool DCScene::getIsModified() const
{
    bool modified = false;
    QMapIterator<QString, DCVCPage*> i(d_pages);
    while (i.hasNext())
    {
        i.next();
        if (i.value()->getIsModified(d_owner))
        {
            modified = true;
            break;
        }
    }
    return modified;
}

DCVCPage* DCScene::getPage(const std::string &location) const
{
    QMutexLocker locker(getSceneLock());

    QMap<QString, DCVCPage*>::const_iterator i = d_pages.find(QString::fromStdString(location));
    if (i != d_pages.end())
    {
        return i.value();
    }
    return NULL;
}

bool DCScene::getIsPageExist(const std::string &location) const
{
    return getIsPageExist(QString::fromStdString(location));
}

bool DCScene::getIsPageExist(const QString &location) const
{
    QMutexLocker locker(getSceneLock());

    QMap<QString, DCVCPage*>::const_iterator i = d_pages.find(location);
    return i != d_pages.end();
}

QString DCScene::getSceneSettingFilePathForPage(DCVCPage *page) const
{
    return getSceneSettingFilePathForPage(QString::fromStdString(d_owner->getContainerRootPath()), page);
}

QString DCScene::getSceneSettingFilePathForPage(const QString &containerRootPath, const DCVCPage *page) const
{
    QString pageFilePath =  containerRootPath + page->getLocationPath();
    return pageFilePath + ".dcvxml";
}

DCVCPage* DCScene::addPage(const std::string &location)
{
    QMutexLocker locker(getSceneLock());
    QString qlocation = QString::fromStdString(location);
    QMap<QString, DCVCPage*>::const_iterator i = d_pages.find(qlocation);
    if (i != d_pages.end())
    {
        return i.value();
    }
    DCVCPage *newPage = new DCVCPage(qlocation);
    d_pages.insert(qlocation, newPage);

    return newPage;
}

bool DCScene::removePage(DCVCPage *page)
{
    QMutexLocker locker(getSceneLock());
    bool removed = false;
    QMap<QString,DCVCPage*>::Iterator it = d_pages.begin();
    while(it != d_pages.end())
    {
        if (it.value() == page)
        {
            d_pages.erase(it);
            delete page;
            removed = true;
            break;
        }
        it++;
    }

    return removed;
}

void DCScene::setScale(const void *requester, float scale)
{
    d_scale = scale;
    emit viewScaleChanged(requester);
}

void DCScene::setAngle(const void *requester, float xangle, float yangle)
{
    d_xAngle = xangle;
    d_yAngle = yangle;
    emit viewAngleChanged(requester);
}

void DCScene::setBrowsModeCenter(const void *requester, float cx, float cy)
{
    d_centerX = cx;
    d_centerBrowsModeY = cy;
    emit viewCenterChanged(requester);
}

void DCScene::setPageModeCenter(const void *requester, float cx, float cy)
{
    d_centerX = cx;
    d_centerEditModeY = cy;
    emit viewCenterChanged(requester);
}

bool DCScene::selectPage(const void *requester, const DCVCPage *page, bool multipleSelection)
{
    QMutexLocker locker(getSceneLock());
    bool pageFound = false;

    QMapIterator<QString, DCVCPage*> i(d_pages);
    while (i.hasNext())
    {
        i.next();
        if (i.value() == page)
        {
            pageFound = true;
            i.value()->setSelected(true,false);
        }
        else if (!multipleSelection && i.value()->getIsSelected())
        {
            i.value()->setSelected(false,false);
        }
    }

    if (pageFound)
    {
        emit selectedPageChanged(requester);
    }
    return pageFound;
}

bool DCScene::unselectPage(const void *requester, const DCVCPage *page)
{
    QMutexLocker locker(getSceneLock());
    bool pageFound = false;

    QMapIterator<QString, DCVCPage*> i(d_pages);
    while (i.hasNext())
    {
        i.next();
        if (i.value() == page)
        {
            pageFound = true;
            i.value()->setSelected(false,false);
        }
    }

    if (pageFound)
    {
        emit selectedPageChanged(requester);
    }
    return pageFound;
}

bool DCScene::selectPage(const void *requester, const QString &locationPath, bool multipleSelection)
{
    QMutexLocker locker(getSceneLock());
    bool pageFound = false;

    QMapIterator<QString, DCVCPage*> i(d_pages);
    while (i.hasNext())
    {
        i.next();
        if (i.value()->getLocationPath() == locationPath)
        {
            pageFound = true;
            i.value()->setSelected(true,false);
        }
        else if (!multipleSelection && i.value()->getIsSelected())
        {
            i.value()->setSelected(false,false);
        }
    }

    if (pageFound)
    {
        emit selectedPageChanged(requester);
    }
    return pageFound;
}

bool DCScene::unselectPage(const void *requester, const QString &locationPath)
{
    QMutexLocker locker(getSceneLock());
    bool pageFound = false;

    QMapIterator<QString, DCVCPage*> i(d_pages);
    while (i.hasNext())
    {
        i.next();
        if (i.value()->getLocationPath() == locationPath)
        {
            pageFound = true;
            i.value()->setSelected(false,false);
        }
    }

    if (pageFound)
    {
        emit selectedPageChanged(requester);
    }
    return pageFound;
}

bool DCScene::unselectPageAll(const void *requester)
{
    QMutexLocker locker(getSceneLock());

    QMapIterator<QString, DCVCPage*> i(d_pages);
    while (i.hasNext())
    {
        i.next();
        i.value()->setSelected(false,false);
    }

    emit selectedPageChanged(requester);
    return true;
}

void DCScene::selectCellObject(const void *requester, DCVComponent *object, bool multipleSelection)
{
    QMutexLocker locker(getSceneLock());

    bool objectSelected = false;
    bool otherObjectSelected = false;

    QList<DCVComponent*> list = d_owner->getSelectedCellObjects();
    for (int i = 0; i < list.length(); i++)
    {
        if (list.at(i) == object)
        {
            objectSelected = true;
        }
        else
        {
            otherObjectSelected = true;
        }
    }

    if ((multipleSelection && objectSelected) || (!multipleSelection && objectSelected && !otherObjectSelected))
    {
        //object selected already
        return;
    }

    if (!multipleSelection)
    {
        d_owner->unselectCellObjectAll();
    }
    object->setSelected(true,false);

    updateVisiblity();
    emit selectedCellObjectChanged(requester);
}

void DCScene::unselectCellObject(const void *requester, DCVComponent *object)
{
    QMutexLocker locker(getSceneLock());

    if (!object->getIsSelected())
    {
        return;
    }

    object->setSelected(false,false);

    updateVisiblity();
    emit selectedCellObjectChanged(requester);
}

void DCScene::unselectCellObjectAll(const void *requester)
{
    QMutexLocker locker(getSceneLock());

    QList<DCVComponent*> list = d_owner->getSelectedCellObjects();
    if (list.size() == 0)
    {
        return;
    }

    d_owner->unselectCellObjectAll();

    updateVisiblity();
    emit selectedCellObjectChanged(requester);
}

int DCScene::getNumberOfSelectedPages() const
{
    QMutexLocker locker(getSceneLock());
    int cnt = 0;
    QMapIterator<QString, DCVCPage*> i(d_pages);
     while (i.hasNext())
     {
        i.next();
        if (i.value()->getIsSelected())
            cnt++;
     }
     return cnt;
}

QList<DCVCPage*> DCScene::getSelectedPages() const
{
    QList<DCVCPage*> list;

    QMutexLocker locker(getSceneLock());
    QMapIterator<QString, DCVCPage*> i(d_pages);
     while (i.hasNext())
     {
        i.next();
        if (i.value()->getIsSelected())
            list.append(i.value());
     }
     return list;
}

QList<DCVComponent*> DCScene::getSelectedCellObjects() const
{
    QMutexLocker lockr(getSceneLock());
    return d_owner->getSelectedCellObjects();
}

void DCScene::updateVisiblity()
{
    QMapIterator<QString, DCVCPage*> i(d_pages);


    bool sp1,sp2,sp3,sp4,sp5,sp6;
    bool np1,np2,np3,np4,np5,np6;

    //xp1:linkedCellVisible
    //xp2:linkedCellRenderFull
    //xp3:linkedCellSelectable
    //xp4:otherCellVisible
    //xp5:otherCellRenderFull
    //xp6:otherCellSelectable
    //selected pages    : x <= s
    //non-selected pages: x <= n

    if (d_veMode == DCV_EDITMODE_LAYOUT)
    {
        switch(d_vpMode)
        {
        case DCV_PERSMODE_NAVIGATION:
            sp1 = true;
            sp2 = true;
            sp3 = true;
            sp4 = d_navigationModeOtherCellVisible;
            sp5 = d_navigationModeOtherCellRenderFull;
            sp6 = d_navigationModeOtherCellSelectable;

            np1 = true;
            np2 = true;
            np3 = true;
            np4 = d_navigationModeOtherCellVisible;
            np5 = d_navigationModeOtherCellRenderFull;
            np6 = d_navigationModeOtherCellSelectable;
            break;
        case DCV_PERSMODE_PAGEEDIT:
            sp1 = true;
            sp2 = true;
            sp3 = true;
            sp4 = true;
            sp5 = d_pageEditModeOtherSamePageRenderFull;
            sp6 = true;

            np1 = d_pageEditModeLinkedOtherPageVisible;
            np2 = d_pageEditModeLinkedOtherPageRenderFull;
            np3 = false;
            np4 = d_pageEditModeOtherOtherPageVisible;
            np5 = d_pageEditModeOtherOtherPageRenderFull;
            np6 = false;
            break;
        }
    }
    else if (d_veMode == DCV_EDITMODE_TERMINAL_FROM_AXON || d_veMode == DCV_EDITMODE_TERMINAL_FROM_RECEPTOR)
    {
        if (d_terminalEditCursor->getIsDragging())
        {
            switch(d_vpMode)
            {
            case DCV_PERSMODE_NAVIGATION:
                sp1 = true;
                sp2 = true;
                sp3 = true;
                sp4 = d_navigationModeOtherCellVisible;
                sp5 = d_navigationModeOtherCellRenderFull;
                sp6 = d_navigationModeOtherCellSelectable;

                np1 = true;
                np2 = false;
                np3 = false;
                np4 = d_navigationModeOtherCellVisible;
                np5 = false;
                np6 = false;
                break;
            case DCV_PERSMODE_PAGEEDIT:
                sp1 = true;
                sp2 = true;
                sp3 = true;
                sp4 = true;
                sp5 = d_pageEditModeOtherSamePageRenderFull;
                sp6 = true;

                np1 = d_pageEditModeLinkedOtherPageVisible;
                np2 = d_pageEditModeLinkedOtherPageRenderFull;
                np3 = false;
                np4 = d_pageEditModeOtherOtherPageVisible;
                np5 = d_pageEditModeOtherOtherPageRenderFull;
                np6 = false;
                break;
            }
        }
        else
        {
            switch(d_vpMode)
            {
            case DCV_PERSMODE_NAVIGATION:
                sp1 = true;
                sp2 = false;
                sp3 = true;
                sp4 = d_navigationModeOtherCellVisible;
                sp5 = false;
                sp6 = false;

                np1 = true;
                np2 = false;
                np3 = false;
                np4 = d_navigationModeOtherCellVisible;
                np5 = false;
                np6 = false;
                break;

            case DCV_PERSMODE_PAGEEDIT:
                sp1 = true;
                sp2 = false;
                sp3 = true;
                sp4 = true;
                sp5 = false;
                sp6 = true;

                np1 = d_pageEditModeLinkedOtherPageVisible;
                np2 = false;
                np3 = false;
                np4 = false;
                np5 = false;
                np6 = false;
                break;
            }
        }
    }
    else
    {
        Q_ASSERT(0);
    }

    while (i.hasNext())
    {
        i.next();
        if (i.value())
        {
            DCVCPage *page = i.value();

            if (page->getIsSelected())
            {
                page->updateVisiblity(sp1, sp2, sp3, sp4, sp5, sp6);
            }
            else
            {
                page->updateVisiblity(np1, np2, np3, np4, np5, np6);
            }
        }
    }

}

void DCScene::draw(bool isAnimationInterval)
{

    getSceneLock()->lock();

    {
        QMapIterator<QString, DCVCPage*> i(d_pages);
        //todo
        float y = 0;

        while (i.hasNext())
        {
           i.next();
           glPushMatrix();

           //TODO
           glTranslatef(0,y,0);
           i.value()->prepareForDraw(isAnimationInterval);
           glPopMatrix();
           y += 1;
        }
    }

    {
        QMapIterator<QString, DCVCPage*> i(d_pages);

        //todo
        float y = 0;
        float selectedPageAlpha;
        float nonSelectedPageAlpha;

        switch (d_vpMode)
        {
        case DCV_PERSMODE_NAVIGATION:
            selectedPageAlpha = d_navigationModeSelectedPageAlpha;
            nonSelectedPageAlpha = d_navigationModeBelowAlpha;
            break;

        case DCV_PERSMODE_PAGEEDIT:
            selectedPageAlpha = d_pageEditModeSelectedPageAlpha;
            nonSelectedPageAlpha = 0;
            break;

        }

        while (i.hasNext())
        {
           i.next();
           glPushMatrix();

           //TODO
           DCVCPage *page = i.value();
           if (page->getIsSelected())
           {
               page->setColor(0.4f, 0.4f, 1, selectedPageAlpha);
               if (d_vpMode == DCV_PERSMODE_NAVIGATION)
               {
                   nonSelectedPageAlpha = d_navigationModeAboveAlpha;
               }
               else
               {
                   selectedPageAlpha = 0;
               }
           }
           else
           {
               page->setColor(0.2f, 0.2f, 0.6f, nonSelectedPageAlpha);
           }
           glTranslatef(0,y,0);
           page->draw(isAnimationInterval);
           glPopMatrix();
           y += 1;
        }
    }
}

void DCScene::drawForSelection(QList<DCVComponent*> *itemList)
{
    //todo
    float y = 0;

    getSceneLock()->lock();

    QMapIterator<QString, DCVCPage*> i(d_pages);
    while (i.hasNext())
    {
        i.next();
        glPushMatrix();

        //TODO
        glTranslatef(0,y,0);
        y += 1;
        i.value()->drawForSelection(itemList);
        glPopMatrix();
    }
    getSceneLock()->unlock();
}

bool DCScene::doAnimation()
{
    const int step = 10;

    bool shouldContinue = false;

    if (sShouldContinueAnimation)
    {
        switch(d_transition)
        {
        case DC_TRANS_TO_PAGEEDIT:
            //xangle -> 90, yangle -> 0, step 10
            if (abs(d_xAngle - 90) < step)
            {
                d_xAngle = 90;
            }
            else
            {
                d_xAngle += d_xAngle < 90 ? step : -step;
                shouldContinue = true;
            }
            if (abs(d_yAngle) < step || abs(d_yAngle - 360) < step)
            {
                d_yAngle = 0;
            }
            else
            {
                d_yAngle += d_yAngle < 0 || d_yAngle > 180 ? step : -step;
                shouldContinue = true;
            }
            if (!shouldContinue)
                updateVisiblity();

            emit viewAngleChanged(this);
            break;

        case DC_TRANS_TO_NAVIGATION:
        {
            //xangle -> 45, yangle -> 20
            float dxAngle = abs(d_xAngle - 45);
            if (dxAngle < step)
            {
                d_xAngle = 45;
            }
            else
            {
                d_xAngle += d_xAngle < 45 ? step : -step;
                shouldContinue = true;
            }
            if (abs(d_yAngle - 20) < step)
            {
                d_yAngle = 20;
            }
            else
            {
                d_yAngle += d_yAngle < 20 ? step : -step;
                shouldContinue = true;
            }

            if (shouldContinue)
            {
                int ls = dxAngle / step;
                if (ls > 0)
                    d_centerEditModeY -= d_centerEditModeY / ls;
                else
                    d_centerEditModeY = 0;
            }
            else
            {
                d_centerEditModeY = 0;
            }

            emit viewAngleChanged(this);
        }
            break;

        case DC_TRANS_NONE:
            break;

        }

        if (!shouldContinue)
            d_transition = DC_TRANS_NONE;
    }

    return shouldContinue;
}

bool DCScene::changePersMode(const void *requester, DCVPersMode mode)
{
    if (mode == d_vpMode)
        return false;

    d_vpMode = mode;
    switch(mode)
    {
        case DCV_PERSMODE_PAGEEDIT:
    {
            d_centerEditModeY = 0;
            QList<DCVCPage*> pages = getSelectedPages();
            QList<DCVComponent*> objs = getSelectedCellObjects();
            for (int i = 0; i < objs.length() ; i++)
            {
                bool isInSelectedPage = false;
                DCVCPage *page = objs.at(i)->getPageBelonging();
                for (int j = 0; j < pages.length() ; j++)
                {
                    if (page == pages.at(j))
                    {
                        isInSelectedPage = true;
                        break;
                    }
                }
                if (!isInSelectedPage)
                {
                    unselectCellObject(this, objs.at(i));
                }
            }
            d_transition = DC_TRANS_TO_PAGEEDIT;
    }
            break;

        case DCV_PERSMODE_NAVIGATION:
            d_transition = DC_TRANS_TO_NAVIGATION;
            updateVisiblity();
            break;

    }
    setShouldContinueAnimation(true);
    emit viewPersModeChanged(requester);

    return true;
}

bool DCScene::startTerminalEditForAxon(const void *requester, DCAxon *ownerAxon)
{
    d_veMode = DCV_EDITMODE_TERMINAL_FROM_AXON;
    d_editAxon = ownerAxon;
    d_terminalEditCursor->setOwnerFromAxon(ownerAxon);

    updateVisiblity();
    emit viewEditModeChanged(requester);

    return true;
}

bool DCScene::startTerminalEditForReceptor(const void *requester, DCReceptor *ownerReceptor)
{
    d_veMode = DCV_EDITMODE_TERMINAL_FROM_RECEPTOR;
    d_editReceptor = ownerReceptor;
    d_terminalEditCursor->setOwnerFromReceptor(ownerReceptor);

    updateVisiblity();
    emit viewEditModeChanged(requester);

    return true;
}

bool DCScene::finishTerminalEdit(const void *requester)
{
    d_veMode = DCV_EDITMODE_LAYOUT;
    d_terminalEditCursor->unsetCursor();
    updateVisiblity();
    emit viewEditModeChanged(requester);

    return true;
}

void  DCScene::setNavModeSelectedPageAlpha(const void *requester, float alpha)
{
    d_navigationModeSelectedPageAlpha = alpha;
    emit viewSettingChanged(requester);
}

void  DCScene::setNavModeAboveAlpha(const void *requester, float alpha)
{
    d_navigationModeAboveAlpha = alpha;
    emit viewSettingChanged(requester);
}

void  DCScene::setNavModeBelowAlpha(const void *requester, float alpha)
{
    d_navigationModeBelowAlpha = alpha;
    emit viewSettingChanged(requester);
}

void  DCScene::setPageModeSelectedAlpha(const void *requester, float alpha)
{
    d_pageEditModeSelectedPageAlpha = alpha;
    emit viewSettingChanged(requester);
}

void  DCScene::setNavModeSelectedCellMovable(const void *requester, bool b)
{
    d_navigationModeSelectedCellMovable = b;
    emit viewSettingChanged(requester);
}

void  DCScene::setNavModeOtherCellSelectable(const void *requester, bool b)
{
    d_navigationModeOtherCellSelectable  = b;
    updateVisiblity();
    emit viewSettingChanged(requester);
}

void  DCScene::setNavModeOtherCellVisible(const void *requester, bool b)
{
    d_navigationModeOtherCellVisible  = b;
    updateVisiblity();
    emit viewSettingChanged(requester);
}

void  DCScene::setNavModeOtherCellRenderFull(const void *requester, bool b)
{
    d_navigationModeOtherCellRenderFull = b;
    updateVisiblity();
    emit viewSettingChanged(requester);
}

void DCScene::setPageModeOtherSamePageRenderFull(const void *requester, bool b)
{
    d_pageEditModeOtherSamePageRenderFull = b;
    updateVisiblity();
    emit viewSettingChanged(requester);
}

void DCScene::setPageModeLinkedOtherPageVisible(const void *requester, bool b)
{
    d_pageEditModeLinkedOtherPageVisible = b;
    updateVisiblity();
    emit viewSettingChanged(requester);
}

void DCScene::setPageModeLinkedOtherPageRenderFull(const void *requester, bool b)
{
    d_pageEditModeLinkedOtherPageRenderFull = b;
    updateVisiblity();
    emit viewSettingChanged(requester);
}

void DCScene::setPageModeOtherOtherPageVisible(const void *requester, bool b)
{
    d_pageEditModeOtherOtherPageVisible = b;
    updateVisiblity();
    emit viewSettingChanged(requester);
}

void DCScene::setPageModeOtherOtherPageRenderFull(const void *requester, bool b)
{
    d_pageEditModeOtherOtherPageRenderFull = b;
    updateVisiblity();
    emit viewSettingChanged(requester);
}

QString DCScene::getTerminalEditCursorLocationString() const
{
    return d_terminalEditCursor->getPageBelonging()->getLocationPath();
}

DCVComponent* DCScene::getTerminalEditCursorDropTarget() const
{
    return d_terminalEditCursor->getDropTarget();
}

void DCScene::slotTerminalEditCursorBelongingPageChagend(DCVCPage *newPage)
{
    emit terminalEditCursorBelongingPageChagend(newPage);
}

void DCScene::slotTerminalEditCursorDropTargetChanged(DCVComponent *newTarget)
{
    emit terminalEditCursorDropTargetChanged(newTarget);
}

bool DCScene::loadSceneAll()
{
    bool r = true;
    QMapIterator<QString, DCVCPage*> i(d_pages);
    while (i.hasNext())
    {
        i.next();
        if (!loadSceneForPage(i.value()))
        {
            r = false;
        }
    }
    return r;
}

bool DCScene::loadSceneForPage(DCVCPage *page)
{
    QString settingFilePath = getSceneSettingFilePathForPage(page);

    QDomDocument doc;
    QFile file(settingFilePath);
    bool r = true;
    if (file.exists())
    {
        QString errMsg;
        int errLine = 0;
        r = doc.setContent(&file, &errMsg, &errLine);

        if (r)
        {
            page->loadAttributesFromXML(doc.firstChildElement("denncoCreator"));
        }
    }

    page->updateSavedState(d_owner);

    return r;
}

bool DCScene::saveSceneForPage(const QString &containerRootPath, const DCVCPage *page)
{
    QString settingFilePath = getSceneSettingFilePathForPage(containerRootPath, page);

    QDomDocument doc;
    QDomElement root = doc.createElement("denncoCreator");
    page->saveAttributesToXML(&doc, &root);
    doc.appendChild(root);

    QFile file(settingFilePath);
    bool r = false;
    if (file.open(QFile::WriteOnly | QFile::Truncate))
    {
        QTextStream out(&file);
        doc.save(out, 4);
        r = true;
    }
    file.close();
    return r;
}

//static
QMutex* DCScene::getSceneLock()
{
    static QMutex sLock(QMutex::Recursive);
    return &sLock;
}

//static
void DCScene::lockScene()
{
    getSceneLock()->lock();
}

//static
void DCScene::unlockScene()
{
    getSceneLock()->unlock();
}

//static
void DCScene::setShouldContinueAnimation(bool shouldContinue)
{
    sShouldContinueAnimation = shouldContinue;
}

//static
bool DCScene::getShouldContinueAnimation()
{
    return sShouldContinueAnimation;
}

