//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Sep-30, 2012.
//
#ifndef DCCELL_H
#define DCCELL_H

#include "TKCell.h"

class DCContainer;
class DCAxon;
class DCAxonTerminal;
class DCReceptor;
class DCCellCode;
class DCVComponent;
class DCVPageComponent;
class DCVCPage;

#include <QObject>
#include <QString>

class DCCell : public QObject, public TKCell
{
    Q_OBJECT

    friend struct DCComponentUtil;
    friend class DCContainer;

    DCCell(DCContainer *container, std::string location, std::string name, std::string type, bool canInputInterface, bool canOutputInterface);
    void bindComponent(DCVPageComponent *component);

    DCVPageComponent    *d_vComponent;
    DCCellCode          *d_cellCode;
    QString             d_type;
    float               d_viewSize;
    float               d_viewHeight;

    const static float S_MINSIZE;
    const static float S_MAXSIZE;

    bool                removeAllConnections();

    /**
      * Change name of this cell.
      * DCContainer will call this method.
      */
    void                changeName(const QString& newName);

    /**
      * Change the location path for this cell.
      * DCContainer will call this method.
      */
    void                changePath(const QString& newPath);


public:
    virtual ~DCCell();

    virtual bool        doTick(float time) { (void) time; return false;}
    virtual bool        doInit() {return false;}
    virtual bool        doDestroy() {return false;}
    virtual bool        setCellCode(TKCellCode *code, const void *data);

    std::string         getLocation() const { return mLocation; }
    DCVPageComponent *  getVComponent() const { return d_vComponent; }
    DCVCPage*           getPageBelonging() const;
    DCCellCode*         getCellCode() const { return d_cellCode; }
    bool                getIsCellCodeAssgined() const;
    DCAxon*             getAxon() const;
    std::string         getReceptorName(DCReceptor *receptor) const;
    DCReceptor*         getReceptor(const QString& receptorName);
    QString             getCustomScript() const;
    QString             getType() const;
    float               getViewPageX() const;
    float               getViewPageY() const;
    float               getViewSize() const;
    float               getViewHeight() const;
    void                getViewHCrossPoint(float dx, float dz, float *outX, float *outZ) const;
    void                getViewVCrossPoint(float dx, float dy, float *outX, float *outY) const;
    QString             getWorkFilePathForCustomScript() const;

    void                setViewPageX(float x);
    void                setViewPageY(float y);
    void                setViewSize(float size);
    void                setViewHeight(float height);
    bool                saveCustomScript(const QString& script);

    /* Following methods are expected to be called from command classes (defined in dceditcommands.h).
     */

    /**
     * Rename receptor name
     */
    bool                renameReceptor(const QString& oldName, const QString& newName);

    /**
     * Remove receptor from the list
     * Note that this remove receptor object but won't remove the axon terminal connected to the receptor.
     * This method is expected to be only called from command classes (defined in dceditcommands.h )
     */
    bool                removeReceptor(const QString &name);


    /**
     * Remove receptor from the list
     * Note that this remove receptor object but won't remove the axon terminal connected to the receptor.
     * This method is expected to be only called from command classes (defined in dceditcommands.h )
     */
    bool                removeReceptor(DCReceptor *receptor);

    /**
     * Create a receptor with the given name and register it into this cell.
     * This method is expected to be only called from command classes (defined in dceditcommands.h )
     */
    virtual DCReceptor* createReceptor(const QString &receptorName);

    /**
     * Change cell code type.
     * change of cell's own type attribute only take effort when no cell code class is assined to this cell.
     * In the case a cell code class is assiend to this cell, type of this cell rely on the cell code class's API type.
     */
    void                changeType(const QString& type);

signals:
    void    cellCodeChanged();


};

#endif // DCCELL_H
