//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Sep-30, 2012.
//
#ifndef DCVCOMPONENT_H
#define DCVCOMPONENT_H

#include <QtGui>
#include <QDomElement>

class DCVCPage;
class DCCell;
class DCVCRefHolder;

class DCVComponent
{
friend class DCVCRefHolder;

public:
    enum DCVVisibility {DCV_VISIBLE_FULL, DCV_VISIBLE_WIREFRAME, DCV_VISIBLE_NONE };

private:
    double                  d_modelview[16];
    DCVVisibility           d_isVisible;
    bool                    d_isSelectable;
    bool                    d_isSelected;

    QList<DCVCRefHolder*>   d_refs;
    void                    addRefHolder(DCVCRefHolder* holder);
    void                    removeRefHolder(DCVCRefHolder* holder);

public:
    DCVComponent();
    virtual ~DCVComponent();

    virtual DCVCPage *      getPageBelonging() const = 0;
    virtual DCCell*         getOwnerCell() const = 0;
    virtual bool            isResizingArea(float x, float y, float z) const = 0;
    virtual DCVVisibility   getIsVisible() const { return d_isVisible; }
    virtual bool            getIsSelectable() const { return d_isSelectable; }
    virtual bool            getIsSelected() const { return d_isSelected; }
    virtual const double *  getModelViewMatrix() const { return d_modelview; }

    virtual void    prepareChildrenForDraw(bool isAnimationInterval) = 0;
    virtual void    drawChildren(bool isAnimationInterval) = 0;
    virtual void    drawChildrenForSelection(QList<DCVComponent*> *itemList) = 0;
    virtual void    translate() = 0;
    virtual void    setVisible(DCVVisibility visibleSelf, DCVVisibility childrenAreVisible) { Q_UNUSED(childrenAreVisible); d_isVisible = visibleSelf; }
    virtual void    setSelectable(bool selectable) { d_isSelectable = selectable; }
    virtual void    setSelected(bool selectedSelf, bool updateChildren) { Q_UNUSED(updateChildren); d_isSelected = selectedSelf; }
    virtual void    renderOwnShape(bool isAnimationInterval, bool renderAsWireframe) = 0;

    virtual bool    startDrag(float x, float y, float z, bool isResizingDrag) = 0;
    virtual bool    dragging(float x, float y, float z, bool isResizingDrag) = 0;
    virtual bool    endDrag(float x, float y, float z, bool isResizingDrag) = 0;

    virtual void    prepareForDraw(bool isAnimationInterval);
    virtual void    draw(bool isAnimationInterval);
    virtual void    drawForSelection(QList<DCVComponent*> *itemList);
    virtual void    renderOwnShapeForSelection();
    virtual void    updateShape() = 0;

    virtual void    saveAttributesToXML(QDomDocument *document, QDomElement* element) const = 0;
    virtual void    loadAttributesFromXML(QDomElement element) = 0;

};

class DCVCRefHolder
{
    DCVComponent *d_ref;
public:
    DCVCRefHolder();
    ~DCVCRefHolder();

    void assign(DCVComponent *component);
    DCVComponent* ref() const { return d_ref; }
    void referencingObjectRemoved();

};

#endif // DCVCOMPONENT_H
