﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Diagnostics;
using ClipClop.User;
using ClipClop.Model;
using CaLib.User;
using System.IO;

namespace ClipClop.View
{
	public partial class MainForm : Form
	{
		/// <summary>
		/// 履歴に残すかどうかフラグ
		/// </summary>
		private bool valid_;

		/// <summary>
		/// クリップボード監視クラス
		/// </summary>
		private CaLib.User.ClipboardViewer cpViewer_;

		/// <summary>
		/// ホットキーの管理
		/// </summary>
		private CaLib.User.Hotkey hotkeyRegister_;

		/// <summary>
		/// 動作設定
		/// </summary>
		private ClipClop.Model.AplSetting setting_;

		/// <summary>
		/// 定型文字列ペースト用メニュー
		/// </summary>
		private DynamicContextMenuStrip contextMenuStripPopup_;


		public MainForm()
		{
			valid_ = true;
			cpViewer_ = new CaLib.User.ClipboardViewer(this);
			cpViewer_.ClipboardHandler += this.OnClipBoardChanged;

			InitializeComponent();

			this.ShowInTaskbar = false;

			//設定の復元を行う
			setting_ = ClipClop.Model.AplSetting.Create(global::ClipClop.Properties.Settings.Default.AplSetting);

			contextMenuStripPopup_ = new DynamicContextMenuStrip(this.components);

			contextMenuStripPopup_.AplSetting = this.setting_;

			contextMenuStripPopup_.Closed += new ToolStripDropDownClosedEventHandler(contextMenuStripPopup_Closed);
		}

		#region 仮想関数

		protected override void OnLoad(EventArgs e)
		{
			base.OnLoad(e);

			//ホットキーを設定する
			if (!SetupHotkey())
			{
				Util.ShowError(global::ClipClop.Properties.Resources.EF002);
			}

			//設定を反映する
			AdaptSetting();		

			listBoxHistory.Clear();

			if (!File.Exists(this.setting_.templateFilePath_) && !global::ClipClop.Properties.Settings.Default.CopiedInitialTemplate)
			{ 
				//定型文定義ファイルが存在しない場合で、まだ初期ファイルをコピーしていないならば
				string src = Path.Combine( Path.GetDirectoryName(Application.ExecutablePath), "InitialTemplate.xml");
				try
				{
					File.Copy(src, this.setting_.templateFilePath_);
					global::ClipClop.Properties.Settings.Default.CopiedInitialTemplate = true;
				}
				catch (Exception exp1)
				{
					Trace.WriteLine( "初期定型文定義ファイルコピーで例外：" + exp1.Message );
				}
			}

			try
			{
				contextMenuStripPopup_.Read(this.setting_.templateFilePath_, ContextMenuSettingManager.FileType.TreeXml);

				if (this.setting_.bInheritHistory_)
				{
					//履歴の読み込み
					this.listBoxHistory.Read(Path.Combine(AplSetting.APP_DATA_FOLDER_, global::ClipClop.Properties.Settings.Default.HistoryFileName));

				}
			}
			catch (Exception exp)
			{
				Util.ShowError(exp.Message);
			}

			HistoryToolStripMenuItem.Checked = this.valid_;


			if (setting_.bMinimizeOnStart_)
			{
				//起動時に最小化する。
				this.WindowState = FormWindowState.Minimized;
				this.Hide();
				this.Visible = false;
			}

		}

		void AdaptSetting()
		{
			this.TopMost = setting_.bAlwaysTop_;
			this.Opacity = setting_.Opacity_;

			this.listBoxHistory.Font = setting_.WindowFont_;

			//履歴最大数設定
			listBoxHistory.SetMax(setting_.maxHistory_);
		}

		/// <summary>
		/// ホットキーを設定する
		/// </summary>
		bool SetupHotkey()
		{
			//ホットキー登録解除
			if (hotkeyRegister_ != null)
			{
				hotkeyRegister_.Clear();
			}

			bool ret = true;
			hotkeyRegister_ = new CaLib.User.Hotkey(this.Handle);

			for (int i = 0; i < setting_.GetHotKeyCount(); i++)
			{
				System.Windows.Forms.Keys s = setting_.GetHotKeyAt(i);
				Debug.WriteLine(string.Format("SetupHotkey[{0}] {1}", i, s.ToString()));

				if (s == Keys.None)
					continue;

				CaLib.User.Hotkey.Input ipt = null;
				
				switch( (AplSetting.Hotkeys)i)
				{
					case AplSetting.Hotkeys.Active:
						ipt = new Hotkey.Input(s, this.HotkeyEventHandler_Active);
						break;
					case AplSetting.Hotkeys.DeleteHistory:
						ipt = new Hotkey.Input(s, this.HotkeyEventHandler_DeleteHistory);
						break;
					case AplSetting.Hotkeys.TemplatePopup:
						ipt = new Hotkey.Input(s, this.HotkeyEventHandler_TemplatePopup);
						break;
				}
				ret &= hotkeyRegister_.Register(i, ipt);
			}

			return ret;
		}

		#region ホットキーイベント

		void HotkeyEventHandler_Active()
		{
			this.Open();
		}
		
		void HotkeyEventHandler_DeleteHistory()
		{
			this.listBoxHistory.RemoveLast();
		}

		void HotkeyEventHandler_TemplatePopup()
		{
			Trace.WriteLine(string.Format("WindowState={0}, Visible={1}", this.WindowState.ToString(), this.Visible));

			//マウスカーソルの位置を画面座標で取得
			Point mp = Control.MousePosition;

			//設定ファイル読み込み
			bool readNewSetting = contextMenuStripPopup_.Read(this.setting_.templateFilePath_, ContextMenuSettingManager.FileType.TreeXml);
			if (readNewSetting || contextMenuStripPopup_.Items.Count < 1)
			{
				contextMenuStripPopup_.ConstructMenu();
			}

			//TODO ○メニューをスマートに閉じる方法の検討
			// メニューからフォーカスが外れたら閉じるために、メインフォームを表示する。
			contextMenuStripPopup_.ParentVisible = this.Visible;
			this.Open();

			contextMenuStripPopup_.Show(mp);
		}

		#endregion

		/// <summary>
		/// 定型文メニューが閉じたときに呼ばれます。
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		public void contextMenuStripPopup_Closed(object sender, ToolStripDropDownClosedEventArgs e)
		{
			if( false == contextMenuStripPopup_.ParentVisible )
				base.Hide();
		}


		protected override void WndProc(ref Message m)
		{
			if ((m.Msg == (int)WinApi.Message.id.WM_SYSCOMMAND) && (m.WParam == (IntPtr)WinApi.Message.wparam.SC_MINIMIZE))
			{
				//最小化ボタンクリックで、FormをHideするだけにする
				base.Hide(); // 隠す 
				return;
			}

			if (hotkeyRegister_ != null )
			{
				hotkeyRegister_.OnKeyDown(m);
			}

			base.WndProc(ref m);
		}

		#endregion

		#region イベントハンドラー

		/// <summary>
		/// ダブルクリック
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void notifyIcon_MouseDoubleClick(object sender, MouseEventArgs e)
		{
			Open();
		}

		/// <summary>
		/// 終了メニュー
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void ExitToolStripMenuItem_Click(object sender, EventArgs e)
		{
			// タスクトレイからアイコンを取り除く
			this.notifyIcon.Visible = false;

			// アプリケーション終了
			Application.Exit();
		}

		private void MainForm_FormClosing(object sender, FormClosingEventArgs e)
		{
			if (e.CloseReason == CloseReason.UserClosing)
			{
				e.Cancel = true;						// 終了処理キャンセル
				this.Visible = false;					// フォーム非表示
				this.notifyIcon.Visible = true;			// Notifyアイコン表示
				//this.notifyIcon.ShowBalloonTip(500); // バルーンTip表示
				return;
			}

			//ホットキー登録解除
			hotkeyRegister_.Clear();

			//設定保存
			Properties.Settings.Default.Save();

			if (this.setting_.bInheritHistory_)
			{
				//履歴保存
				this.listBoxHistory.Save(Path.Combine(AplSetting.APP_DATA_FOLDER_,global::ClipClop.Properties.Settings.Default.HistoryFileName));
			}
		}

		/// <summary>
		/// 開くメニュー
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void OpenToolStripMenuItem_Click(object sender, EventArgs e)
		{
			Open();
		}

		/// <summary>
		/// クリップボードにテキストがコピーされると呼び出される
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="args"></param>
		private void OnClipBoardChanged(object sender, CaLib.User.ClipboardEventArgs args)
		{
			if (valid_ && false == string.IsNullOrEmpty(args.Text))
			{
				listBoxHistory.InsertHead(args.Text);
			}
		}


		/// <summary>
		/// 履歴ON/OFFメニューがクリックされた
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void HistoryToolStripMenuItem_Click(object sender, EventArgs e)
		{
			System.Windows.Forms.ToolStripMenuItem m = sender as System.Windows.Forms.ToolStripMenuItem;
			valid_ = m.Checked;
		}

		private void SettingToolStripMenuItem_Click(object sender, EventArgs e)
		{
			//アプリケーション設定画面を表示する。
			using (SettingForm sf = new SettingForm(this.setting_))
			{
				DialogResult result = sf.ShowDialog();
				if (result == DialogResult.OK)
				{
					this.setting_ = sf.Setting;
					global::ClipClop.Properties.Settings.Default.AplSetting = setting_.ToString();

					//ホットキーを設定する
					if (!SetupHotkey())
					{
						Util.ShowError(global::ClipClop.Properties.Resources.EF002);
					}

					//設定を反映する
					AdaptSetting();
				}
			}
		}

        /// <summary>
        /// 定型文設定メニュー
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void templateToolStripMenuItem_Click(object sender, EventArgs e)
        {
			ContextMenuSettingManager mngr = contextMenuStripPopup_.Manager;

			using (TemplateSetting dlg = new TemplateSetting(mngr, this.setting_.templateFilePath_))
			{
				DialogResult result = dlg.ShowDialog();
				if (result == DialogResult.OK)
				{
					mngr.Document = dlg.Document;

                    try
                    {
                        mngr.Write(this.setting_.templateFilePath_);
                    }
                    catch (Exception exp)
                    {
						Util.ShowError(global::ClipClop.Properties.Resources.EF004 + Environment.NewLine + exp.Message);
                    }
				}
			}
        }


		private void 先頭に移動ToolStripMenuItem_Click(object sender, EventArgs e)
		{
			listBoxHistory.MoveHead(false);
		}

		private void 削除ToolStripMenuItem_Click(object sender, EventArgs e)
		{
			listBoxHistory.DeleteSelectedItem();
		}

		private void 改行を削除して先頭へToolStripMenuItem_Click(object sender, EventArgs e)
		{
			listBoxHistory.MoveHead(true);
		}

		/// <summary>
		/// リストボックスのアイテム選択が変わったときに呼ばれます。
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void listBoxHistory_SelectedIndexChanged(object sender, EventArgs e)
		{
			//操作メニューの有効無効更新
			if (listBoxHistory.SelectedIndex < 0)
				SetListContextMenuStripEnabled(false);
			else
				SetListContextMenuStripEnabled(true);
		}

		private void aboutAToolStripMenuItem_Click(object sender, EventArgs e)
		{
			using (AboutForm f = new AboutForm())
			{
				f.ShowDialog();
			}
		}

		private void helpGToolStripMenuItem_Click(object sender, EventArgs e)
		{
			try
			{
				System.Diagnostics.Process.Start(ClipClop.Properties.Resources.ProjectPage);
			}
			catch (Exception exp)
			{
				Util.ShowError(exp.Message);
			}
		}

		#endregion


		void Open()
		{
			this.Visible = true;
			if (this.WindowState == FormWindowState.Minimized)
			{
				// 最小化をやめる
				this.WindowState = FormWindowState.Normal;
			}
			this.Activate();
		}

		private void SetListContextMenuStripEnabled(bool enable)
		{
			foreach (ToolStripItem item in listContextMenuStrip.Items)
			{
				item.Enabled = enable;
			}
			listContextMenuStrip.Enabled = enable;
		}

		private void 全て削除toolStripMenuItem_Click(object sender, EventArgs e)
		{
			listBoxHistory.Clear();
			SetListContextMenuStripEnabled(false);
		}

	}
}
