﻿/*
 * DrFx - FxCop Report Translator and Visualizer.
 * Copyright (C) 2010 Sasa Yuan
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */



using NUnit.Framework;
using Sasa.QualityTools.DrFx.Console;
using Sasa.QualityTools.DrFx.Console.Tests.Integration.Helpers;
using Sasa.QualityTools.DrFx.Core;
using System;
using System.Collections.Generic;
using System.IO;

namespace Sasa.QualityTools.DrFx.Console.Tests.Integration
{
    [TestFixture]
    public class ProgramTest
    {
        /// <summary>
        /// 作業ディレクトリにテスト用のファイル、ディレクトリを作成します。
        /// </summary>
        [TestFixtureSetUp]
        public void SetUpFixture()
        {
            SetUpHelper.Create("pgmt/");
            SetUpHelper.Create("pgmt/asm.dll");
            SetUpHelper.Create("pgmt/dir1/");
            SetUpHelper.Create("pgmt/dir1/asm.dll");
            SetUpHelper.Create("pgmt/dir2/");
            SetUpHelper.Create("pgmt/dir2/asm.dll");
            SetUpHelper.Create("pgmt/dir3/");
            SetUpHelper.Create("pgmt/dir3/asm.dll");
        }

        /// <summary>
        /// SetUpFixture メソッドが作成したテスト用ファイル、ディレクトリを削除します。
        /// </summary>
        [TestFixtureTearDown]
        public void TearDownFixture()
        {
            Directory.Delete("pgmt", true);
        }

        /// <summary>
        /// --dependency オプションが正しく解析され、構成値に反映されることをテストします。
        /// </summary>
        /// <param name="argument">コマンドライン引数。</param>
        /// <returns>依存先ディレクトリのコレクション。</returns>
        [Test]
        [TestCaseSource(typeof(ConfigureWithDependencyOptionTestCaseGenerator), "Generate")]
        public IEnumerable<string> TestConfigureWithDependencyOption(string argument)
        {
            FxCopProcessConfiguration configuration = new FxCopProcessConfiguration();
            Program.ConfigureWithCommandLineArguments(configuration, argument.Split(' '));
            return configuration.DependencyDirectories;
        }

        /// <summary>
        /// --output オプションが正しく解析され、構成値に反映されることをテストします。
        /// </summary>
        /// <param name="argument">コマンドライン引数。</param>
        /// <returns>出力先ファイル。</returns>
        [Test]
        [TestCaseSource(typeof(ConfigureWithOutputOptionTestCaseGenerator), "Generate")]
        public string TestConfigureWithOutputOption(string argument)
        {
            FxCopProcessConfiguration configuration = new FxCopProcessConfiguration();
            Program.ConfigureWithCommandLineArguments(configuration, argument.Split(' '));
            return configuration.OutputFile;
        }

        /// <summary>
        /// --rule オプションが正しく解析され、構成値に反映されることをテストします。
        /// </summary>
        /// <param name="argument">コマンドライン引数。</param>
        /// <returns>ルールファイルのコレクション。</returns>
        [Test]
        [TestCaseSource(typeof(ConfigureWithRuleOptionTestCaseGenerator), "Generate")]
        public IEnumerable<string> TestConfigureWithRuleOption(string argument)
        {
            FxCopProcessConfiguration configuration = new FxCopProcessConfiguration();
            Program.ConfigureWithCommandLineArguments(configuration, argument.Split(' '));
            return configuration.RuleFiles;
        }

        /// <summary>
        /// --target オプションが正しく解析され、構成値に反映されることをテストします。
        /// </summary>
        /// <param name="argument">コマンドライン引数。</param>
        /// <returns>ルールファイルのコレクション。</returns>
        [Test]
        [TestCaseSource(typeof(ConfigureWithTargetOptionTestCaseGenerator), "Generate")]
        public IEnumerable<string> TestConfigureWithTargetOption(string argument)
        {
            FxCopProcessConfiguration configuration = new FxCopProcessConfiguration();
            Program.ConfigureWithCommandLineArguments(configuration, argument.Split(' '));
            return configuration.TargetFiles;
        }

        /// <summary>
        /// コマンドライン引数に未定義のオプションが含まれている場合は、例外が発生することをテストします。
        /// </summary>
        [Test]
        [ExpectedException(typeof(InvalidOperationException))]
        [Description("コマンドライン引数に未定義のオプションが含まれている場合は、例外が発生することをテストします。")]
        public void TestConfigureWithUndefinedOption()
        {
            FxCopProcessConfiguration configuration = new FxCopProcessConfiguration();
            Program.ConfigureWithCommandLineArguments(configuration, "--undefined".Split(' '));
        }

        /// <summary>
        /// --dependency オプション解析テストのテストケースデータ作成クラスです。
        /// </summary>
        private class ConfigureWithDependencyOptionTestCaseGenerator
        {
            /// <summary>
            /// テストケースデータのコレクションを生成します。
            /// </summary>
            /// <returns>生成したテストケースデータのコレクション。</returns>
            internal IEnumerable<TestCaseData> Generate()
            {
                yield return new TestCaseData(@"")
                    .Returns(new string[] { })
                    .SetDescription("--dependency オプションが指定されていない場合は、依存先ディレクトリコレクションが空であることをテストします。");

                yield return new TestCaseData(@"--dependency pgmt")
                    .Returns(new string[] { @"pgmt" })
                    .SetDescription("--dependency にディレクトリが指定された場合は、そのディレクトリのみが依存先ディレクトリに含まれることをテストします。");

                yield return new TestCaseData(@"--dependency pgmt\*")
                    .Returns(new string[] { @"pgmt\dir1", @"pgmt\dir2", @"pgmt\dir3" })
                    .SetDescription("--dependency にワイルドカードが指定された場合は、パターンにマッチする全ディレクトリが依存先ディレクトリに含まれることをテストします。");

                yield return new TestCaseData(@"--dependency pgmt\dir2 --dependency pgmt\dir3")
                    .Returns(new string[] { @"pgmt\dir2", @"pgmt\dir3" })
                    .SetDescription("--dependency が複数回指定された場合は、指定された各ディレクトリが依存先ディレクトリに含まれることをテストします。");

                yield return new TestCaseData(@"-d pgmt")
                    .Returns(new string[] { @"pgmt" })
                    .SetDescription("-d にディレクトリが指定された場合は、そのディレクトリのみが依存先ディレクトリに含まれることをテストします。");

                yield return new TestCaseData(@"-d pgmt\*")
                    .Returns(new string[] { @"pgmt\dir1", @"pgmt\dir2", @"pgmt\dir3" })
                    .SetDescription("-d にワイルドカードが指定された場合は、パターンにマッチする全ディレクトリが依存先ディレクトリに含まれることをテストします。");

                yield return new TestCaseData(@"-d pgmt\dir2 -d pgmt\dir3")
                    .Returns(new string[] { @"pgmt\dir2", @"pgmt\dir3" })
                    .SetDescription("-d が複数回指定された場合は、指定された各ディレクトリが依存先ディレクトリに含まれることをテストします。");
            }
        }

        /// <summary>
        /// --output オプション解析テストのテストケースデータ作成クラスです。
        /// </summary>
        private class ConfigureWithOutputOptionTestCaseGenerator
        {
            /// <summary>
            /// テストケースデータのコレクションを生成します。
            /// </summary>
            /// <returns>生成したテストケースデータのコレクション。</returns>
            internal IEnumerable<TestCaseData> Generate()
            {
                yield return new TestCaseData(@"")
                    .Returns(@"codeanalysisreport.xml")
                    .SetDescription("--output オプションが指定されていない場合は、デフォルトの出力先が設定されていることをテストします。");

                yield return new TestCaseData(@"--output report.xml")
                    .Returns(@"report.xml")
                    .SetDescription("--output オプションが指定された場合は、オプション引数で指定されたファイルが出力先として設定されることをテストします。");

                yield return new TestCaseData(@"--output report.xml --output out\report.xml")
                    .Returns(@"out\report.xml")
                    .SetDescription("--output オプションが複数回指定された場合は、最後に指定されたオプションが有効であることをテストします。");

                yield return new TestCaseData(@"-o report.xml")
                    .Returns(@"report.xml")
                    .SetDescription("-o オプションが指定された場合は、オプション引数で指定されたファイルが出力先として設定されることをテストします。");

                yield return new TestCaseData(@"-o report.xml -o out\report.xml")
                    .Returns(@"out\report.xml")
                    .SetDescription("-o オプションが複数回指定された場合は、最後に指定されたオプションが有効であることをテストします。");
            }
        }

        /// <summary>
        /// --rule オプション解析テストのテストケースデータ作成クラスです。
        /// </summary>
        private class ConfigureWithRuleOptionTestCaseGenerator
        {
            /// <summary>
            /// テストケースデータのコレクションを生成します。
            /// </summary>
            /// <returns>生成したテストケースデータのコレクション。</returns>
            internal IEnumerable<TestCaseData> Generate()
            {
                yield return new TestCaseData(@"")
                    .Returns(new string[] { })
                    .SetDescription("--rule オプションが指定されていない場合は、ルールファイルコレクションが空であることをテストします。");

                yield return new TestCaseData(@"--rule pgmt\asm.dll")
                    .Returns(new string[] { @"pgmt\asm.dll" })
                    .SetDescription("--rule オプションにファイルが指定された場合は、ルールファイルコレクションに指定されたファイルが含まれることをテストします。");

                yield return new TestCaseData(@"--rule pgmt\*\*.dll")
                    .Returns(new string[] { @"pgmt\dir1\asm.dll", @"pgmt\dir2\asm.dll", @"pgmt\dir3\asm.dll" })
                    .SetDescription("--rule オプションにワイルドカードが指定された場合は、パターンにマッチする全ファイルがルールファイルコレクションに含まれることをテストします。");

                yield return new TestCaseData(@"--rule pgmt\asm.dll --rule pgmt\dir1\asm.dll")
                    .Returns(new string[] { @"pgmt\asm.dll", @"pgmt\dir1\asm.dll" })
                    .SetDescription("--rule オプションが複数回指定された場合は、指定された各ファイルがルールファイルコレクションに含まれることをテストします。");

                yield return new TestCaseData(@"-r pgmt\asm.dll")
                    .Returns(new string[] { @"pgmt\asm.dll" })
                    .SetDescription("-r オプションにファイルが指定された場合は、ルールファイルコレクションに指定されたファイルが含まれることをテストします。");

                yield return new TestCaseData(@"-r pgmt\*\*.dll")
                    .Returns(new string[] { @"pgmt\dir1\asm.dll", @"pgmt\dir2\asm.dll", @"pgmt\dir3\asm.dll" })
                    .SetDescription("-r オプションにワイルドカードが指定された場合は、パターンにマッチする全ファイルがルールファイルコレクションに含まれることをテストします。");

                yield return new TestCaseData(@"-r pgmt\asm.dll -r pgmt\dir1\asm.dll")
                    .Returns(new string[] { @"pgmt\asm.dll", @"pgmt\dir1\asm.dll" })
                    .SetDescription("-r オプションが複数回指定された場合は、指定された各ファイルがルールファイルコレクションに含まれることをテストします。");
            }
        }

        /// <summary>
        /// --target オプション解析テストのテストケースデータ作成クラスです。
        /// </summary>
        private class ConfigureWithTargetOptionTestCaseGenerator
        {
            /// <summary>
            /// テストケースデータのコレクションを生成します。
            /// </summary>
            /// <returns>生成したテストケースデータのコレクション。</returns>
            internal IEnumerable<TestCaseData> Generate()
            {
                yield return new TestCaseData(@"")
                    .Returns(new string[] { })
                    .SetDescription("--target オプションが指定されていない場合は、ターゲットファイルコレクションが空であることをテストします。");

                yield return new TestCaseData(@"--target pgmt\asm.dll")
                    .Returns(new string[] { @"pgmt\asm.dll" })
                    .SetDescription("--target オプションにファイルが指定された場合は、ターゲットファイルコレクションに指定されたファイルが含まれることをテストします。");

                yield return new TestCaseData(@"--target pgmt\*\*.dll")
                    .Returns(new string[] { @"pgmt\dir1\asm.dll", @"pgmt\dir2\asm.dll", @"pgmt\dir3\asm.dll" })
                    .SetDescription("--target オプションにワイルドカードが指定された場合は、パターンにマッチする全ファイルがターゲットファイルコレクションに含まれることをテストします。");

                yield return new TestCaseData(@"--target pgmt\asm.dll --target pgmt\dir1\asm.dll")
                    .Returns(new string[] { @"pgmt\asm.dll", @"pgmt\dir1\asm.dll" })
                    .SetDescription("--target オプションが複数回指定された場合は、指定された各ファイルがターゲットファイルコレクションに含まれることをテストします。");

                yield return new TestCaseData(@"-t pgmt\asm.dll")
                    .Returns(new string[] { @"pgmt\asm.dll" })
                    .SetDescription("-t オプションにファイルが指定された場合は、ターゲットファイルコレクションに指定されたファイルが含まれることをテストします。");

                yield return new TestCaseData(@"-t pgmt\*\*.dll")
                    .Returns(new string[] { @"pgmt\dir1\asm.dll", @"pgmt\dir2\asm.dll", @"pgmt\dir3\asm.dll" })
                    .SetDescription("-t オプションにワイルドカードが指定された場合は、パターンにマッチする全ファイルがターゲットファイルコレクションに含まれることをテストします。");

                yield return new TestCaseData(@"-t pgmt\asm.dll -t pgmt\dir1\asm.dll")
                    .Returns(new string[] { @"pgmt\asm.dll", @"pgmt\dir1\asm.dll" })
                    .SetDescription("-t オプションが複数回指定された場合は、指定された各ファイルがターゲットファイルコレクションに含まれることをテストします。");
            }
        }
    }
}
