﻿/*
 * DrFx - FxCop Report Translator and Visualizer.
 * Copyright (C) 2010 Sasa Yuan
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */



using NUnit.Framework;
using Sasa.QualityTools.DrFx.Console;
using Sasa.QualityTools.DrFx.Core;
using System;
using System.Collections.Generic;
using System.Configuration;
using System.IO;

namespace Sasa.QualityTools.DrFx.Console.Tests.Unit
{
    [TestFixture]
    public class ProgramTest
    {
        /// <summary>
        /// AppSettings で FxCop のインストールディレクトリが明示されていない場合は %PROGRAMFILES%\\Microsoft FxCop 1.36 になることをテストします。
        /// </summary>
        [Test]
        [Description("AppSettings で FxCop のインストールディレクトリが明示されていない場合は %PROGRAMFILES%\\Microsoft FxCop 1.36 になることをテストします。")]
        public void TestDefaultInstallDirectory()
        {
            FxCopProcessConfiguration configuration = Program.CreateConfiguration();
            Assert.AreEqual(Path.Combine(Environment.GetEnvironmentVariable("PROGRAMFILES"), "Microsoft FxCop 1.36"), configuration.FxCopInstallDirectory);
        }

        /// <summary>
        /// AppSettings で FxCop のインストールディレクトリが明示されている場合はそちらを使用することをテストします。
        /// </summary>
        [Test]
        [Description("AppSettings で FxCop のインストールディレクトリが明示されている場合はそちらを使用することをテストします。")]
        public void TestAppSettingsInstallDirectory()
        {
            string original = ConfigurationManager.AppSettings["FxCopInstallDirectory"];
            try
            {
                ConfigurationManager.AppSettings["FxCopInstallDirectory"] = @"C:\tmp";
                FxCopProcessConfiguration configuration = Program.CreateConfiguration();
                Assert.AreEqual(@"C:\tmp", configuration.FxCopInstallDirectory);
            }
            finally
            {
                ConfigurationManager.AppSettings["FxCopInstallDirectory"] = original;
            }
        }

        [TestCaseSource(typeof(ParseMessageTestCaseGenerator), "Generate")]
        public Warning TestParseMessage(string message)
        {
            return Program.ParseMessage(message);
        }

        /// <summary>
        /// 警告メッセージフォーマットにマッチしないメッセージを解析すると null になることをテストします。
        /// </summary>
        [Test]
        public void TestParseNonWarningMessage()
        {
            Assert.IsNull(Program.ParseMessage("Loading Assembly.dll..."));
        }

        private class ParseMessageTestCaseGenerator
        {
            internal IEnumerable<TestCaseData> Generate()
            {
                yield return new TestCaseData(@"[Location not stored in Pdb] : warning  : CA0000 : Microsoft.XXX : ISSUE.")
                    .Returns(new Warning { SourceContext = "[ソース情報なし]", Level = "警告", Category = "Microsoft.XXX", CheckId = "CA0000", Issue = "ISSUE." })
                    .SetDescription("ソース情報を持たない警告メッセージが正しく解析されることをテストします。");

                yield return new TestCaseData(@"C:\tmp\src.cs(64,1) : error  : CA0000 : Microsoft.XXX : ISSUE.")
                    .Returns(new Warning { SourceContext = @"C:\tmp\src.cs(64,1)", Level = "エラー", Category = "Microsoft.XXX", CheckId = "CA0000", Issue = "ISSUE." })
                    .SetDescription("ソース情報を持つ警告メッセージが正しく解析されることをテストします。");
            }
        }
    }
}
