﻿/*
 * DrFx - FxCop Report Translator and Visualizer.
 * Copyright (C) 2010 Sasa Yuan
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */



using NUnit.Framework;
using Sasa.QualityTools.DrFx.Explorer.Report;
using System;
using System.Collections.Generic;
using System.IO;

namespace Sasa.QualityTools.DrFx.Explorer.Tests.Integration.Report
{
    /// <summary>
    /// <see cref="Sasa.QualityTools.DrFx.Explorer.Report.FxCopReport"/> クラスのテストクラスです。
    /// </summary>
    [TestFixture]
    public class FxCopReportTest
    {
        /// <summary>
        /// FxCop レポートロードメソッドに null を渡した場合、<see cref="System.ArgumentException"/> がスローされることをテストします。
        /// </summary>
        [Test]
        [Description("FxCop レポートロードメソッドに null を渡した場合、ArgumentException がスローされることをテストします。")]
        [ExpectedException(typeof(ArgumentException))]
        public void TestLoadWithNull()
        {
            FxCopReport.Load(null);
        }

        /// <summary>
        /// FxCop レポートロードメソッドにファイルシステム上に存在しないパスを渡した場合、<see cref="System.ArgumentException"/> がスローされることをテストします。
        /// </summary>
        [Test]
        [Description("FxCop レポートロードメソッドにファイルシステム上に存在しないパスを渡した場合、ArgumentException がスローされることをテストします。")]
        [ExpectedException(typeof(ArgumentException))]
        public void TestLoadWithNotExistFile()
        {
            FxCopReport.Load(@"TestData\Report\FxCopReportTest\notexist.xml");
        }

        /// <summary>
        /// FxCop レポートロードメソッドに FxCop レポートファイルのパスを渡した場合に、
        /// ファイルの内容を反映した <see cref="Sasa.QualityTools.DrFx.Explorer.Report.FxCopReport"/> オブジェクトが作成されることをテストします。
        /// </summary>
        [Test]
        [Description("FxCop レポートロードメソッドに FxCop レポートファイルのパスを渡した場合に、ファイルの内容を反映した FxCopReport オブジェクトが作成されることをテストします。")]
        public void TestLoad()
        {
            FxCopReport report = FxCopReport.Load(@"TestData\Report\FxCopReportTest\fxcopreport.xml");

            string reportPath = Path.Combine(Environment.CurrentDirectory, @"TestData\Report\FxCopReportTest\fxcopreport.xml");
            Assert.AreEqual(reportPath, report.ReportFile);

            // NOTE: XML の各属性の読み込みは単体テストで確認しているので、読み込んだ要素数だけテストする。
            {
                var assemblies = new List<AssemblyWarning>(report.Assemblies);
                Assert.AreEqual(1, assemblies.Count);
                Assert.AreEqual(1, assemblies[0].Issues.Count);

                var namespaces = new List<NamespaceWarning>(assemblies[0].Namespaces);
                Assert.AreEqual(1, namespaces.Count);
                Assert.AreEqual(0, namespaces[0].Issues.Count);

                var types = new List<TypeWarning>(namespaces[0].Types);
                Assert.AreEqual(3, types.Count);
                Assert.AreEqual(1, types[0].Issues.Count);

                var members = new List<MemberWarning>(types[0].Members);
                Assert.AreEqual(2, members.Count);
                Assert.AreEqual(2, new List<FxCopIssue>(members[0].Issues).Count);
                Assert.AreEqual(1, new List<FxCopIssue>(members[1].Issues).Count);

                Assert.AreEqual(1, types[1].Issues.Count);
                Assert.AreEqual(0, new List<MemberWarning>(types[1].Members).Count);

                Assert.AreEqual(3, types[2].Issues.Count);
                Assert.AreEqual(0, new List<MemberWarning>(types[2].Members).Count);
            }

            {
                var namespaces = new List<NamespaceWarning>(report.Namespaces);
                Assert.AreEqual(2, namespaces.Count);
                Assert.AreEqual(3, namespaces[0].Issues.Count);
                Assert.AreEqual(3, namespaces[1].Issues.Count);
            }

            {
                Assert.AreEqual(7, report.Rules.Count);
            }
        }
    }
}
