﻿//  Copyright (c) 2010-2012, www.dvgtu.org
//  All rights reserved.

//  Redistribution and use in source and binary forms, with or without modification, 
//  are permitted provided that the following conditions are met:

//  * Redistributions of source code must retain the above copyright notice, 
//    this list of conditions and the following disclaimer. 
//  * Redistributions in binary form must reproduce the above copyright notice, 
//    this list of conditions and the following disclaimer in the documentation 
//    and/or other materials provided with the distribution. 

//  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
//  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
//  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
//  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
//  INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT 
//  NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR 
//  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
//  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
//  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
//  POSSIBILITY OF SUCH DAMAGE.
// $Id: ChatForm.cs 127 2011-11-08 16:23:05Z takeshich $

using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;
using System.Collections.Generic;
using System.Text.RegularExpressions;
using System.Threading;
using WeifenLuo.WinFormsUI.Docking;
using OpenMetaverse;
using dvgtuViewer.SLNetworkComm;

namespace dvgtuViewer
{
	/// <summary>
	/// Description of ChatForm.
	/// </summary>
	public partial class ChatForm : WeifenLuo.WinFormsUI.Docking.DockContent
	{
		private DvgtuInstance dvgtuInstance;
		private GridClient Client{ get { return dvgtuInstance.Client; } }
		private SLNetCom NetCom{ get { return dvgtuInstance.Netcom; } }
		
		private RichTextBox[] txtIMView; 
		private TabPage[] tabPages;
		private int TabCount;
		
		private Dictionary<UUID,int> tab2ID = new Dictionary<UUID,int>();
		private Dictionary<UUID, UUID> usessionID = new Dictionary<UUID, UUID>();
		
		private ManualResetEvent NameSearchEvent = new ManualResetEvent(false);
		
		
		
		public ChatForm(DvgtuInstance dvgtuInstance)
		{
			//
			// The InitializeComponent() call is required for Windows Forms designer support.
			//
			InitializeComponent();
			
			//
			// TODO: Add constructor code after the InitializeComponent() call.
			//
			this.dvgtuInstance = dvgtuInstance;
			this.Load += new EventHandler(ChatForm_Load);

		}

		void ChatForm_Load(object sender, EventArgs e)
		{
			NetCom.NetcomSync = this;
			
//			//設定の読み込み
//			Settings.LoadFromXmlFile();
//			ChatFont = Settings.Instance.ChatFont;
//			
//			FontConverter fx = new FontConverter();
//			Regex r = new Regex(".*[0-9.]*");
//			Match m = r.Match(ChatFont);
//			ChatFont = ChatFont.Replace(m.ToString(), "9");
//			
//			this.txtView.Font = (Font)fx.ConvertFromString(ChatFont);
			
			this.AcceptButton = this.BtnChat;
			
			//initail value
			this.cmbChat.SelectedIndex = 0;
			this.txtIMView = new RichTextBox[16];
			this.tabPages = new TabPage[16];
			
			//イベントハンドラ
			this.txtView.LinkClicked += new LinkClickedEventHandler(ChatForm_LinkClicked);
			
			NetCom.ClientLoginStatus += new EventHandler<LoginProgressEventArgs>(NetCom_ClientLoginStatus);
			NetCom.ChatReceived += new EventHandler<OpenMetaverse.ChatEventArgs>(NetCom_ChatReceived);
			NetCom.InstantMessageReceived += new EventHandler<OpenMetaverse.InstantMessageEventArgs>(NetCom_InstantMessageReceived);
			NetCom.TeleportStatusChanged += new EventHandler<TeleportEventArgs>(NetCom_TeleportStatusChanged);
			NetCom.AlertMessageReceived += new EventHandler<OpenMetaverse.AlertMessageEventArgs>(NetCom_AlertMessageReceived);
			NetCom.FriendOnline += new EventHandler<FriendInfoEventArgs>(NetCom_FriendOnline);
			NetCom.FriendOffline += new EventHandler<FriendInfoEventArgs>(NetCom_FriendOffline);
			NetCom.FriendshipResponse += new EventHandler<OpenMetaverse.FriendshipResponseEventArgs>(NetCom_FriendshipResponse);
			
			//Client.Avatars.UUIDNameReply += new EventHandler<UUIDNameReplyEventArgs>(Client_Avatars_UUIDNameReply);
			
			//this.OnIMPreparate += new EventHandler<IMPreparateEventArgs>(IMPreparate);
			this.tabtxtArea.TabCloseButtonClick += new EventHandler(ChatForm_TabCloseButtonClick);
		}

		void NetCom_FriendshipResponse(object sender, OpenMetaverse.FriendshipResponseEventArgs e)
		{
			string Message;
			if (e.Accepted){
				Message = "さんが友達申請を受け入れました。";
			}else{
				Message = "さんが友達申請を拒否った。";
			}
			
			txtView.Text += e.AgentName + Message + Environment.NewLine;
			txtView.SelectionStart = txtView.TextLength;
			txtView.ScrollToCaret();
		}

		void NetCom_FriendOffline(object sender, FriendInfoEventArgs e)
		{
			string ViewMessage = "さんがオフラインになった！";
			
			if (e.Friend.Name == null) return;
			
			txtView.Text += e.Friend.Name + ViewMessage + Environment.NewLine;
			txtView.SelectionStart = txtView.TextLength;
			txtView.ScrollToCaret();

		}

		void NetCom_FriendOnline(object sender, FriendInfoEventArgs e)
		{
			string ViewMessage = "さんがオンラインになった！";
			
			if (e.Friend.Name == null) return;
			
			txtView.Text += e.Friend.Name + ViewMessage + Environment.NewLine;
			txtView.SelectionStart = txtView.TextLength;
			txtView.ScrollToCaret();
		}

		void ChatForm_TabCloseButtonClick(object sender, EventArgs e)
		{
			if (tabtxtArea.SelectedIndex == 0) return;
			tab2ID.Remove((UUID)tabtxtArea.SelectedTab.Name);
			tabtxtArea.Controls.RemoveAt(tabtxtArea.SelectedIndex);
		}

		void ChatForm_LinkClicked(object sender, LinkClickedEventArgs e)
		{
			Process.Start(e.LinkText);
		}

//		public void SetClient(GridClient client){
//			Client = client;
//		}
//		
//		public void SetNetCom(SLNetCom netCom){
//			NetCom = netCom;
//		}

		public void IMPreparate(UUID AgentID,string AgentName)
		{
			tabGenerate(AgentID,AgentName);
		}
		
		void NetCom_AlertMessageReceived(object sender, OpenMetaverse.AlertMessageEventArgs e)
		{
			txtView.Text += "AlertMessageReceived:" + e.Message + Environment.NewLine;
			txtView.SelectionStart = txtView.TextLength;
			txtView.ScrollToCaret();
		}		
		
		private void NetCom_TeleportStatusChanged(object sender, TeleportEventArgs e)
		{
			switch (e.Status){
				case TeleportStatus.Start:
					txtView.Text += "テレポート開始" + Environment.NewLine;
					break;
				case TeleportStatus.Progress:
					txtView.Text += "テレポート中" + Environment.NewLine;
					break;
				case TeleportStatus.Finished:
					txtView.Text += "テレポート完了" + Environment.NewLine;
					break;
				case TeleportStatus.Failed:
					txtView.Text += "テレポート失敗" + Environment.NewLine;
					break;
				case TeleportStatus.None:
					break;
				case TeleportStatus.Cancelled:
					break;
				default:
					break;
			}
			txtView.SelectionStart = txtView.TextLength;
			txtView.ScrollToCaret();
		}
		
		private void NetCom_InstantMessageReceived(object sender, OpenMetaverse.InstantMessageEventArgs e)
		{
			try{
				//GIMについては現バージョンでは無視
				if (e.IM.GroupIM) return;
				
				//avatarじゃないUUIDからだったら、無視。
				
					
				//対象がない場合作成する
				tabGenerate(e.IM.FromAgentID,e.IM.FromAgentName);
				
				int pintTabCount;
//				if (!usessionID.ContainsKey(e.IM.FromAgentID))
//					usessionID.Add(e.IM.FromAgentID,e.IM.IMSessionID);
				
				
				
				
				switch (e.IM.Dialog){
					case InstantMessageDialog.MessageFromAgent:
						tab2ID.TryGetValue(e.IM.FromAgentID, out pintTabCount);
//						txtIMView[pintTabCount].Text += e.IM.FromAgentName + "<" + DateTime.Now.Hour + ":" +
//							DateTime.Now.Minute + "> " + e.IM.Message + Environment.NewLine;
//						txtIMView[pintTabCount].Text += e.IM.FromAgentName + "<" + 
//							e.IM.Timestamp.Year + "/" +e.IM.Timestamp.Month + "/" + e.IM.Timestamp.Date + " " + 
//							e.IM.Timestamp.Hour + ":" +e.IM.Timestamp.Minute + "> " + 
						txtIMView[pintTabCount].Text += "["+ DateTime.Now.ToShortTimeString() +"]" + "<" + e.IM.FromAgentName + "> " +
							": " + e.IM.Message + Environment.NewLine;
						
						txtIMView[pintTabCount].SelectionStart = txtIMView[pintTabCount].TextLength;
						txtIMView[pintTabCount].ScrollToCaret();
						Logger.Log("IMTimestamp:" + e.IM.Timestamp.ToString(),OpenMetaverse.Helpers.LogLevel.Debug);
						break;
					case InstantMessageDialog.ConsoleAndChatHistory:
//						txtIMView[pintTabCount].Text += e.IM.FromAgentName + "<" + 
//							e.IM.Timestamp.Year + "/" +e.IM.Timestamp.Month + "/" + e.IM.Timestamp.Date + " " + 
//							e.IM.Timestamp.Hour + ":" +e.IM.Timestamp.Minute + "> " + 
//							e.IM.Message + Environment.NewLine;
//						txtIMView[pintTabCount].SelectionStart = txtIMView[pintTabCount].TextLength;
//						txtIMView[pintTabCount].ScrollToCaret();
						break;
					case InstantMessageDialog.FriendshipAccepted:
						txtView.Text += "友達が受け入れられた"+ Environment.NewLine;
						txtView.ScrollToCaret();
						break;
					case InstantMessageDialog.FriendshipDeclined:
						txtView.Text += "友達が拒まれた"+ Environment.NewLine;
						txtView.ScrollToCaret();
						break;
					case InstantMessageDialog.FriendshipOffered:
						txtView.Text += "友達依頼が来た"+ Environment.NewLine;
						txtView.ScrollToCaret();
						break;
					case InstantMessageDialog.RequestTeleport:
						txtView.Text += "テレポートのリクエストが来た"+ Environment.NewLine;
						//TODO:
						//teleportのメッセージをMainFormにおくってあげて、メッセージボックスを出して
						//Client.Self.TeleportLureRespondをたたくように実装する必要がある。
						txtView.ScrollToCaret();
						break;
					default:
						break;
				}
				
			}catch(Exception exp){
				Logger.Log("ChatForm_NetCom_InstantMessageReceived:" + exp.Message,OpenMetaverse.Helpers.LogLevel.Error);
			}
		}
		
		private void NetCom_ChatReceived(object sender, OpenMetaverse.ChatEventArgs e)
		{
			string FromName = e.FromName;
			string ReceivedMessage = e.Message;
			string ViewMessage = string.Empty;
			int Disti = (int)Vector3.Distance(e.Position,Client.Self.SimPosition);
			string Dist = " (" + Disti.ToString() + "m)";
			
			try{
				if (string.IsNullOrEmpty(ReceivedMessage)) return;

				//表示する送られてきたチャットメッセージを整形する
				switch (e.Type){
					case ChatType.Normal:
						ViewMessage = "["+ DateTime.Now.ToShortTimeString() +"]" + "<" + FromName + ">" + Dist + ": " + ReceivedMessage;
						break;
					case ChatType.Shout:
						ViewMessage = "["+ DateTime.Now.ToShortTimeString() +"]" + "<" + FromName + ">" + Dist + " Shout: " + ReceivedMessage;
						break;
					case ChatType.Whisper:
						ViewMessage = "["+ DateTime.Now.ToShortTimeString() +"]" + "<" + FromName + ">" + Dist + " Whisper: " + ReceivedMessage;
						break;
					case ChatType.RegionSay:
						ViewMessage = "RegionSay: " + FromName + " " + ReceivedMessage;
						break;
					case ChatType.OwnerSay:
						ViewMessage = "OwnerSay: " + FromName + " " + ReceivedMessage;
						break;
					case ChatType.Debug:
						ViewMessage = "Debug: " + FromName + " " + ReceivedMessage;
						break;
					default:
						break;	
				}
				
				this.txtView.Text += ViewMessage + Environment.NewLine;
				txtView.SelectionStart = txtView.TextLength;
				txtView.ScrollToCaret();
				
			}catch(Exception exp){
				Logger.Log("ChatForm_NetCom_ChatReceived:" + exp.Message,OpenMetaverse.Helpers.LogLevel.Error);
			}
		}
		
		private void NetCom_ClientLoginStatus(object sender, LoginProgressEventArgs e)
		{
			try{
				switch(e.Status){
					case LoginStatus.Success:
						//offline時に受信したIMを調べる
						Client.Self.RetrieveInstantMessages();
						txtView.Text += Client.Network.LoginMessage + Environment.NewLine;
						Logger.Log(Client.Network.LoginMessage,OpenMetaverse.Helpers.LogLevel.Debug);
						//TODO:ログインが成功した場合のまとめ処理を作成する
//						mapForm.SimMapDraw();
//						mapForm.SimInfoDraw();

//						//TODO:LOginFormを消す処理
//						labelld.Text = "L$:" + Client.Self.Balance;
						break;
					case LoginStatus.ConnectingToLogin:
						txtView.Text += "Connecting To Login..." + Environment.NewLine;
						Logger.Log("Connecting To Login...",OpenMetaverse.Helpers.LogLevel.Debug);
						break;
					case LoginStatus.ConnectingToSim:
						txtView.Text += "Connecting To Sim..." + Environment.NewLine;
						Logger.Log("Connecting To Sim...",OpenMetaverse.Helpers.LogLevel.Debug);
						break;
					case LoginStatus.None:
						txtView.Text += "logouted" + Environment.NewLine;
						Logger.Log("logouted",OpenMetaverse.Helpers.LogLevel.Debug);
						break;
					case LoginStatus.Failed:
						txtView.Text += "Login Failed" + Environment.NewLine;
						Logger.Log("Login Failed",OpenMetaverse.Helpers.LogLevel.Debug);
//						//破棄して
//						LoginForm loginForm = new LoginForm();
//						loginForm.ShowDialog();
//						this.Close();
						break;
					default:
						break;
				}
			}catch(Exception exp){
				Logger.Log("ChatForm_NetCom_ClientLoginStatus:" + exp.Message,OpenMetaverse.Helpers.LogLevel.Error);
			}
			
		}
		
		void BtnChatClick(object sender, EventArgs e)
		{
			string ChatMessage = txtChat.Text;
			
			if (!NetCom.IsLoggedIn) return;
			
			if (tabtxtArea.SelectedIndex == 0){
				//chatの場合
				switch (cmbChat.SelectedIndex){
					case 0:
						NetCom.ChatOut(ChatMessage,ChatType.Normal,0);
						break;
					case 1:
						NetCom.ChatOut(ChatMessage,ChatType.Shout,0);
						break;
					case 2:
						NetCom.ChatOut(ChatMessage,ChatType.Whisper,0);
						break;
					default:
						break;
				}
			}else{
				//IMの場合
				SendIm();
			}
			
			txtChat.Text = string.Empty;
		}
		
		void SendIm()
		{
			string ChatMessage = txtChat.Text;
			UUID UUID = (UUID)tabtxtArea.SelectedTab.Name;
			UUID sessionID;
			int TabCount;
			
			usessionID.TryGetValue(UUID,out sessionID);
			tab2ID.TryGetValue(UUID, out TabCount);
			
			NetCom.SendInstantMessage(ChatMessage,UUID,sessionID);
			txtIMView[TabCount].Text += "["+ DateTime.Now.ToShortTimeString() +"]" + "<" + Client.Self.Name.ToString() + 
						"> " + ": " + ChatMessage + Environment.NewLine;
			txtIMView[TabCount].SelectionStart = txtIMView[TabCount].TextLength;
			txtIMView[TabCount].ScrollToCaret();

		}
		
		void tabGenerate(UUID AgentID,string AgentName)
		{
			try{
				if (Client.Self.AgentID == AgentID) return;
				
				if (!tab2ID.ContainsKey(AgentID)){			
					
					//txtImViewの作成
					tab2ID.Add(AgentID,TabCount);
					txtIMView[TabCount] = new RichTextBox();
					txtIMView[TabCount].Font = new System.Drawing.Font("ＭＳ Ｐゴシック", 9F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((byte)(128)));
					txtIMView[TabCount].Location = new System.Drawing.Point(0, 0);
					txtIMView[TabCount].Name = "txtIMView" + AgentID.ToString();
					txtIMView[TabCount].ScrollBars = System.Windows.Forms.RichTextBoxScrollBars.Vertical;
					txtIMView[TabCount].BackColor = System.Drawing.SystemColors.Window;
					txtIMView[TabCount].Text = "";
					txtIMView[TabCount].ReadOnly = true;
					txtIMView[TabCount].Dock = DockStyle.Fill;
					this.Controls.Add(txtIMView[TabCount]);
					
					//tabpageの作成
					tabPages[TabCount] = new TabPage();
					tabPages[TabCount].Controls.Add(txtIMView[TabCount]);
					tabPages[TabCount].Location = new System.Drawing.Point(4, 4);
					tabPages[TabCount].Name = AgentID.ToString();
					tabPages[TabCount].Padding = new System.Windows.Forms.Padding(3);
					tabPages[TabCount].UseVisualStyleBackColor = true;
					tabPages[TabCount].Text = AgentName + "  ";
					tabPages[TabCount].Dock = DockStyle.Fill;
					tabtxtArea.Controls.Add(tabPages[TabCount]);
					
					TabCount++;
				}
			}catch(Exception exp){
				Logger.Log("ChatForm_tabGenerate:" + exp.Message,OpenMetaverse.Helpers.LogLevel.Error);
			}
		}
		
	}
}
