/*
 * Copyright (c) 2012, Takeyuki Nagao
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the
 * following conditions are met:
 * 
 *  * Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *  * Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *    
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */

package jp.sourceforge.dvibrowser.dvi2epub;

import java.awt.image.BufferedImage;
import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.util.Date;
import java.util.List;
import java.util.ArrayList;
import java.util.UUID;
import java.util.logging.LogManager;

import javax.imageio.ImageIO;
import org.apache.commons.lang.*;
import org.apache.commons.io.*;

import jp.sourceforge.dvibrowser.dvi2epub.cmd.AnnotatedCommand;
import jp.sourceforge.dvibrowser.dvi2epub.cmd.Command;
import jp.sourceforge.dvibrowser.dvi2epub.cmd.CommandException;
import jp.sourceforge.dvibrowser.dvi2epub.cmd.CommandUtils;
import jp.sourceforge.dvibrowser.dvi2epub.epub.EpubWriter;
import jp.sourceforge.dvibrowser.dvi2epub.opt.DoubleValueOption;
import jp.sourceforge.dvibrowser.dvi2epub.opt.IntValueOption;
import jp.sourceforge.dvibrowser.dvicore.DviRect;
import jp.sourceforge.dvibrowser.dvicore.api.DviDocument;
import jp.sourceforge.dvibrowser.dvicore.api.DviPage;
import jp.sourceforge.dvibrowser.dvicore.ctx.DefaultDviContext;
import jp.sourceforge.dvibrowser.dvicore.ctx.DviToolkit;
import jp.sourceforge.dvibrowser.dvicore.gui.swing.ViewSpec;

public class Dvi2EpubCmd extends AnnotatedCommand {
	private int paddingSize;
	private double dpi = 100;

	public Dvi2EpubCmd() throws CommandException {
		super();
	}

	@Override
	public String getCommandLineSyntax() throws CommandException {
		return "[options] <DVI file> <Output File(EPUB)>";
	}

	@Override
	protected int processCommandLine() throws CommandException {
		try {
			{
				InputStream in = Dvi2EpubCmd.class.getResourceAsStream
           ("logging.properties");
        if (in != null) {
          LogManager.getLogManager().readConfiguration(in);
          in.close();
        } else {
          System.err.println("Failed to load logging.properties.");
        }
			}
			
			DefaultDviContext ctx = new DefaultDviContext();
			ViewSpec viewSpec = new ViewSpec(ctx);
			viewSpec.setResolution(viewSpec.getResolution().approximate(dpi));
			
			String formatName = "PNG";
			String formatExt = "png";
			String formatMime = "image/png";
			
			String[] args = getArgs();
			if (args.length != 2) {
				showUsage();
				return Command.EXIT_ERROR;
			}
			
			File inputFile = new File(args[0]);
			File outputFile = new File(args[1]);
			FileOutputStream fos = new FileOutputStream(outputFile);
			EpubWriter epubWriter = new EpubWriter(fos);
			String title = inputFile.getName();
			epubWriter.setTitle(title);
			epubWriter.setAuthor("dvi2epub");
			
			DviDocument doc = ctx.openDviDocument(inputFile);
			DviToolkit toolkit = ctx.getDviToolkit();
      List<String> recs = new ArrayList<String>();
			for (DviPage page : doc.getPages()) {
				int pn = page.getPageNumber() + 1;
				System.out.printf("[%d]", pn);
				DviRect[] bboxes = toolkit.computeBoundingBoxes(doc, viewSpec.getResolution());
				
				DviRect bbox = DviRect.union(bboxes);
				bbox = bbox.addPadding(getPaddingSize());
				BufferedImage img = toolkit.renderToBufferedImage(page, bbox, viewSpec);
				String imageFile = String.format("images/page%04d.%s", pn, formatExt);
				File png = File.createTempFile("dvi2epub", "." + formatExt);
				png.deleteOnExit();
				ImageIO.write(img, formatName, png);
				epubWriter.writeFile(imageFile, formatMime, png, false, false, false);
				String docType = "<?xml version='1.0' encoding='UTF-8'?>\n<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.1//EN\" \"http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd\">\n";
				String body = "";
				epubWriter.writeFile(String.format("page%04d.html", pn), "application/x-html",
						String.format("%s<html xmlns='http://www.w3.org/1999/xhtml' xml:lang='ja' lang='ja'><body>%s<img src='images/page%04d.%s' /></body></html>", docType, body, pn, formatExt)
						, true, true, true);
				png.delete();
        recs.add(
          String.format("{'id':%d, url:'%s', image:'%s'}",
            pn, "page.html", imageFile
          )
        );
			}
      if (true) {
        // SCORM
        UUID orgUUID = UUID.randomUUID();
        UUID itemUUID = UUID.randomUUID();
        UUID resUUID = UUID.randomUUID();
        List<String> manifest = IOUtils.readLines(
          Dvi2EpubCmd.class.getResourceAsStream("tpl/manifest.txt"),
          "UTF-8"
        );
        String pages = '[' + StringUtils.join(recs, ',') + ']';
        for (String line : manifest) {
          String [] rec = line.split(",", -1);
          String mimeType = rec[0];
          String path = rec[1];
          String c = IOUtils.toString(
            Dvi2EpubCmd.class.getResourceAsStream("tpl/" + path), "UTF-8"
          );
          c = c.replace("$(UUID)",       epubWriter.getUUID().toString());
          c = c.replace("$(ORG_UUID)",   orgUUID.toString());
          c = c.replace("$(TITLE)",      title);
          c = c.replace("$(ITEM_UUID)",  itemUUID.toString());
          c = c.replace("$(RES_UUID)",   resUUID.toString());
          c = c.replace("$(ITEM_TITLE)", title);
          c = c.replace("$(PAGES)",      pages);
          epubWriter.writeFile(path, mimeType, c, false, false, false);
        }
      }
			epubWriter.close();
			System.out.println();
			System.out.println("Output written on " + outputFile);
			
			return Command.EXIT_SUCCESS;
		} catch (Exception e) {
			throw new CommandException(e);
		}
	}
	
	@Override
	public boolean wantExit() {
		return false;
	}
	
	public static void main(String[] args) throws CommandException {
		CommandUtils.executeCommand(Dvi2EpubCmd.class, args);
	}

	public int getPaddingSize() {
		return paddingSize;
	}

	@IntValueOption(shortName="p<SIZE>", longName="padding=<SIZE>", description="Set padding size to <SIZE>")
	public void setPaddingSize(int paddingSize) {
		this.paddingSize = paddingSize;
	}
	
	public double getDpi() {
		return dpi;
	}

	@DoubleValueOption(shortName="d<N>", longName="dpi=<N>", description="Set image resolution to <N> DPI")
	public void setDpi(double dpi) {
		this.dpi = dpi;
	}

}
