/*
 * Copyright (c) 2009, Takeyuki Nagao
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the
 * following conditions are met:
 * 
 *  * Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *  * Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *    
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */

package jp.sourceforge.dvibrowser.dvicore;

import jp.sourceforge.dvibrowser.dvicore.util.Canonicalizer;
import jp.sourceforge.dvibrowser.dvicore.util.SimpleCanonicalizer;

// immutable

public final class DviUnit
implements java.io.Serializable
{
  private static final long serialVersionUID = -3040236351321035760L;

  public static final DviUnit DEFAULT;

  private static final Canonicalizer<DviUnit> canonicalizer;
  static {
    canonicalizer = new SimpleCanonicalizer<DviUnit>();
    DEFAULT = canonicalizer.canonicalize(new DviUnit());
  }

  private final int num, den, mag;
  private final double inchPerDvi, pointPerDvi;
  private final int hash;

  private DviUnit() {
    this.num = DviConstants.DEFAULT_NUM;
    this.den = DviConstants.DEFAULT_DEN;
    this.mag = DviConstants.DEFAULT_MAG;
    inchPerDvi = (num/254000.0)/den*(mag/1000.0);
    pointPerDvi = (num/254000.0)/(den/DviConstants.POINT_PER_INCH)*(mag/1000.0);
    hash = hashCodeInternal();
  }

  private DviUnit(int num, int den, int mag) {
    this.num = num;
    this.den = den;
    this.mag = mag;
    checkVars();
    inchPerDvi = (num/254000.0)/den*(mag/1000.0);
    pointPerDvi = (num/254000.0)/(den/DviConstants.POINT_PER_INCH)*(mag/1000.0);
    hash = hashCodeInternal();
  }

  public static DviUnit getInstance(int num, int den, int mag)
  {
    return canonicalizer.canonicalize(
      new DviUnit(num, den, mag)
    );
  }

  private int hashCodeInternal() {
    return num + 33*(den + 33*mag);
  }

  public int numerator()     { return num; }
  public int denominator()   { return den; }
  public int magnification() { return mag; }

  public double inchPerDvi() {
    return inchPerDvi;
  }
  public double pointPerDvi() {
    return pointPerDvi;
  }
  public double dotPerDvi(int dpi) {
    return inchPerDvi * dpi;
  }
  public double mmPerDvi(int dpi) {
    return inchPerDvi * DviConstants.MM_PER_INCH;
  }

  public int mapToPixel(int a, int dpi) {
    return (int) mapToPixelDouble(a, dpi);
  }
  public int mapToPixel(int a, DviResolution res) {
    return (int) mapToPixelDouble(a, res);
  }

  public double factorDouble(int dpi) {
    return dpi * inchPerDvi;
  }
  public double factorDouble(DviResolution res) {
    return res.actualDpi() * inchPerDvi;
  }
  public double mapToPixelDouble(int a, int dpi) {
    return (dpi * inchPerDvi * a + 0.5);
  }
  public double mapToPixelDouble(int a, DviResolution res) {
    return (res.actualDpi() * inchPerDvi * a + 0.5);
  }

  private void checkVars() {
    if (num <= 0 || den <= 0 || mag <= 0)
      throw new IllegalArgumentException
        ("Invalid values of num/den*mag");
  }

  private volatile String string = null;
  public String toString() {
    if (string == null) {
      string = getClass().getName() + "[" + num + "/" + den + "*" + mag + "]";
    }
    return string;
  }
  public boolean equals(Object obj) {
    if (this == obj) return true;
    if (obj instanceof DviUnit) {
      DviUnit a = (DviUnit) obj;
      return (a.num == num && a.den == den && a.mag == mag);
    }
    return false;
  }
  public int hashCode() {
    return hash;
  }
}
