package jp.sourceforge.dvibrowser.dvi2epub.epub;

import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

public class EpubWriter
{
	private static final String NCX_VERSION = "2005-1";
	private static final String BOOK_OPF_PATH = "book.opf";
	private static final String BOOK_NCX_PATH = "book.ncx";
	public static final String DEFAULT_LANGUAGE = "en";
	public static final String CONTAINER_NS = "urn:oasis:names:tc:opendocument:xmlns:container";
	public static final String OPF_NS = "http://www.idpf.org/2007/opf";
	public static final String NCX_NS = "http://www.daisy.org/z3986/2005/ncx/";
	public static final String PURL_NS = "http://purl.org/dc/elements/1.1/";
	private final ZipOutputStream zipOutput;
	private final UUID uuid;
	private String title;
	private String author;
	private String language;
	
	private final List<BookEntry> entries = new ArrayList<BookEntry>();

	public EpubWriter(OutputStream os) throws TransformerException, IOException, ParserConfigurationException
	{
		zipOutput = new ZipOutputStream(os);
		writeXml("META-INF/container.xml", createContainerDOM());
		uuid = UUID.randomUUID();
	}
	
	public void close() throws IOException, ParserConfigurationException, TransformerException
	{
		writeXml(BOOK_NCX_PATH, createNCXDOM());
		writeXml(BOOK_OPF_PATH, createOPFDOM());
		writeRawFile("mimetype", "application/epub+zip");
		zipOutput.flush();
		zipOutput.close();
	}
	
	public void writeXml(String path, Document doc) throws TransformerException, IOException {
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		TransformerFactory tf = TransformerFactory.newInstance();
		Transformer t = tf.newTransformer();
		t.setOutputProperty(OutputKeys.INDENT, "yes");
		t.setOutputProperty(OutputKeys.ENCODING, "UTF-8");
		DOMSource source = new DOMSource(doc);
		StreamResult result = new StreamResult(baos);
		t.transform(source, result);
		baos.close();
		byte[] data = baos.toByteArray();
		
		ZipEntry entry = new ZipEntry(path);
		zipOutput.putNextEntry(entry);
		zipOutput.write(data);
		zipOutput.closeEntry();
	}
	
	public void writeFile(String path, String mimeType, String data, boolean wantSpine, boolean wantManifest, boolean wantNCX) throws UnsupportedEncodingException, IOException {
		writeRawFile(path, data);
		BookEntry bookEntry = new BookEntry(path, path, mimeType, wantSpine, wantManifest, wantNCX);
		entries.add(bookEntry);
	}
	
	public void writeFile(String path, String mimeType, File file, boolean wantSpine, boolean wantManifest, boolean wantNCX) throws UnsupportedEncodingException, IOException {
		writeRawFile(path, file);
		BookEntry bookEntry = new BookEntry(path, path, mimeType, wantSpine, wantManifest, wantNCX);
		entries.add(bookEntry);
	}
	
	public void writeRawFile(String path, File file) throws UnsupportedEncodingException, IOException {
		ZipEntry entry = new ZipEntry(path);
		zipOutput.putNextEntry(entry);
		{
			FileInputStream fis = new FileInputStream(file);
			BufferedInputStream bis = new BufferedInputStream(fis);
			byte [] buf = new byte[1024];
			int len;
			while (-1 != (len = bis.read(buf))) {
				zipOutput.write(buf, 0, len);
			}
			zipOutput.flush();
			bis.close();
			fis.close();
		}
		zipOutput.closeEntry();
	}
	
	public void writeRawFile(String path, String data) throws UnsupportedEncodingException, IOException {
		ZipEntry entry = new ZipEntry(path);
		zipOutput.putNextEntry(entry);
		zipOutput.write(data.getBytes("UTF-8"));
		zipOutput.closeEntry();
	}

	protected Document createContainerDOM() throws ParserConfigurationException {
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		factory.setNamespaceAware(true);
		DocumentBuilder db = factory.newDocumentBuilder();
		DOMImplementation dom = db.getDOMImplementation();
		Document doc = dom.createDocument(CONTAINER_NS, "container", null);
		Element container = doc.getDocumentElement();
		container.setAttribute("version", "1.0");
		Element rootfiles = doc.createElementNS(CONTAINER_NS, "rootfiles");
		container.appendChild(rootfiles);
		Element rootfile = doc.createElementNS(CONTAINER_NS, "rootfile");
		rootfile.setAttribute("full-path", BOOK_OPF_PATH);
		rootfile.setAttribute("media-type", "application/oebps-package+xml");
		rootfiles.appendChild(rootfile);
		return doc;
	}
	
	protected Document createNCXDOM() throws ParserConfigurationException {
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		factory.setNamespaceAware(true);
		DocumentBuilder db = factory.newDocumentBuilder();
		DOMImplementation dom = db.getDOMImplementation();
		Document doc = dom.createDocument(NCX_NS, "ncx", null);
		Element ncx = doc.getDocumentElement();
		ncx.setAttribute("version", NCX_VERSION);
		ncx.setAttribute("lang", getLanguage());
		Element head = doc.createElementNS(NCX_NS, "head");
		ncx.appendChild(head);
		
		addMeta(doc, head, "dtb:uid", uuid.toString());
		addMeta(doc, head, "dtb:depth", "1");
		addMeta(doc, head, "dtb:totalPageCount", "0");
		addMeta(doc, head, "dtb:maxPageNumber", "0");
		
		Element docTitle = doc.createElementNS(NCX_NS, "docTitle");
		Element titleElem2 = doc.createElement("text");
		titleElem2.setTextContent(title == null ? "NO TITLE" : title);
		docTitle.appendChild(titleElem2);
		ncx.appendChild(docTitle);
		
		Element docAuthor = doc.createElementNS(NCX_NS, "docAuthor");
		Element authorElem2 = doc.createElement("text");
		authorElem2.setTextContent(author == null ? "NO AUTHOR" : author);
		docAuthor.appendChild(authorElem2);
		ncx.appendChild(docAuthor);
		
		Element navMap = doc.createElementNS(NCX_NS, "navMap");
		buildNavMap(doc, navMap);
		ncx.appendChild(navMap);

		return doc;
	}

	protected void addMeta(Document doc, Element head, String name, String value) {
		Element meta = doc.createElementNS(NCX_NS, "meta");
		meta.setAttribute("name", name);
		meta.setAttribute("content", value);
		head.appendChild(meta);
	}

	public String getLanguage() {
		return language == null ? DEFAULT_LANGUAGE : language;
	}

	protected Document createOPFDOM() throws ParserConfigurationException {
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		factory.setNamespaceAware(true);
		DocumentBuilder db = factory.newDocumentBuilder();
		DOMImplementation dom = db.getDOMImplementation();
		Document doc = dom.createDocument(OPF_NS, "package", null);
		Element pkg = doc.getDocumentElement();
		pkg.setAttribute("unique-identifier", getUniqueIdentifierName());
		Element metadata = doc.createElementNS(OPF_NS, "metadata");
		pkg.appendChild(metadata);
		
		Element titleElem = doc.createElement("title");
		titleElem.setTextContent(title == null ? "NO TITLE" : title);
		metadata.appendChild(titleElem);
		
		Element langElem = doc.createElement("language");
		langElem.setTextContent(getLanguage());
		metadata.appendChild(langElem);
		
		Element identifierElem = doc.createElement("identifier");
		identifierElem.setAttribute("scheme", "URI");
		identifierElem.setAttribute("id", getUniqueIdentifierName());
		identifierElem.setTextContent(uuid.toString());
		metadata.appendChild(identifierElem);
		
		Element manifestElem = doc.createElementNS(OPF_NS, "manifest");
		pkg.appendChild(manifestElem);
		
		buildManifest(doc, manifestElem);

		Element spineElem = doc.createElementNS(OPF_NS, "spine");
		spineElem.setAttribute("toc", "ncx");
		pkg.appendChild(spineElem);
		buildSpine(doc, spineElem);
		
		return doc;
	}

	protected void buildSpine(Document doc, Element spine) {
		for (BookEntry entry : entries) {
			if (!entry.wantSpine()) continue;
			Element itemref = doc.createElement("itemref");
			itemref.setAttribute("idref", entry.getPath());
			spine.appendChild(itemref);
		}
	}

	protected void buildManifest(Document doc, Element manifest) {
		for (BookEntry entry : entries) {
			if (!entry.wantManifest()) continue;
			Element item = doc.createElementNS(OPF_NS, "item");
			item.setAttribute("id", entry.getId());
			item.setAttribute("href", entry.getPath());
			item.setAttribute("media-type", entry.getMimetype());
			manifest.appendChild(item);
		}
		
		Element item = doc.createElementNS(OPF_NS, "item");
		item.setAttribute("id", "ncx");
		item.setAttribute("href", BOOK_NCX_PATH);
		item.setAttribute("media-type", "application/x-dtbncx+xml");
		manifest.appendChild(item);
	}
	
	protected void buildNavMap(Document doc, Element navMap) {
		int count = 1;
		for (BookEntry entry : entries) {
			if (!entry.wantNCX()) continue;
			Element navPoint = doc.createElementNS(NCX_NS, "navPoint");
			navPoint.setAttribute("class", "chapter");
			navPoint.setAttribute("id", entry.getPath());
			navPoint.setAttribute("playOrder", String.valueOf(count));
			
			Element navLabel = doc.createElementNS(NCX_NS, "navLabel");
			Element text = doc.createElementNS(NCX_NS, "text");
			text.setTextContent(String.valueOf(count));
			navLabel.appendChild(text);
			navPoint.appendChild(navLabel);
			
			Element content = doc.createElementNS(NCX_NS, "content");
			content.setAttribute("src", entry.getPath());
			
			navPoint.appendChild(content);
			navMap.appendChild(navPoint);
			count++;
		}
	}

	public String getUniqueIdentifierName() {
		return "BookId";
	}

	public ZipOutputStream getZipOutput() {
		return zipOutput;
	}

	public String getTitle() {
		return title;
	}
	
	public void setTitle(String title) {
		this.title = title;
	}

	public String getAuthor() {
		return author;
	}

	public void setAuthor(String author) {
		this.author = author;
	}

	public UUID getUuid() {
		return uuid;
	}
}
