/*
 * Copyright (c) 2009, Takeyuki Nagao
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the
 * following conditions are met:
 * 
 *  * Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *  * Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *    
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */

package jp.sourceforge.dvibrowser.dvicore.image.split;
import java.awt.Image;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.logging.Level;
import java.util.logging.Logger;

import jp.sourceforge.dvibrowser.dvicore.DviException;
import jp.sourceforge.dvibrowser.dvicore.DviRect;
import jp.sourceforge.dvibrowser.dvicore.DviRectSplitter;
import jp.sourceforge.dvibrowser.dvicore.DviResolution;
import jp.sourceforge.dvibrowser.dvicore.util.DviUtils;


public class DviImage
implements SplitImage
{
  private static final Logger LOGGER = Logger.getLogger(DviImage.class
      .getName());
  private final Image img;
  private final DviResolution res;
  private DviRect rect;
  private final ArrayList<SplitPiece> pieces = new ArrayList<SplitPiece>();
  
  public DviImage(Image img, int dpi)
  {
    if (img == null) throw new IllegalArgumentException("img can't be null");
    this.img = img;
    this.res = new DviResolution(dpi, 1);
    pieces.add(new SplitPiece() {
      public int getColumn() throws DviException {
        return 0;
      }

      public Image getImage() throws DviException {
        return DviImage.this.img;
      }

      public DviRect getRect() throws DviException {
        return DviImage.this.getRect();
      }

      public DviRectSplitter getRectSplitter() throws DviException {
        return DviImage.this.getRectSplitter();
      }

      public DviResolution getResolution() throws DviException {
        return DviImage.this.getResolution();
      }

      public int getRow() throws DviException {
        return 0;
      }

      public SplitImage getSplitImage() throws DviException {
        return DviImage.this;
      }
    });
  }

  public Image getImage()
  {
    return img;
  }

  public int getDpi()
  {
    return res.dpi();
  }

  public Collection<SplitPiece> getPieces() throws DviException {
    return pieces;
  }

  public synchronized DviRect getRect() throws DviException {
    if (rect == null) {
      rect = new DviRect(0, 0, img.getWidth(null), img.getHeight(null));
    }
    return rect;
  }

  public DviRectSplitter getRectSplitter() throws DviException {
    DviRect r = getRect();
    return new DviRectSplitter(r, r.size());
  }

  public DviResolution getResolution() {
    return res;
  }

  public Iterator<SplitPiece> iterator() {
    try {
      return getPieces().iterator();
    } catch (DviException e) {
      DviUtils.logStackTrace(LOGGER, Level.WARNING, e);
      // TODO: Think about better handle this.  Should we really use a RuntimeException?
      throw new RuntimeException(e);
    }
  }
}
