package jp.sourceforge.dvibrowser.dvicore.util;
/*
 * Copyright (c) 2009, Takeyuki Nagao
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the
 * following conditions are met:
 * 
 *  * Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *  * Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *    
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;

public class BufferFilter
extends InputStream
{
  protected ByteArrayInputStream byteArrayInput;
  private InputStream input;
  private final int bufferSize;
  private byte [] buffer;
  private int len;
  
  public BufferFilter(int bufferSize) {
    this.byteArrayInput = null;
    this.input = null;
    this.bufferSize = bufferSize;
    this.buffer = null;
    this.len = 0;
  }
  
  public synchronized void beginInput(InputStream in) throws IOException
  {
    input = in;
  }
  
  public synchronized void endInput() throws IOException
  {
    input = null;
  }
  
  protected synchronized void fill() throws IOException
  {
    if (input == null) {
      throw new IllegalStateException("Input read before beginInput() is called.");
    }
    if (byteArrayInput != null && byteArrayInput.available() > 0) {
      return;
    }
    
    byteArrayInput = fillBuffer();
  }

  protected synchronized ByteArrayInputStream fillBuffer() throws IOException
  {
    byte [] buf = new byte [bufferSize];
    
    int len = input.read(buf);
    if (-1 == len) {
      throw new IOException("EOF found while reading input.");
    }
    if (0 == len) {
      throw new IOException("Read failed.");
    }
    
    this.len = len;
    this.buffer = buf;
    
    return new ByteArrayInputStream(buf, 0, len);
  }
  
  public synchronized byte [] getBuffer()
  {
    if (buffer == null) return new byte[0];
    byte [] ret = new byte[len];
    System.arraycopy(buffer, 0, ret, 0, len);
    return ret;
  }
  
  @Override
  public synchronized int available() throws IOException
  {
    fill();
    return byteArrayInput.available();
  }
  
  @Override
  public synchronized void close() throws IOException
  {
    input.close();
  }
  
  @Override
  public void mark(int readLimit)
  {
    throw new UnsupportedOperationException("mark is not supported");
  }
  
  @Override
  public void reset()
  {
    throw new UnsupportedOperationException("reset is not supported");
  }
  
  @Override
  public boolean markSupported()
  {
    return false;
  }
  
  @Override
  public synchronized int read() throws IOException
  {
    fill();
    return byteArrayInput.read();
  }

  @Override
  public synchronized int read(byte [] buf, int off, int len) throws IOException
  {
    fill();
    return byteArrayInput.read(buf, off, len);
  }

  public synchronized InputStream getInput() {
    return input;
  }

  public int getBufferSize() {
    return bufferSize;
  }
}
