/*
 * Copyright (c) 2009, Takeyuki Nagao
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the
 * following conditions are met:
 * 
 *  * Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *  * Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *    
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */

package jp.sourceforge.dvibrowser.dvicore;

import jp.sourceforge.dvibrowser.dvicore.util.Canonicalizer;
import jp.sourceforge.dvibrowser.dvicore.util.SimpleCanonicalizer;

// immutable.

public final class DviFontSpec
implements java.io.Serializable
{
  private static final long serialVersionUID = 2374998771511821276L;
  private final int cs;          // check sum.
  private final int ss;          // Space size.
  private final int ds;          // Design size.
  private final DviFontName fontName;

  private final double dviPerTfmw;
  private final int hash; // QUESTION: Should we make this transient?

  private DviFontSpec(int cs, int ss, int ds, DviFontName fontName) {
    this.cs = cs;
    this.ss = ss;
    this.ds = ds;
    this.fontName = fontName;
    this.hash = (
      cs + 33*(ss + 33*(ds + 33*fontName.hashCode()))
    );
    dviPerTfmw = ss / (double) (1 << 20);
  }

  private static final Canonicalizer<DviFontSpec> canonicalizer
    = new SimpleCanonicalizer<DviFontSpec>();
  public static DviFontSpec getInstance(
    int cs, int ss, int ds, DviFontName fontName)
  {
    return canonicalizer.canonicalize(
      new DviFontSpec(cs, ss, ds, fontName)
    );
  }

  public static DviFontSpec getInstance(
    int cs, int ss, int ds, int al, int nl, byte [] name)
  {
    return getInstance(cs, ss, ds,
      DviFontName.getInstance(al, nl, new String(name))
    );
  }

  public static DviFontSpec getInstance(
    int cs, int ss, int ds, int al, int nl, String name)
  {
    return getInstance(cs, ss, ds,
      DviFontName.getInstance(al, nl, name)
    );
  }

  public int checkSum()    { return cs; }
  public int spaceSize()   { return ss; }
  public int designSize()  { return ds; }
  public int areaLength()  { return fontName.areaLength(); }
  public int nameLength()  { return fontName.nameLength(); }
  public DviFontName fontName() { return fontName; }
  public String name() { return fontName.name(); }

  public int tfmToDvi(int tfmw) {
    return (int)(dviPerTfmw * tfmw);
  }

  private volatile String string = null;
  public String toString() {
    if (string == null) {
      string = "FontSpec"
             + "[cs=" + cs + ",ss=" + ss + ",ds=" + ds
             + ",fontName=\"" + fontName + "\"]"
             ;
    }
    return string;
  }
  public boolean equals(Object obj) {
    if (this == obj) return true;
    if (obj instanceof DviFontSpec) {
      DviFontSpec fs = (DviFontSpec) obj;
      return (isCompatibleWith(fs) && (cs == fs.cs));
    }
    return false;
  }
  public boolean isCompatibleWith(DviFontSpec fs) {
    return (
             (fs != null)
             && (ss == fs.ss) && (ds == fs.ds)
             && fontName.equals(fs.fontName)
           );
  }

  public int hashCode() {
    return hash;
  }

  public DviFontSpec rename(String newName)
  {
    return getInstance(cs, ss, ds,
        DviFontName.getInstance(newName)
      );
  }
}
