/*
 * Copyright (c) 2009, Takeyuki Nagao
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the
 * following conditions are met:
 * 
 *  * Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *  * Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *    
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */

package dvi.gs;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.logging.Logger;

import dvi.DviException;
import dvi.DviObject;
import dvi.DviSize;
import dvi.api.DviContextSupport;
import dvi.util.CommandShell;

public class GhostscriptCommandBuilder
extends DviObject
{
  private static final Logger LOGGER = Logger.getLogger(GhostscriptCommandBuilder.class.getName());

  public GhostscriptCommandBuilder(DviContextSupport dcs) {
    super(dcs);
  }
  
  private String dev = "png256";
  public void setDevice(String dev)
  {
    this.dev = dev;
  }

  private File outputFile;
  public void setOutputFile(File outputFile)
  {
    this.outputFile = outputFile;
  }

  private File inputFile;
  public void setInputFile(File inputFile)
  {
    this.inputFile = inputFile;
  }
  
  

  private int dpi = 300;
  public void setResolution(int dpi)
  {
    this.dpi = dpi;
  }
  
  private int firstPage = -1;
  private int lastPage = -1;

  private DviSize size = null;
  public void setPaperSize(DviSize size)
  {
    this.size = size;
  }
  
  private String gsPath = null;
  public String getGhostscriptExecutablePath() throws DviException
  {
    if (gsPath == null) {
      return getDviContext().getExecutableName("gs");
    }
    return gsPath;
  }
  
  private int textAlpha = 0;
  private int graphicsAlpha = 0;
  
  private String stdoutData = null;
  private String stderrData = null;
  
  // TODO: externalize ghostscript configurations
  protected Collection<String> buildCommandLine()
  throws DviException
  {
    ArrayList<String> cl = new ArrayList<String>();
    String gscmd = getGhostscriptExecutablePath();
    cl.add(gscmd);
    cl.add("-q");
    cl.add("-dSAFER");
    cl.add("-dPARANOIDSAFER");
    cl.add("-dDELAYSAFER");
    cl.add("-dNOPAUSE");
    cl.add("-dBATCH");
    cl.add("-sDEVICE=" + dev);
    if (outputFile != null) {
      cl.add("-sOutputFile=" + outputFile); // TODO: Escape characters in outputFile.
    } else {
      cl.add("-sOutputFile=-"); // the output will be sent to stdout.
    }
    if (firstPage > 0) {
      cl.add("-dFirstPage=" + firstPage);
    }
    if (lastPage > 0) {
      cl.add("-dLastPage=" + lastPage);
    }
    cl.add("-r" + dpi);
    if (size != null)
      cl.add("-g" + size);
    if (textAlpha > 0) {
      cl.add("-dTextAlphaBits=" + textAlpha);
    }
    if (graphicsAlpha > 0) {
      cl.add("-dGraphicsAlphaBits=" + graphicsAlpha);
    }
    
    if (inputFile != null) {
      cl.add("-f" + inputFile);
    } else {
      cl.add("-");
    }
    
    return cl;
  }
  
  public CommandShell createCommandShell() throws DviException
  {
    CommandShell cs = new CommandShell();
    Collection<String> cl = buildCommandLine();
    cs.setCommandLine(cl);
    return cs;
  }
    

  public void setTextAlpha(int textAlpha)
  {
    this.textAlpha = textAlpha;
  }

  public int getTextAlpha()
  {
    return textAlpha;
  }

  public void setGraphicsAlpha(int graphicsAlpha)
  {
    this.graphicsAlpha = graphicsAlpha;
  }

  public int getGraphicsAlpha()
  {
    return graphicsAlpha;
  }

  public void setFirstPage(int firstPage) {
    this.firstPage = firstPage;
  }

  public int getFirstPage() {
    return firstPage;
  }

  public void setLastPage(int lastPage) {
    this.lastPage = lastPage;
  }

  public int getLastPage() {
    return lastPage;
  }

  public void setGhostscriptExecutablePath(String gsPath) {
    this.gsPath = gsPath;
  }
}
