/*
 * Copyright (c) 2009, Takeyuki Nagao
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the
 * following conditions are met:
 * 
 *  * Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *  * Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *    
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */

package dvi.io;

import java.io.ByteArrayInputStream;

import dvi.DviException;
import dvi.DviFontTable;
import dvi.DviUnit;
import dvi.api.DviData;
import dvi.api.DviInput;

public class ByteArrayDviData
implements DviData
{
  private final byte [] buf;
  private final DviFontTable fontTable;
  private final DviUnit dviUnit;

  public ByteArrayDviData(byte [] buf)
  {
    this(buf, null, null);
  }

  public ByteArrayDviData(byte [] buf, DviUnit dviUnit)
  {
    this(buf, dviUnit, null);
  }

  public ByteArrayDviData(byte [] buf, DviUnit dviUnit, DviFontTable fontTable)
  {
    this.buf = buf;
    this.dviUnit = dviUnit;
    this.fontTable = fontTable;
    if (buf == null)
      throw new IllegalArgumentException
        ("dvi data buffer cannot be null");
  }

  public DviInput getInput() {
    return
      new DviInputStreamReader(
        new ByteArrayInputStream(buf)
      );
  }

  public DviFontTable getFontTable() {
    return fontTable;
  }

  public DviUnit getDviUnit() {
    return dviUnit;
  }

  public long getDataSize() throws DviException
  {
    return buf.length;
  }

  public DviInput getInput(long start, long end) throws DviException
  {
    if (start < 0 || end < 0 || end < start)
      throw new IllegalArgumentException
        (String.format("Invalid byte range: (%d,%d)", start, end));
    if (start >= buf.length)
      throw new IllegalArgumentException
      ("start position out of bounds: " + start);
    if (end >= buf.length)
      throw new IllegalArgumentException
      ("end position out of bounds: " + end);

    DviInputStreamReader in = new DviInputStreamReader(
      new ByteArrayInputStream(
          this.buf, (int) start, (int)(end - start) + 1
        )
      );
    in.setOffset(start);
    return in;
  }
}
