/*
 * Copyright (c) 2009, Takeyuki Nagao
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the
 * following conditions are met:
 * 
 *  * Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *  * Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *    
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */

package dvi.render;

import dvi.api.GammaCorrector;
import dvi.util.DviCache;

public class GammaCorrectorCache
implements GammaCorrector
{
  private static final int MAX_LOOKUP_TABLE_SIZE = 65536;
  private int maxval;
  private LookupTable table;
  private boolean canLookUp;
  private final GammaCorrector gammaCorrector;
  
  private static class LookupTable
  {
    private final int[] data;

    public LookupTable(int maxval)
    {
      this.data = new int [maxval+1];
    }
  }
  
  private static final DviCache<String, LookupTable>
    cache = new DviCache<String, LookupTable>(10);
 
  public static GammaCorrector wrap(GammaCorrector gammaCorrector)
  {
    return new GammaCorrectorCache(gammaCorrector);
  }
  
  private GammaCorrectorCache(GammaCorrector gammaCorrector)
  {
    this.gammaCorrector = gammaCorrector;
  }
  
  public void setMaxValue(int maxval)
  {
    this.maxval = maxval;
    if (maxval < MAX_LOOKUP_TABLE_SIZE) {
      String key = gammaCorrector.getCacheKey() + "--" + maxval;
      LookupTable table = cache.get(key);
      if (table == null) {
        table = new LookupTable(maxval);
        for (int i=0; i<=maxval; i++) {
          table.data[i] = doCorrectGamma(i, maxval);
        }
        cache.put(key, table);
      }
      this.table = table;
      canLookUp = true;
    } else {
      canLookUp = false;
    }
  }

  public int correctGamma(int c, int maxval)
  {
    if (c < 0) throw new IllegalArgumentException("c is negative");
    if (maxval < 0) throw new IllegalArgumentException("maxval is negative");
    
    if (gammaCorrector == null) return 0;
    if (maxval != this.maxval) setMaxValue(maxval);
    if (canLookUp && c <= maxval) {
      return table.data[c];
    }
    return doCorrectGamma(c, maxval);
  }
  
  private int doCorrectGamma(int c, int maxval)
  {
    return gammaCorrector.correctGamma(c, maxval);
  }

  public GammaCorrector getGammaCorrector()
  {
    return gammaCorrector;
  }

  public String getCacheKey() {
    return null;
  }
}
