/*
 * Copyright (c) 2009, Takeyuki Nagao
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the
 * following conditions are met:
 * 
 *  * Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *  * Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *    
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */

package dvi.util;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.URI;
import java.util.logging.Logger;

public class DviDesktop {
  private static final Logger LOGGER = Logger.getLogger(DviDesktop.class
      .getName());
  
  private static final DviDesktop instance = new DviDesktop();
  
  private DviDesktop()
  {
  }
  
  public static DviDesktop getDesktop()
  {
    return instance;
  }
  
  public static boolean isDesktopSupported()
  {
    if (DviUtils.isMacOSX()) {
      return true;
    } else if (DviUtils.isWindows()) {
      return true;
    } else {
      return false;
    }
  }
  
  public void browse(URI uri)
  throws IOException
  {
    throw new UnsupportedOperationException();
  }

  public void open(File file)
  throws IOException
  {
    if (DviUtils.isMacOSX()) {
      try {
        Class<?> cls = Class.forName("com.apple.eio.FileManager");
        Method openURL = cls.getDeclaredMethod("openURL",
            new Class[] { String.class });
        openURL.invoke(null, new Object[] { file.toURL().toExternalForm() });
      } catch (SecurityException e) {
        LOGGER.warning(e.toString());
      } catch (IllegalArgumentException e) {
        LOGGER.warning(e.toString());
      } catch (ClassNotFoundException e) {
        LOGGER.warning(e.toString());
      } catch (NoSuchMethodException e) {
        LOGGER.warning(e.toString());
      } catch (IllegalAccessException e) {
        LOGGER.warning(e.toString());
      } catch (InvocationTargetException e) {
        LOGGER.warning(e.toString());
      }
    } else if (DviUtils.isWindows()) {
      String[] commandLine = { "cmd.exe", "/c", "start", file.getAbsolutePath() };
      try {
        Process p = Runtime.getRuntime().exec(commandLine, null, null);
        int ret = p.waitFor();
        LOGGER.fine("Process exit with retcode: " + ret);
      } catch (InterruptedException e) {
        LOGGER.warning(e.toString());
      }
    } else {
      LOGGER.severe("This platform does not support file association.");
    }
  }
}
