/*
 * Copyright (c) 2009, Takeyuki Nagao
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the
 * following conditions are met:
 * 
 *  * Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *  * Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *    
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */

package dvi.v2.xdoc.ps;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.logging.Logger;

import dvi.DviException;
import dvi.DviObject;
import dvi.DviPaperSize;
import dvi.DviRect;
import dvi.api.DviContextSupport;
import dvi.gs.GhostscriptUtils;
import dvi.v2.xdoc.XDocument;
import dvi.v2.xdoc.XPage;
public class GhostscriptXDocument
extends DviObject
implements XDocument
{
  private static final Logger LOGGER = Logger
      .getLogger(GhostscriptXDocument.class.getName());
  
  private final File file;
  private XPage [] pages;
  protected DviRect [] bboxes;
  private boolean parsed = false;

  private DviPaperSize paperSize;

  public GhostscriptXDocument(DviContextSupport dcs, File file, DviPaperSize paperSize)
  {
    super(dcs);
    this.file = file;
    this.paperSize = paperSize;
  }

  public synchronized XPage getPage(int page) throws DviException {
    delayedParse();
    if (page < 0 || page > bboxes.length)
      throw new IllegalArgumentException("page number out of bounds: " + page);
    return pages[page];
  }

  protected synchronized void delayedParse() throws DviException {
    if (!parsed) {
      doParse();
      parsed = true;
    }
  }

  protected synchronized void doParse() throws DviException {
    try {
      ArrayList<XPage> list = new ArrayList<XPage>();
      bboxes = GhostscriptUtils.computePostScriptBoundingBoxes
        (getDviContext(), file, 72);
      for (int i=0; i<bboxes.length; i++) {
        XPage page = new GhostscriptXPage(this, i);
        list.add(page);
      }
      pages = list.toArray(new XPage[list.size()]);
    } catch (IOException e) {
      throw new DviException(e);
    } catch (InterruptedException e) {
      throw new DviException(e);
    }
  }

  public synchronized XPage[] getPages() throws DviException {
    delayedParse();
    return pages;
  }

  public synchronized int getTotalPages() throws DviException {
    delayedParse();
    return bboxes.length;
  }
  
  public File getFile() {
    return file;
  }

  public DviPaperSize getPaperSize() {
    return paperSize;
  }

}
