/*
 * Copyright (c) 2009, Takeyuki Nagao
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the
 * following conditions are met:
 * 
 *  * Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *  * Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *    
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */

package dvi.browser;

import java.net.URL;
import java.util.LinkedList;

import dvi.DviObject;
import dvi.api.DviContextSupport;

public class DviURLHistory
extends DviObject
{
  private final LinkedList<DviURLHistoryRecord> list
    = new LinkedList<DviURLHistoryRecord>();
  
  public DviURLHistory(DviContextSupport dcs) {
    super(dcs);
  }
  
  public synchronized DviURLHistoryRecord moveTo(URL url)
  {
    if (getCurrentURL() != url) 
      recordURL(url);
    return getCurrentRecord();
  }
  
  public synchronized DviURLHistoryRecord goBack()
  {
    if (list.size() <= 1) return null;
    list.removeLast();
    return getCurrentRecord();
  }
  
  public synchronized DviURLHistoryRecord getCurrentRecord()
  {
    if (list.size() > 0)
      return list.getLast();
    return null;
  }
  
  public synchronized URL getCurrentURL()
  {
    DviURLHistoryRecord rec = getCurrentRecord();
    if (rec == null) return null;
    return rec.getURL();
  }
  
  public synchronized DviURLHistoryRecord [] listRecords()
  {
    return list.toArray(new DviURLHistoryRecord[list.size()]);
  }

  protected synchronized DviURLHistoryRecord recordURL(URL url)
  {
    long timestamp = getTimestamp();
    DviURLHistoryRecord rec = new DviURLHistoryRecord(this, url, timestamp);
    list.add(rec);
    return rec;
  }
  
  protected long getTimestamp()
  {
    return System.currentTimeMillis();
  }
  
  public synchronized String toString()
  {
    StringBuilder sb = new StringBuilder();
    for (DviURLHistoryRecord rec : list) {
      sb.append(rec);
      sb.append("\n");
    }
    return sb.toString();
  }
}
