/*
 * Copyright (c) 2009, Takeyuki Nagao
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the
 * following conditions are met:
 * 
 *  * Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *  * Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *    
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */

package dvi.gs;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Scanner;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;

import dvi.DviException;
import dvi.DviPaperSize;
import dvi.DviRect;
import dvi.DviResolution;
import dvi.DviSize;
import dvi.api.DviContext;
import dvi.image.pnm.PnmSplitter;
import dvi.image.split.SplitImageWriter;
import dvi.util.BufferFilter;
import dvi.util.CommandShell;
import dvi.util.CommandShellHandler;
import dvi.util.DviUtils;
public class GhostscriptUtils {
  private static final class CommandShellHandlerImpl implements
      CommandShellHandler {
    private final ArrayList<String> list;
    private final String path;
    public CommandShellHandlerImpl(String path, ArrayList<String> list) {
      this.list = list;
      this.path = path;
    }

    public void handleStderr(InputStream in) throws IOException,
        DviException {
      DviUtils.logLinesFromStream("[stderr]", in, LOGGER, Level.WARNING);
    }

    public void handleStdin(OutputStream out) throws IOException,
        DviException {
      out.close();
    }

    public void handleStdout(InputStream in) throws IOException,
        DviException {
      try {
        Scanner s = new Scanner(in);
        String line = s.nextLine();
        if (line.indexOf("Ghostscript") != -1) {
          list.add(path);
        }
        in.close();
      } finally {
        DviUtils.silentClose(in);
      }
      
    }
  }

  private static final Logger LOGGER = Logger.getLogger(GhostscriptUtils.class
      .getName());
  
  public static DviRect [] computePostScriptBoundingBoxes(DviContext ctx, File file, int dpi) throws DviException, IOException, InterruptedException
  {
    if (ctx == null)
      throw new IllegalArgumentException("ctx can't be null");
    
    if (file == null)
      throw new IllegalArgumentException("file can't be null");

    if (dpi <= 0)
      throw new IllegalArgumentException("dpi <= 0: " + dpi);

    GhostscriptCommandBuilder gs = new GhostscriptCommandBuilder(ctx);
    gs.setInputFile(file);
    gs.setResolution(dpi);
    gs.setDevice("bbox");
    CommandShell shell = gs.createCommandShell();
    GhostscriptBBOXParser handler = new GhostscriptBBOXParser();
    shell.setHandler(handler);
    shell.execute();

    ArrayList<DviRect> list = new ArrayList<DviRect>();

    LOGGER.info("total pages=" + handler.getTotalPages());
    for (int i = 0; i < handler.getTotalPages(); i++) {
      DviRect bbox = handler.getBoundingBoxOfPage(i);
      LOGGER.info("bounding box[" + i + "]=" + bbox);
      list.add(bbox);
    }
    return list.toArray(new DviRect[list.size()]);
  }
   
  public static void renderAndSplit(final DviContext ctx, final InputStream is,
      DviSize unitSize, final SplitImageWriter imageWriter,
      DviResolution res,
      String deviceName, final int pageNum, final long timeout,
      final TimeUnit timeUnit) throws DviException {
    if (deviceName == null) {
      deviceName = "pnmraw";
    }

    GhostscriptCommandBuilder gs = new GhostscriptCommandBuilder(ctx);
    gs.setInputFile(null);
    gs.setOutputFile(null);
    gs.setResolution(res.dpi());
    gs.setDevice(deviceName);
    gs.setFirstPage(pageNum);
    gs.setLastPage(pageNum);
    gs.setTextAlpha(0); // textAlpha and graphicsAlpha have no effects for PNM
                        // devices.
    gs.setGraphicsAlpha(0);
    CommandShell shell = gs.createCommandShell();
    shell.setTimeout(timeout, timeUnit);
    final PnmSplitter splitter = new PnmSplitter(unitSize, imageWriter);
    splitter.setShrinkFactor(res.shrinkFactor());
    final BufferFilter stdoutFilter = new BufferFilter(16384);
    CommandShellHandler handler = new CommandShellHandler() {
      public void handleStderr(InputStream in) throws IOException {
        try {
          DviUtils.dumpStreamAsync("gs stderr", in, System.err).join();
        } catch (InterruptedException e) {
          DviUtils.logStackTrace(LOGGER, Level.WARNING, e);
        }
      }

      public void handleStdin(OutputStream out) throws IOException {
        try {
          byte[] buf = new byte[16384];
          BufferedInputStream bis = new BufferedInputStream(is);
          do {
            int len = bis.read(buf);
            if (len < 0)
              break;
            out.write(buf, 0, len);
          } while (true);
        } finally {
          DviUtils.silentClose(out);
          DviUtils.silentClose(is);
        }
      }

      public void handleStdout(InputStream in) throws IOException, DviException {
        stdoutFilter.beginInput(in);
        try {
          splitter.splitImageFromStream(stdoutFilter);
        } finally {
          try {
            stdoutFilter.endInput();
          } catch (Exception e) {
            DviUtils.logStackTrace(LOGGER, Level.WARNING, e);
          }
          DviUtils.silentClose(in);
        }
      }
    };
    shell.setHandler(handler);
    int ret;
    
    try {
      ret = shell.execute();
    } catch (IOException e) {
      LOGGER.warning("Ghostscript failed with output: " + new String(stdoutFilter.getBuffer()));
      throw new DviException(e);
    } catch (InterruptedException e) {
      LOGGER.warning("Ghostscript failed with output: " + new String(stdoutFilter.getBuffer()));
      throw new DviException(e);
    }
    if (ret != 0) {
      LOGGER.warning("Ghostscript failed with output: " + new String(stdoutFilter.getBuffer()));
      throw new DviException("Ghostscript failed with exit code " + ret);
    }
  }
  
  public static DviRect mapToDviCoordinate(DviRect postScriptRect, DviPaperSize paperSize, DviResolution res)
  {
    DviRect paperRect = paperSize.toBoundingBox(res);
    return new DviRect(postScriptRect.x(), paperRect.y() - postScriptRect.y(), postScriptRect.width(), postScriptRect.height());
  }
  
  public static String [] listGhostscriptExecutables()
  {
    final ArrayList<String> list = new ArrayList<String>();
    final ArrayList<String> candidates = new ArrayList<String>();
    
    if (DviUtils.isWindows()) {
      candidates.add("gswin32c");
    }
    candidates.add("gs");
    
    LOGGER.fine("Start detecting Ghostscript.");
    for (String path : candidates) {
      LOGGER.fine("Trying " + path);
      try {
        CommandShell cs = new CommandShell();
        cs.setCommandLine(path, "-dSAFER", "-dPARANOIDSAFER", "-dDELAYSAFER", "-dNOPAUSE", "-dBATCH", "-help");
        cs.setHandler(new CommandShellHandlerImpl(path, list));
        cs.execute();
      } catch (Exception e) {
        DviUtils.logStackTrace(LOGGER, Level.WARNING, e);
      }
    }
    LOGGER.fine("Finished detecting Ghostscript.");
    
    return list.toArray(new String[list.size()]);
  }
}
