﻿// In the original BSD license, both occurrences of the phrase "COPYRIGHT HOLDERS AND CONTRIBUTORS"
// in the disclaimer read "REGENTS AND CONTRIBUTORS".
//
// Here is the license template:
//
// Copyright (c) 2010, Masanori Usami
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without modification, are permitted provided
// that the following conditions are met:
//
//  * Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
//  * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer
//    in the documentation and/or other materials provided with the distribution.
//  * Neither the name of the <ORGANIZATION> nor the names of its contributors may be used to endorse or promote products derived
//    from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
// BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
// IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, 
// OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
// OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
// OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF 
// THE POSSIBILITY OF SUCH DAMAGE.using System;

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Collections;

namespace EAAddinDSM
{
    /// <summary>
    /// MVCパターンのModelに相当するクラス
    /// DSMのデータ本体を格納する
    /// </summary>
    class DSMModel : Model
    {
        public const string ELEMENT_TYPE_OBJECT = "object";
        public const string ELEMENT_TYPE_CLASS = "class";
        
        private int[,] dsm_;
        private ArrayList elementList_;
        private string elementType_;
        private EA.Repository repository_;

        /// <summary>
        /// コンストラクタ
        /// EAのリポジトリを受け取る
        /// DSMの処理対象となるダイアグラムとDSMのタイプ（クラス/オブジェクト）を受け取る
        /// </summary>
        /// <param name="repository">EAリポジトリ</param>
        /// <param name="diagram">処理対象のダイアグラム</param>
        /// <param name="type">クラス／オブジェクト</param>
        public DSMModel(EA.Repository repository, EA.Diagram diagram, string type)
        {
            if ((repository == null || diagram == null) ||
                (type == ELEMENT_TYPE_CLASS || type == ELEMENT_TYPE_OBJECT))
            {
                throw new System.ArgumentException("Invalid Parameter", "original");
            }
            else
            {
                elementType_ = type;
                repository_ = repository;
                elementList_ = new ArrayList();
                Hashtable dsmHash = new Hashtable();
                foreach (EA.DiagramObject dgmObject in diagram.DiagramObjects)
                {
                    EA.Element source = repository.GetElementByID(dgmObject.ElementID);
                    if (source.Type == elementType_)
                    {
                        elementList_.Add(source.ElementID);
                        dsmHash[source.ElementID] = getDependencyList(source);
                    }
                }
                dsm_ = makeDSMArray(dsmHash);
            }
        }

        /// <summary>
        /// コンストラクタ
        /// EAのリポジトリを受け取る
        /// DSMの処理対象となるパッケージとDSMのタイプ（クラス/オブジェクト）を受け取る
        /// </summary>
        /// <param name="repository">EAリポジトリ</param>
        /// <param name="package">処理対象のパッケージ</param>
        /// <param name="type">クラス／オブジェクト</param>
        public DSMModel(EA.Repository repository, EA.Package package, string type)
        {
            if ((repository == null || package == null) ||
                (type == ELEMENT_TYPE_CLASS || type == ELEMENT_TYPE_OBJECT))
            {
                throw new System.ArgumentException("Invalid Parameter", "original");
            }
            else
            {
                elementType_ = type;
                repository_ = repository;
                elementList_ = new ArrayList();
                Hashtable dsmHash = new Hashtable();
                foreach (EA.Element source in getAllPackageElements(package))
                {
                    elementList_.Add(source.ElementID);
                    dsmHash[source.ElementID] = getDependencyList(source);
                }
                dsm_ = makeDSMArray(dsmHash);
            }
        }

        /// <summary>
        /// コンストラクタ
        /// 呼ばれないようにPrivateで定義する
        /// </summary>
        private DSMModel()
        {
            
        } 

        /// <summary>
        /// DSMを返す（手持ちのDSMをそのまま返す）
        /// </summary>
        /// <returns>DSMの配列</returns>
        public int[,] getDSM()
        {
            return dsm_;
        }

        /// <summary>
        /// 要素の名称リストを返す（これはバグの元。getDSMと同じ扱いにしないとまずい）
        /// </summary>
        /// <returns>名称リストをArrayList（string）の形式で返す</returns>
        public ArrayList getElementNameList()
        {
            ArrayList list = new ArrayList();

            for (short i = 0; i < elementList_.Count; i++)
            {
                EA.Element iElement = repository_.GetElementByID((int)elementList_[i]);
                list.Add(iElement.Name);
            }

            return list;
        }

        /// <summary>
        /// DSM（と名称リスト）を指定の順序に並べ替える
        /// パーティショニング用の仕組み
        /// </summary>
        /// <param name="newSeq">並べ替える順番（数字のリスト）</param>
        /// <returns>true:成功 false:失敗</returns>
        public bool setNewSequence(ArrayList newSeq)
        {
            if (newSeq.Count == dsm_.GetLength(0))
            {
                int[,] partitionedDSM = new int[dsm_.GetLength(0), dsm_.GetLength(1)];
                ArrayList newElementList = (ArrayList)elementList_.Clone();
                for (int i = 0; i < dsm_.GetLength(0); i++)
                {
                    for (int j = 0; j < dsm_.GetLength(1); j++)
                    {
                        partitionedDSM[i, j] = dsm_[(int)newSeq[i], (int)newSeq[j]];
                    }
                    newElementList[i] = elementList_[(int)newSeq[i]];
                }

                dsm_ = partitionedDSM;
                elementList_ = newElementList;
                update();
                return true;
            }

            return false;
        }

        /// <summary>
        /// 指定されたパッケージ中の要素を全て再帰的に取得する
        /// パッケージが再起構造の場合は、再帰的に処理する
        /// 要素はelementType_で指定されたタイプに一致するものだけ取得する
        /// </summary>
        /// <param name="package">要素を取得するパッケージ</param>
        /// <returns>要素のリスト</returns>
        private ArrayList getAllPackageElements(EA.Package package)
        {
            ArrayList packageList = new ArrayList();
            foreach (EA.Element source in package.Elements)
            {
                if (source.Type == elementType_)
                {
                    packageList.Add(source);
                }
            }
            foreach (EA.Package pack in package.Packages)
            {
                foreach (EA.Element source in getAllPackageElements(pack))
                {
                    packageList.Add(source);
                }
            }
            return packageList;
        }

        /// <summary>
        /// source要素が依存する要素のリストを返す
        /// </summary>
        /// <param name="source">対象の要素</param>
        /// <returns>source要素が依存する要素のリスト</returns>
        private ArrayList getDependencyList(EA.Element source)
        {
            ArrayList array = new ArrayList();
            foreach (EA.Connector conn in source.Connectors)
            {
                EA.Element dest;
                if (conn.SupplierID == source.ElementID)
                {
                    dest = repository_.GetElementByID(conn.ClientID);
                }
                else
                {
                    dest = repository_.GetElementByID(conn.SupplierID);
                }
                if (dest.Type == elementType_)
                {
                    // パッケージ中のエレメントの接続先をチェック
                    if (checkDirection(source, dest, conn))
                    {
                        array.Add(dest.ElementID);
                    }
                }
            }
            return array;
        }

        /// <summary>
        /// SOURCEがDESTINATIONに依存しているかどうかのチェック
        /// DESITINATIONからSOURCEの場合は依存していないとみなす
        /// </summary>
        /// <param name="source">チェック対象の要素</param>
        /// <param name="dest">SOURCEが依存しているかどうかの調査対象の要素</param>
        /// <param name="conn">SOURCEが依存しているかどうかの調査対象の接続</param>
        /// <returns>依存しているときtrue / 依存していないときfalse</returns>
        private bool checkDirection(EA.Element source, EA.Element dest, EA.Connector conn)
        {
            switch (conn.Type)
            {
                case "Aggregation":
                    return checkAggregationDependency(source, dest, conn);
                //case "Association":
                //    break;
                //case "Dependency":
                //    break;
                //case "Generalization":
                //    break;
                //case "Nesting":
                //    break;
                //case "NoteLink":
                //    break;
                //case "Realization":
                //    break;
                //case "Sequence":
                //    break;
                //case "Transition":
                //    break;
                //case "UseCase":
                //    break;
            }
            switch (conn.Direction)
            {
                case "Unspecified":
                    if ((conn.SupplierID == source.ElementID) && (conn.ClientID == dest.ElementID))
                    {
                        return true;
                    }
                    break;
                case "Bi-Directional":
                    return true;
                case "Source -> Destination":
                    if ((conn.SupplierID == source.ElementID) && (conn.ClientID == dest.ElementID))
                    {
                        return true;
                    }
                    break;
                case "Destination -> Source":
                    if ((conn.SupplierID == dest.ElementID) && (conn.ClientID == source.ElementID))
                    {
                        return true;
                    }
                    break;
            }
            return false;
        }

        /// <summary>
        /// SOURCEがDESTINATIONに依存しているかどうかのチェック
        /// AssociationのみDestinationからSourceに向かって依存する
        /// </summary>
        /// <param name="source">チェック対象の要素</param>
        /// <param name="dest">SOURCEが依存しているかどうかの調査対象の要素</param>
        /// <param name="conn">SOURCEが依存しているかどうかの調査対象の接続</param>
        /// <returns>依存しているときtrue / 依存していないときfalse</returns>
        private bool checkAggregationDependency(EA.Element source, EA.Element dest, EA.Connector conn)
        {
            if ((conn.SupplierID == dest.ElementID) && (conn.ClientID == source.ElementID))
            {
                return true;
            }
            return false;
        }

        /// <summary>
        /// HashTableから2次元配列のDSM配列を作成する
        /// </summary>
        /// <param name="dsm">DSMを含むHashTable</param>
        /// <returns>DSMの2次元配列</returns>
        private int[,] makeDSMArray(Hashtable dsm)
        {
            int[,] result = new int[elementList_.Count, elementList_.Count];

            for (short i = 0; i < elementList_.Count; i++)
            {
                for (short j = 0; j < elementList_.Count; j++)
                {
                    if (i == j)
                    {
                        result[i, j] = 1;
                    }
                    else
                    {
                        if (dsm.ContainsKey((int)elementList_[i]))
                        {
                            if (((ArrayList)dsm[(int)elementList_[i]]).Contains((int)elementList_[j]))
                            {
                                result[i, j] = 1;
                            }
                            else
                            {
                                result[i, j] = 0;
                            }
                        }
                    }
                }
            }
            return result;
        }

        /// <summary>
        /// tearing用の依存関係消去処理
        /// 依存関係がないところを指定するとエラー
        /// </summary>
        /// <param name="x">横位置 (0～) </param>
        /// <param name="y">縦位置 (0～) </param>
        /// <returns>true:消去成功 false:消去失敗</returns>
        public bool removeDependency(int x, int y)
        {
            if (dsm_[x, y] == 1)
            {
                dsm_[x, y] = 0;
                return true;
            }
            else
            {
                return false;
            }
        }
    }
}
