﻿// In the original BSD license, both occurrences of the phrase "COPYRIGHT HOLDERS AND CONTRIBUTORS"
// in the disclaimer read "REGENTS AND CONTRIBUTORS".
//
// Here is the license template:
//
// Copyright (c) 2010, Masanori Usami
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without modification, are permitted provided
// that the following conditions are met:
//
//  * Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
//  * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer
//    in the documentation and/or other materials provided with the distribution.
//  * Neither the name of the <ORGANIZATION> nor the names of its contributors may be used to endorse or promote products derived
//    from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
// BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
// IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, 
// OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
// OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
// OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF 
// THE POSSIBILITY OF SUCH DAMAGE.

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Collections;
using System.IO;

namespace EAAddinDSM
{

    /// <summary>
    /// UMLモデルのDSMを取得するためのクラス
    /// Enterprise Architect Add-In用
    /// </summary>
    public class DependencyStructureMatrix
    {
        private const string parentMenuNames_ = "-UML DSM";
        private const string childMenuItem1_ = "ダイアグラムのDSM出力（クラス図）";
        private const string childMenuItem2_ = "パッケージのDSM出力（クラス図）";
        private const string childMenuItem3_ = "ダイアグラムのDSM出力（オブジェクト図）";
        private const string childMenuItem4_ = "パッケージのDSM出力（オブジェクト図）";
        private string[] childMenuMenu_ = { childMenuItem1_, childMenuItem2_, childMenuItem3_, childMenuItem4_ };
        private string[] childMenuDiagram_ = { childMenuItem1_, childMenuItem3_ };
        private string[] childMenuTree_ = { childMenuItem2_, childMenuItem4_ };
        private string elementType_ = "Class";

        private DSMHelper helper_;

        /// <summary>
        /// コンストラクタ
        /// helper関数の準備をする
        /// </summary>
        public DependencyStructureMatrix()
        {
            helper_ = new DSMHelper();
        }

        /// <summary>
        /// メニューの設定
        /// </summary>
        /// <param name="repository">EAリポジトリオブジェクト</param>
        /// <param name="menuLocation">メニューが呼ばれた位置 "TreeView"/"MainMenu"/"Diagram"</param>
        /// <param name="menuName">親メニューの文字列</param>
        /// <returns></returns>
        public object EA_GetMenuItems(EA.Repository repository, string menuLocation, string menuName)
        {
            switch (menuName)
            {
                case "":
                    return parentMenuNames_;
                case parentMenuNames_:
                    switch (menuLocation)
                    {
                        case "MainMenu":
                            return childMenuMenu_;
                        case "Diagram":     // 子メニュー ダイアグラムから選択された場合
                            return childMenuDiagram_;
                        case "TreeView":    // 子メニュー ツリービューから呼ばれた場合
                            return childMenuTree_;
                    }
                    return null;
                default:
                    return null;
            }
        }

        /// <summary>
        /// メニュークリック時の処理
        /// </summary>
        /// <param name="repository">EAリポジトリオブジェクト</param>
        /// <param name="menuLocation">メニューが呼ばれた位置 "TreeView"/"MainMenu"/"Diagram"</param>
        /// <param name="menuName">親メニューの文字列</param>
        /// <param name="itemName">選択されたメニューの文字列</param>
        public void EA_MenuClick(EA.Repository repository, string menuLocation, string menuName, string itemName)
        {
            EA.Diagram diagram = repository.GetCurrentDiagram();
            EA.Package package = repository.GetTreeSelectedPackage();
            if ((itemName == childMenuItem1_ || itemName == childMenuItem3_) && (diagram == null))
            {
                MessageBox.Show("ダイアグラムが開かれていません");
                return;
            }
            if ((itemName == childMenuItem2_ || itemName == childMenuItem4_) && (package == null))
            {
                MessageBox.Show("パッケージが選択されていません");
                return;
            }
            switch (itemName)
            {
                case childMenuItem1_:
                    elementType_ = "Class";
                    makeDiagramDSM(repository, diagram);
                    break;
                case childMenuItem2_:
                    elementType_ = "Class";
                    makePackageDSM(repository, package);
                    break;
                case childMenuItem3_:
                    elementType_ = "Object";
                    makeDiagramDSM(repository, diagram);
                    break;
                case childMenuItem4_:
                    elementType_ = "Object";
                    makePackageDSM(repository, package);
                    break;
                default:
                    break;
            }
        }

        /// <summary>
        /// 選択されているダイアグラムのDSMファイルを出力する
        /// </summary>
        /// <param name="repository">EAリポジトリオブジェクト</param>
        /// <param name="diagram">現在表示されているダイアグラムオブジェクト</param>
        protected void makeDiagramDSM(EA.Repository repository, EA.Diagram diagram)
        {
            Hashtable dsmHash = new Hashtable();
            ArrayList elemList = new ArrayList();
            foreach (EA.DiagramObject dgmObject in diagram.DiagramObjects)
            {
                EA.Element source = repository.GetElementByID(dgmObject.ElementID);
                if (source.Type == elementType_)
                {
                    elemList.Add(source.ElementID);
                    ArrayList dest = getDependencyList(repository, source);
                    dsmHash[source.ElementID] = dest;
                }
            }
            int[,] dsm = makeDSMArray(repository, dsmHash, elemList);
            ArrayList newSeq = helper_.partitioning(dsm);
            int[,] partitionedDSM = new int[dsm.GetLength(0), dsm.GetLength(0)];
            ArrayList newElemList = (ArrayList)elemList.Clone();
            for (int i = 0; i < dsm.GetLength(0); i++)
            {
                for (int j = 0; j < dsm.GetLength(0); j++)
                {
                    partitionedDSM[i, j] = dsm[(int)newSeq[i], (int)newSeq[j]];
                }
                newElemList[i] = elemList[(int)newSeq[i]];
            }
            //showDSM(repository, partitionedDSM, newElemList);
            writeDSM(repository, partitionedDSM, newElemList);
        }

        /// <summary>
        /// 選択されているパッケージのDSMファイルを出力する
        /// </summary>
        /// <param name="repository">EAリポジトリオブジェクト</param>
        /// <param name="package">現在選択されているパッケージオブジェクト</param>
        protected void makePackageDSM(EA.Repository repository, EA.Package package)
        {
            Hashtable dsmHash = new Hashtable();
            ArrayList elemList = new ArrayList();
            foreach (EA.Element source in getAllPackageElements(package))
            {
                elemList.Add(source.ElementID);
                dsmHash[source.ElementID] = getDependencyList(repository, source);
            }
            int[,] dsm = makeDSMArray(repository, dsmHash, elemList);
            ArrayList newSeq = helper_.partitioning(dsm);
            int[,] partitionedDSM = new int[dsm.GetLength(0), dsm.GetLength(0)];
            ArrayList newElemList = (ArrayList)elemList.Clone();
            for (int i = 0; i < dsm.GetLength(0); i++)
            {
                for (int j = 0; j < dsm.GetLength(0); j++)
                {
                    partitionedDSM[i, j] = dsm[(int)newSeq[i], (int)newSeq[j]];
                }
                newElemList[i] = elemList[(int)newSeq[i]];
            }
            //showDSM(repository, partitionedDSM, newElemList);
            writeDSM(repository, partitionedDSM, newElemList);
        }

        /// <summary>
        /// 指定されたパッケージ中の要素を全て再帰的に取得する
        /// パッケージが再起構造の場合は、再帰的に処理する
        /// 要素はelementType_で指定されたタイプに一致するものだけ取得する
        /// </summary>
        /// <param name="package">要素を取得するパッケージ</param>
        /// <returns>要素のリスト</returns>
        protected ArrayList getAllPackageElements(EA.Package package)
        {
            ArrayList elemList = new ArrayList();
            foreach (EA.Element source in package.Elements)
            {
                if (source.Type == elementType_)
                {
                    elemList.Add(source);
                }
            }
            foreach (EA.Package pack in package.Packages)
            {
                foreach (EA.Element source in getAllPackageElements(pack))
                {
                    elemList.Add(source);
                }
            }
            return elemList;
        }

        /// <summary>
        /// source要素が依存する要素のリストを返す
        /// </summary>
        /// <param name="repository">EAリポジトリオブジェクト</param>
        /// <param name="source">対象の要素</param>
        /// <returns>source要素が依存する要素のリスト</returns>
        protected ArrayList getDependencyList(EA.Repository repository, EA.Element source)
        {
            ArrayList array = new ArrayList();
            foreach (EA.Connector conn in source.Connectors)
            {
                EA.Element dest;
                if (conn.SupplierID == source.ElementID)
                {
                    dest = repository.GetElementByID(conn.ClientID);
                }
                else
                {
                    dest = repository.GetElementByID(conn.SupplierID);
                }
                if (dest.Type == elementType_)
                {
                    // パッケージ中のエレメントの接続先をチェック
                    if (checkDirection(source, dest, conn))
                    {
                        array.Add(dest.ElementID);
                    }
                }
            }
            return array;
        }

        /// <summary>
        /// SOURCEがDESTINATIONに依存しているかどうかのチェック
        /// DESITINATIONからSOURCEの場合は依存していないとみなす
        /// </summary>
        /// <param name="source">チェック対象の要素</param>
        /// <param name="dest">SOURCEが依存しているかどうかの調査対象の要素</param>
        /// <param name="conn">SOURCEが依存しているかどうかの調査対象の接続</param>
        /// <returns>依存しているときtrue / 依存していないときfalse</returns>
        protected bool checkDirection(EA.Element source, EA.Element dest, EA.Connector conn)
        {
            switch (conn.Type)
            {
                case "Aggregation":
                    return checkAggregationDependency(source, dest, conn);
                //case "Association":
                //    break;
                //case "Dependency":
                //    break;
                //case "Generalization":
                //    break;
                //case "Nesting":
                //    break;
                //case "NoteLink":
                //    break;
                //case "Realization":
                //    break;
                //case "Sequence":
                //    break;
                //case "Transition":
                //    break;
                //case "UseCase":
                //    break;
            }
            switch (conn.Direction)
            {
                case "Unspecified":
                    if ((conn.SupplierID == source.ElementID) && (conn.ClientID == dest.ElementID))
                    {
                        return true;
                    }
                    break;
                case "Bi-Directional":
                    return true;
                case "Source -> Destination":
                    if ((conn.SupplierID == source.ElementID) && (conn.ClientID == dest.ElementID))
                    {
                        return true;
                    }
                    break;
                case "Destination -> Source":
                    if ((conn.SupplierID == dest.ElementID) && (conn.ClientID == source.ElementID))
                    {
                        return true;
                    }
                    break;
            }
            return false;
        }

        /// <summary>
        /// SOURCEがDESTINATIONに依存しているかどうかのチェック
        /// AssociationのみDestinationからSourceに向かって依存する
        /// </summary>
        /// <param name="source">チェック対象の要素</param>
        /// <param name="dest">SOURCEが依存しているかどうかの調査対象の要素</param>
        /// <param name="conn">SOURCEが依存しているかどうかの調査対象の接続</param>
        /// <returns>依存しているときtrue / 依存していないときfalse</returns>
        protected bool checkAggregationDependency(EA.Element source, EA.Element dest, EA.Connector conn)
        {
            if ((conn.SupplierID == dest.ElementID) && (conn.ClientID == source.ElementID))
            {
                return true;
            }
            return false;
        }

        /// <summary>
        /// HashTableから2次元配列のDSM配列を作成する
        /// </summary>
        /// <param name="repository">EAリポジトリオブジェクト</param>
        /// <param name="dsm">DSMを含むHashTable</param>
        /// <param name="objList">エレメントIDのリスト</param>
        /// <returns>DSMの2次元配列</returns>
        protected int[,] makeDSMArray(EA.Repository repository, Hashtable dsm, ArrayList objList)
        {
            int[,] result = new int[objList.Count, objList.Count];

            for (short i = 0; i < objList.Count; i++)
            {
                for (short j = 0; j < objList.Count; j++)
                {
                    if (i == j)
                    {
                        result[i, j] = 1;
                    }
                    else{
                        int iElementID = (int)objList[i];
                        if (dsm.ContainsKey(iElementID))
                        {
                            if (((ArrayList)dsm[iElementID]).Contains((int)objList[j]))
                            {
                                result[i, j] = 1;
                            }
                            else
                            {
                                result[i, j] = 0;
                            }
                        }
                    }
                }
            }
            return result;
        }

        private void showDSM(EA.Repository repository, int[,] dsm, ArrayList objList)
        {
            ArrayList nameList = new ArrayList();
            for (short i = 1; i <= objList.Count; i++)
            {
                EA.Element iElement = repository.GetElementByID((int)objList[i - 1]);
                nameList.Add(iElement.Name);
            }

            DSMDialog dialog = new DSMDialog();
            dialog.setDSM(dsm, nameList);

            dialog.ShowDialog();
        }

        /// <summary>
        /// DSMをCSVファイルに書き出す
        /// </summary>
        /// <param name="repository">EAリポジトリオブジェクト</param>
        /// <param name="dsm">DSMの2次元配列</param>
        /// <param name="objList">エレメントIDのリスト</param>
        private void writeDSM(EA.Repository repository, int[,] dsm, ArrayList objList)
        {
            string path = getSaveFileName();
            if (path == null)
            {
                MessageBox.Show("保存先ファイルが指定されませんでした。\nDSMは保存されません。");
                return;
            }
            StreamWriter dsmWriter = new StreamWriter(path);
            string line = "Name,";

            for (short count = 1; count <= objList.Count; count++)
            {
                line += "," + count.ToString();
            }
            dsmWriter.WriteLine(line);

            for (short i = 1; i <= objList.Count; i++)
            {
                EA.Element iElement = repository.GetElementByID((int)objList[i - 1]);
                line = iElement.Name + "," + i.ToString();
                for (short j = 1; j <= objList.Count; j++)
                {
                    if (i == j)
                    {
                        line += "," + i.ToString();
                    }
                    else if (dsm[i - 1, j - 1] == 1)
                    {
                        line += "," + "1";
                    }
                    else
                    {
                        line += "," + "0";
                    }
                }
                dsmWriter.WriteLine(line);
            }

            dsmWriter.Close();
            MessageBox.Show("ファイル:" + path + "にDSMを出力しました。");
        }

        /// <summary>
        /// SaveFileDialogを利用して保存ファイル名を取得
        /// </summary>
        /// <returns></returns>
        protected string getSaveFileName()
        {
            string path = null;
            SaveFileDialog saveFileDialog = new SaveFileDialog();

            saveFileDialog.Filter = "txt files (*.csv)|*.txt|All files (*.*)|*.*";
            saveFileDialog.FilterIndex = 2;
            saveFileDialog.RestoreDirectory = true;

            if (saveFileDialog.ShowDialog() == DialogResult.OK)
            {
                path = saveFileDialog.FileName;
                return path;
            }
            return null;
        }

    }
}
