﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Collections;
using System.IO;

namespace EAAddinDSM
{
    class DSMController : Observer
    {
        private DSMModel dsm_;

        private DSMController()
        {

        }

        public DSMController(ref DSMModel dsm)
        {
            dsm_ = dsm;
        }

        public bool initialize()
        {
            try
            {
                return dsm_.attach(this);
            }
            catch (System.NotSupportedException e)
            {
                throw e;
            }
        }

        public bool partitioning()
        {
            int[,] dsm = dsm_.getDSM();
            int[,] reach = reachabilityMatrix(dsm);
            ArrayList[] row = partitionRow(reach);
            ArrayList[] column = partitionColumn(reach);
            dsm_.setNewSequence(getNewSequence(row, column));

            return true;
        }

        /// <summary>
        /// 与えられたDSM配列からReachability Matrixを作る
        /// </summary>
        /// <param name="dsm">DSM配列</param>
        /// <returns>Reachability Matrix</returns>
        public int[,] reachabilityMatrix(int[,] dsm)
        {
            // 対角には１を設定
            for (int i = 0; i < dsm.GetLength(0); i++)
            {
                dsm[i, i] = 1;
            }
            // Reachability Matrixの計算
            int[,] reach = (int[,])dsm.Clone();
            for (int i = 0; i < dsm.GetLength(0); i++)
            {
                int[,] temp = (int[,])reach.Clone();
                for (int j = 0; j < dsm.GetLength(0); j++)
                {
                    for (int m = 0; m < dsm.GetLength(0); m++)
                    {
                        for (int k = 0; k < dsm.GetLength(0); k++)
                        {
                            reach[j, m] = temp[j, k] * dsm[k, m] + reach[j, m];
                            if (reach[j, m] > 1)
                            {
                                reach[j, m] = 1;
                            }
                        }
                    }
                }
            }
            return reach;
        }

        /// <summary>
        /// Reachability Matrixの行成分をArrayListにして返す
        /// </summary>
        /// <param name="reach">Reachability Matrix（2次元配列）</param>
        /// <returns>Reachability Matrixの行成分の配列</returns>
        private ArrayList[] partitionRow(int[,] reach)
        {
            ArrayList[] row = new ArrayList[reach.GetLength(0)];
            for (int i = 0; i < reach.GetLength(0); i++)
            {
                row[i] = new ArrayList();
                for (int j = 0; j < reach.GetLength(0); j++)
                {
                    if (reach[i, j] != 0)
                    {
                        row[i].Add(j);
                    }
                }
            }
            return row;
        }

        /// <summary>
        /// Reachability Matrixの列成分をArrayListにして返す
        /// </summary>
        /// <param name="reach">Reachability Matrix（2次元配列）</param>
        /// <returns>Reachability Matrixの列成分の配列</returns>
        private ArrayList[] partitionColumn(int[,] reach)
        {
            ArrayList[] column = new ArrayList[reach.GetLength(0)];
            for (int i = 0; i < reach.GetLength(0); i++)
            {
                column[i] = new ArrayList();
                for (int j = 0; j < reach.GetLength(0); j++)
                {
                    if (reach[j, i] != 0)
                    {
                        column[i].Add(j);
                    }
                }
            }
            return column;

        }

        /// <summary>
        /// 与えられたReachability Matrixから新しい要素の並びを計算する
        /// </summary>
        private ArrayList getNewSequence(ArrayList[] row, ArrayList[] column)
        {
            int[] taken = new int[row.GetLength(0)];
            for (int i = 0; i < taken.GetLength(0); i++)
            {
                taken[i] = 0;
            }

            ArrayList newSeq = new ArrayList();
            for (; newSeq.Count < row.GetLength(0); )
            {
                for (int i = 0; i < row.GetLength(0); i++)
                {
                    if (checkRow(taken, row[i], column[i]))
                    {
                        foreach (int val in row[i])
                        {
                            if (taken[val] == 0)
                            {
                                newSeq.Add(val);
                                taken[val] = 2;
                            }
                        }
                    }
                }
                for (int i = 0; i < taken.GetLength(0); i++)
                {
                    if (taken[i] == 2)
                    {
                        taken[i] = 1;
                    }
                }
            }
            return newSeq;
        }

        /// <summary>
        /// 処理対象の要素（Row）かどうかのチェック
        ///  未処理の要素に参照されていれば処理対象外
        ///  相互参照の要素と処理済の要素にのみ参照されていれば処理対象
        /// </summary>
        /// <param name="taken">処理済要素を示す配列</param>
        /// <param name="row">チェック対象の行</param>
        /// <param name="column"></param>
        /// <returns>true:処理対象 false:処理対象外</returns>
        private bool checkRow(int[] taken, ArrayList row, ArrayList column)
        {
            bool signal = false;
            foreach (int val in row)
            {
                if (taken[val] != 1)
                {
                    signal = column.Contains(val);
                }
                else if (taken[val] == 1)
                {
                    signal = true;
                }
                if (!signal)
                {
                    break;
                }
            }
            return signal;
        }

        /// <summary>
        /// DSMをCSVファイルに書き出す
        /// </summary>
        /// <param name="repository">EAリポジトリオブジェクト</param>
        /// <param name="dsm">DSMの2次元配列</param>
        /// <param name="objList">エレメントIDのリスト</param>
        public void save(string path)
        {
            ArrayList nameList = dsm_.getElementNameList();
            int[,] dsm = dsm_.getDSM();
            StreamWriter dsmWriter = new StreamWriter(path);
            string line = "Name,";

            for (short count = 1; count <= nameList.Count; count++)
            {
                line += "," + count.ToString();
            }
            dsmWriter.WriteLine(line);

            for (short i = 1; i <= nameList.Count; i++)
            {
                line = (string)nameList[i - 1] + "," + i.ToString();
                for (short j = 1; j <= nameList.Count; j++)
                {
                    if (i == j)
                    {
                        line += "," + i.ToString();
                    }
                    else if (dsm[i - 1, j - 1] == 1)
                    {
                        line += "," + "1";
                    }
                    else
                    {
                        line += "," + "0";
                    }
                }
                dsmWriter.WriteLine(line);
            }

            dsmWriter.Close();
        }


        public void update()
        {
        }

    }
}
