#include "EditorWidget.h"

#include "SequenceModel.h"
#include "VsqImporter.h"
#include "TrackView.h"
#include "ControlArea.h"

#include "WaveBuffer.h"
#include "dsp/world.h"

#include "task/Manager.h"
#include "helper/WaveAnalysisTask.h"
#include "ViewSettings.h"


#include <QUndoStack>
#include <QFileDialog>
#include <QMessageBox>

#include <QThread>

using namespace stand::gui;

void EditorWidget::undo()
{
    if(_taskManager)
    {
        return;
    }
    if(_model && _model->undoStack())
    {
        _model->undoStack()->undo();
    }
    _isSaved = false;
    _framePeriod = 2.0;
}

void EditorWidget::redo()
{
    if(_taskManager)
    {
        return;
    }
    if(_model && _model->undoStack())
    {
        _model->undoStack()->redo();
    }
}

void EditorWidget::save()
{
    if(_currentFilePath == QString::null)
    {
        saveas();
    }
    else if(_model)
    {
        _model->write(_currentFilePath);
        _isSaved = true;
    }
    else
    {
        qFatal("EditorWidget::save(); // No set model.");
    }
}

void EditorWidget::saveas()
{
    QString filepath = QFileDialog::getSaveFileName(this, tr("Save as"), QDir::current().path(), tr("Stand Xml File (*.stfx)"));
    if(filepath != QString::null && _model)
    {
        if(_model->write(filepath))
        {
            _currentFilePath = filepath;
            _isSaved = true;
        }
        else
        {
            QMessageBox::critical(this, tr("Error"), tr("Could not save file; ") + filepath);
        }
    }
}

bool EditorWidget::confirmClearCurrent()
{
    // ToDo :: make sure if user minds opening file without saving.
    if(!_model->undoStack()->isClean())
    {
        int ret = QMessageBox::question(this, tr("Confirm"), tr("Current sequence is modified.\nDo you want to save changes; ") + _currentFilePath,
                                        QMessageBox::Yes, QMessageBox::No, QMessageBox::Cancel);
        _isSaved = false;

        switch(ret)
        {
        case QMessageBox::Yes:
            save();
            if(!_isSaved)
            {
                return false;
            }
        case QMessageBox::No:
            return true;
        case QMessageBox::Cancel:
            return false;
        }
    }
    return true;
}

void EditorWidget::newfile()
{
    if(_taskManager || !confirmClearCurrent())
    {
        return;
    }
    stand::model::SequenceModel *model = new stand::model::SequenceModel(this);
    _resetModel(model);
}

void EditorWidget::open()
{
    if(_taskManager || !confirmClearCurrent())
    {
        return;
    }
    stand::model::SequenceModel *model = new stand::model::SequenceModel(this);
    QString filepath = QFileDialog::getOpenFileName(this, tr("Open File"), QDir::current().path(), tr("Stand file Xml (*.stfx)"));
    if(model->read(filepath))
    {
        _resetModel(model);
    }
}

void EditorWidget::openVsqFile()
{
    if(_taskManager || !confirmClearCurrent())
    {
        return;
    }
    QString filepath = QFileDialog::getOpenFileName(this, tr("Open Vsq File"), QDir::current().path(), tr("Vocaloid MIDI (*.vsq | *.mid)"));
    if(filepath != QString::null)
    {
        stand::sequence::importer::VsqImporter vsq;
        stand::model::SequenceModel *model = new stand::model::SequenceModel(this);
        if(!vsq.open(filepath))
        {
            QMessageBox::critical(this, tr("File Open Error!"), filepath + tr(" may not be vsq file."));
            return;
        }
        model->import(&vsq);
        _resetModel(model);
    }
}

void EditorWidget::_resetModel(stand::model::SequenceModel *model)
{
    _trackView->setModel(model);

    delete _model;
    _model = model;
}

void EditorWidget::reactivate()
{
    this->setEnabled(true);
    delete _taskManager;
    _taskManager = NULL;
    _trackView->settings()->setTrackId(_trackView->settings()->trackId());
}

void EditorWidget::importRawWave()
{
    using namespace stand::utility::task;
    QString filepath = QFileDialog::getOpenFileName(this, tr("Open Wave File"), QDir::current().path(), tr("Singing Wave File (*.wav)"));
    if(filepath == QString::null)
    {
        return;
    }

    this->setEnabled(false);
    helper::WaveAnalysisTask *analysis = new helper::WaveAnalysisTask(_model, _trackView->settings()->tracks()[_trackView->settings()->trackId()], filepath, _framePeriod, this);
    QList<stand::utility::task::Task *> l;
    l.append(analysis);

    _taskManager = new stand::utility::task::Manager(l, 1, true, this);
    connect(_taskManager, SIGNAL(finished()), this, SLOT(reactivate()));
    _taskManager->start();
}
