#ifndef GMMSET_H
#define GMMSET_H

namespace stand
{
namespace math
{

class NormalDistribution;
class GmmData;

class GmmSet
{
public:
    const static int DefaultMixtureNumber = 4;
    const static double DefaultEpsilon;
    const static int DefaultMaxRepeat = 1000;

    explicit GmmSet(int numMix, int dim);
    virtual ~GmmSet();

    /*!
     *  @brief 与えられたデータ列から GMM のパラメタを推定します．
     *  @param[in] x データ列．
     *  @param[in] length データの長さ（＝個数）．
     *  @param[in] dim データの次元数．
     */
    virtual void estimate(double **x, int length, int dim);

    virtual void estimate(GmmData *data, int length);

    /*!
     *  @brief 混合数の設定を変更します．このメソッドが行う操作は保持しているデータを破壊します．
     *  @param[in] numMix 指定する混合数．
     */
    void setNumMix(int numMix, int dim);

    /*!
     *  @brief 与えられたデータが GMM から出力される確率を返します．
     *  @param[in] x ベクトルデータ
     *  @param[in] dim データの次元数．セーフガード．
     *  @return 与えられたベクトルに対する確率．
     *  @retval -1.0 GMM の次元数とベクトルの次元が異なる場合 -1 を返します．
     */
    double f(const double *x, int dim);

    bool write(const char *path);

    static double logSumExp(double *array, int length);

    int numMix()
    {
        return _numMix;
    }

private:
    void _destroy();
    void _create(int numMix, int dim);
    void _setInitialValues(double **x, int length, double **mus, double ***sigmas);

    int _numMix;
    int _dim;
    double *_pi;
    NormalDistribution *_gaussians;
};

}
}

#endif // GMMSET_H
