//-----------------------------------------------------------------------------
// Spectral envelope estimation based on STAR (Synchronous Technique and Adroit
// Restoration).
// Please see styleguide.txt to show special rules on names of variables
// and fnctions.
//-----------------------------------------------------------------------------
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "world.h"

namespace stand
{
namespace math
{
namespace dsp
{

#define MIN(a,b) (((a)<(b))?(a):(b))
#define MAX(a,b) (((a)<(b))?(b):(a))


//-----------------------------------------------------------------------------
// GetFFTSizeForStar() calculates the FFT size based on the sampling frequency
// and the lower limit of f0 (It is defined in world.h).
// Input: 
//   fs      : Sampling frequency
// Output: 
//   FFT size
//-----------------------------------------------------------------------------
int GetFFTSizeForStar(int fs)
{
  return (int)pow(2.0, 1.0+(int)(log(3.0*fs/FLOOR_F0+1) / log(2.0)));
}

//-----------------------------------------------------------------------------
// AdroitSmoothing() carries out the smoothing by rectangular window.
// This function is only used in StarGeneralBody().
// Input:
//   current_f0     : f0 at the current_f0 sec.
//   fs             : Sampling frequency
//   fft_size       : FFT size
//   power_spectrum : Power spectrum of the windowed signal
// Output:
//   star_spectrum  : Estimated spectral envelope by STAR
//-----------------------------------------------------------------------------
void AdroitSmoothing(double current_f0, int fs, int fft_size, 
  double *power_spectrum, 
  double *star_spectrum)
{
  int boundary = (int)(current_f0 / (fs/(double)fft_size))+1;
  double *mirroring_spectrum = 
    (double *)malloc(sizeof(double) * (fft_size+boundary*2 + 1) );

  int i;
  for(i = 0;i < boundary;i++) 
    mirroring_spectrum[i] = power_spectrum[boundary-i];
  for(int j = 0;i < fft_size/2+boundary;i++,j++) 
    mirroring_spectrum[i] = power_spectrum[j];
  for(int j = 1;i < fft_size/2+boundary*2+1;i++,j++) 
    mirroring_spectrum[i] = power_spectrum[fft_size/2-j];

  int tmp = (int)(current_f0*fft_size/(double)fs);

  double *mirroring_segment = (double *)malloc(sizeof(double) * fft_size*2);
  mirroring_segment[0] = 
    log(mirroring_spectrum[0])*(double)fs/(double)fft_size;
  for(int i = 1;i < fft_size/2+boundary*2+1;i++)
    mirroring_segment[i] = log(mirroring_spectrum[i])*(double)fs / 
    (double)fft_size + mirroring_segment[i-1];

  double *frequency_axis = 
    (double *)malloc(sizeof(double) * (fft_size/2 + 1) );
  for(int i = 0;i <= fft_size/2;i++)
    frequency_axis[i] = (double)i / (double)fft_size * 
    (double)fs - current_f0/2.0;

  double *low_levels  = (double *)malloc(sizeof(double) * (fft_size/2+1));
  double *high_levels = (double *)malloc(sizeof(double) * (fft_size/2+1));
  double origin_of_mirroring_axis = 
    -((double)boundary-0.5)*(double)fs/(double)fft_size;
  double discrete_frequency_interval = (double)fs/(double)fft_size;

  interp1Q(origin_of_mirroring_axis, discrete_frequency_interval, 
    mirroring_segment, fft_size/2+boundary*2+1, frequency_axis, 
    (fft_size/2 + 1), low_levels);
  for(int i = 0;i <= fft_size/2;i++) frequency_axis[i] += current_f0;

  interp1Q(origin_of_mirroring_axis, discrete_frequency_interval, 
    mirroring_segment, fft_size/2+boundary*2+1, frequency_axis, 
    (fft_size/2 + 1), high_levels);

  for(int i = 0;i <= fft_size/2;i++)
    star_spectrum[i] = exp( (high_levels[i]-low_levels[i])/current_f0);

  free(low_levels); 
  free(high_levels);
  free(mirroring_segment); 
  free(frequency_axis);
  free(mirroring_spectrum);
}

//-----------------------------------------------------------------------------
// GetPowerSpectrum() carries out (1) designing the window,
// (2) windowing the waveform and (3) calculation of the power_spectrum
// Input:
//   x                  : Input signal
//   x_length           : Length of x
//   fs                 : Sampling Frequency
//   current_f0         : f0 at the current_f0 sec.
//   temporal_position  : Position used for the estimation
//   forward_real_fft   : Struct for FFT. 
// Output:
//   power_spectrum     : Calculated power_spectrum
//-----------------------------------------------------------------------------
void GetPowerSpectrum(const double *x, int x_length, int fs, double current_f0,
  double temporal_position, ForwardRealFFT *forward_real_fft,
  double *power_spectrum)
{
  int half_window_length = (int)(0.5 + 3.0*(double)fs/current_f0/2.0);
  int *base_index = (int *)malloc(sizeof(int) * (half_window_length*2+1));
  int *index  = (int *)malloc(sizeof(int) * (half_window_length*2+1));

  for(int i = -half_window_length;i <= half_window_length;i++) 
    base_index[i+half_window_length] = i;
  for(int i = 0;i <= half_window_length*2;i++) 
    index[i]  = MIN(x_length, MAX(1, matlab_round(temporal_position * 
    (double)fs+1+base_index[i]) ) ) - 1;

  // Designing of the window function
  double *window  = 
    (double *)malloc(sizeof(double) * (half_window_length*2+1));
  double average = 0.0;
  double position;
  for(int i = 0;i <= half_window_length*2;i++)
  {
    position  = (double)(base_index[i]/(double)fs/(3.0/2.0) ) + 
      (temporal_position*(double)fs - 
      (double)(matlab_round(temporal_position*(double)fs))) / 
      (double)fs;
    window[i]  = 0.5*cos(PI*position*current_f0) +0.5;
    average  += window[i]*window[i];
  }
  average  = sqrt(average);
  for(int i = 0;i <= half_window_length*2;i++) 
    window[i]  /= average;

  // Windowing and FFT
  for(int i = 0;i <= half_window_length*2;i++) 
    forward_real_fft->waveform[i] = x[index[i]] * window[i];
  for(int i = half_window_length*2+1;i < forward_real_fft->fft_size;i++) 
    forward_real_fft->waveform[i] = 0.0;
  fft_execute(forward_real_fft->forward_fft);

  // Calculation of the power spectrum.
  for(int i = 1;i <= forward_real_fft->fft_size/2;i++) 
    power_spectrum[i] = forward_real_fft->spectrum[i][0]*forward_real_fft->spectrum[i][0] + 
    forward_real_fft->spectrum[i][1]*forward_real_fft->spectrum[i][1];
  power_spectrum[0] = power_spectrum[1];

  free(window);
  free(base_index); 
  free(index);
}

//-----------------------------------------------------------------------------
// StarGeneralBody() calculates a spectral envelope at a temporal position.
// This function is only used in Star().
// Input:
//   x                  : Input signal
//   x_length           : Length of x
//   fs                 : Sampling Frequency
//   current_f0         : f0 at the current_f0 sec.
//   temporal_position  : Position used for the estimation
//   fft_size           : FFT size
//   windowed_waveform  : Array arranged in advance
//   y_spectrum         : Array arranged in advance
//   forward_fft        : fft_plan arranged in advance
// Output:
//   star_spectrum      : Estimated STAR spectrum
// Caution:
//   windowed_waveform, y_spectrum and forward_fft is allocated in advance in
//   Star() to speed up the processing. If you want to develop real-time 
//   application, you should modify this function not to use these arguments
//   and edit this function.
//-----------------------------------------------------------------------------
void StarGeneralBody(const double *x, int x_length, int fs, double current_f0,
  double temporal_position, ForwardRealFFT *forward_real_fft, 
  double * star_spectrum)
{
  double *power_spectrum = 
    (double *)malloc(sizeof(double) * forward_real_fft->fft_size);

  // Synchronous analysis
  GetPowerSpectrum(x, x_length, fs, current_f0, temporal_position,
    forward_real_fft, power_spectrum);

  // Adroit smoothing
  AdroitSmoothing(current_f0, fs, forward_real_fft->fft_size, 
    power_spectrum, star_spectrum);

  free(power_spectrum);
}

//-----------------------------------------------------------------------------
// Star() calculates the spectrogram that consists of spectral envelopes 
// estimated by STAR.
// Input:
//   x            : Input signal
//   xLen         : Length of x
//   fs           : Sampling frequency
//   timeAxis     : Time axis
//   f0           : F0 contour
// Output:
//   spectrogram  : Spectrogram estimated by STAR.
//-----------------------------------------------------------------------------
void Star(const double *x, int x_length, int fs, double *time_axis, double *f0,
     double **spectrogram)
{
  double frame_period = (time_axis[1]-time_axis[0])*1000.0;

  int  fft_size = (int)pow(2.0, 1.0+(int)(log(3.0*fs/FLOOR_F0+1) / log(2.0)));
  int f0_length = GetSamplesForDIO(fs, x_length, frame_period);

  double *star_spectrum = (double *)malloc(sizeof(double) * fft_size);

  // Following three variables are shared in StarGeneralBody()
  ForwardRealFFT forward_real_fft = {0};
  InitializeForwardRealFFT(fft_size, &forward_real_fft);

  double current_f0;
  for(int i = 0;i < f0_length;i++)
  {
    current_f0 = f0[i] <= FLOOR_F0 ? DEFAULT_F0 : f0[i];
    StarGeneralBody(x, x_length, fs, current_f0, time_axis[i],  
      &forward_real_fft, star_spectrum);

    for(int j = 0;j <= fft_size/2;j++) spectrogram[i][j] = star_spectrum[j];
  }

  DestroyForwardRealFFT(&forward_real_fft);
  free(star_spectrum);
}

}
}
}
