#include "MeasureItem.h"

#include "../configure.h"
#include "Measure.h"

#include <QPoint>

using namespace stand::model;

MeasureItem::MeasureItem(const QString &name, int numerator, int denominator, TreeItem *parent) :
    TreeItem(name, parent)
{
    _length = 0;
    if(numerator <= 0 || denominator <= 0 || (denominator & (denominator - 1)))
    {
        numerator = denominator = 4;
    }
    QPoint p(numerator, denominator);
    // 十分小さい ∧ アンダーフローしない 値ならよい
    int measure = INT_MIN / stand::sequence::DefaultTicksPerBeat / 320;
    setData(measure, p, Qt::EditRole);
}

MeasureItem::~MeasureItem()
{
}

void MeasureItem::setMeasure(const stand::sequence::Measure *measure)
{
    if(!measure || measure->data().empty())
    {
        return;
    }

    _contour[0].numerator = measure->at(0).up;
    _contour[0].denominator = measure->at(0).down;

    int currentMeasure = 0;
    int currentTick = 0;
    for(int i = 0; i < measure->data().size(); i++)
    {
        int measureCount = (measure->data()[i].beginTick - currentTick) / (stand::sequence::DefaultTicksPerBeat * 4 * measure->data()[i].up / measure->data()[i].down);
        measureCount += currentMeasure;
        QPoint p(measure->data()[i].up, measure->data()[i].down);
        setData(measureCount, p, 0);
        currentMeasure = measureCount;
        currentTick = measure->data()[i].beginTick;
    }
}

int MeasureItem::columnCount() const
{
    return _length;
}

QVariant MeasureItem::data(int column, int role) const
{
    if(role != Qt::DisplayRole && role != Qt::EditRole)
    {
        return QVariant();
    }
    if(_contour.empty())
    {
        return QVariant();
    }
    QPoint p(_contour.at(0).numerator, _contour.at(0).denominator);
    if(column < 0)
    {
        return p;
    }

    for(QList<Rhythm>::const_iterator it = _contour.begin(); it != _contour.end(); it++)
    {
        if((*it).measure > column)
        {
            return p;
        }
        p = QPoint((*it).numerator, (*it).denominator);
    }
    return QPoint(_contour.last().numerator, _contour.last().denominator);
}

void MeasureItem::setData(int column, const QVariant &d, int role)
{
    if(role != Qt::DisplayRole && role != Qt::EditRole)
    {
        return;
    }
    QPoint pos = d.toPoint();
    Rhythm p = {column, pos.x(), pos.y()};
    for(int i = 0; i < _contour.size(); i++)
    {
        if(_contour[i].measure == column)
        {
            if(i != 0 && _contour[i-1].numerator == pos.x() && _contour[i-1].denominator == pos.y())
            {
                _contour.removeAt(i);
                return;
            }
            _contour[i] = p;
            return;
        }
        if(_contour[i].measure > column)
        {
            if(i != 0 && _contour[i].numerator == pos.x() && _contour[i].denominator == pos.y())
            {
                return;
            }
            _contour.insert(i, p);
            return;
        }
    }
    _length = column;
    if(_contour.empty() || _contour.last().numerator != pos.x() || _contour.last().denominator != pos.y())
    {
        _contour.push_back(p);
    }
}

void MeasureItem::removeColumn(int first, int last)
{
    QList<Rhythm>::iterator it_first, it_last;
    for(it_first = _contour.begin(); it_first != _contour.end(); it_first++)
    {
        if((*it_first).measure >= first)
        {
            break;
        }
    }
    if(it_first == _contour.end())
    {
        return;
    }
    for(it_last = it_first; it_last != _contour.end(); it_last++)
    {
        if((*it_last).measure >= last)
        {
            break;
        }
    }
    _contour.erase(it_first, it_last);
}

QList<MeasureItem::Rhythm> MeasureItem::contour(int first, int last) const
{
    QList<Rhythm> ret;

    QList<Rhythm>::const_iterator it;
    for(it = _contour.begin(); it != _contour.end(); it++)
    {
        if((*it).measure >= first)
        {
            break;
        }
    }
    for(; it != _contour.end(); it++)
    {
        if((*it).measure >= last)
        {
            break;
        }
        ret.push_back((*it));
    }

    return ret;
}

void MeasureItem::setUpDomElement(QDomDocument &doc, QDomElement &element)
{
    element.setTagName("measure");

    if(!_contour.empty())
    {
        element.setAttribute("default_numerator", _contour[0].numerator);
        element.setAttribute("default_denominator", _contour[0].denominator);
    }

    for(int i = 1; i < _contour.size(); i++)
    {
        QDomElement e = doc.createElement("beat");
        e.setAttribute("measure", _contour[i].measure);
        e.setAttribute("numerator", _contour[i].numerator);
        e.setAttribute("denominator", _contour[i].denominator);
        element.appendChild(e);
    }
}

