#include "Track.h"

#include "ControlCurve.h"
#include "NoteEvent.h"
#include "SingerEvent.h"
#include "Sequence.h"
#include "Measure.h"
#include "../configure.h"

#include "envelope/Envelope.h"

#include <QStringList>

using namespace stand::sequence;

QHash<QString, importer::Interpreter *(Track::*)(const QString &)> Track::Factories;

Track::Track(Sequence *parent)
{
    _parent = parent;
    if(Factories.empty())
    {
        Factories["Control"] = &Track::_createControl;
        Factories["Note"] = &Track::_createNote;
        Factories["Singer"] = &Track::_createSinger;
    }
}

Track::~Track()
{
    _destroy();
}

void Track::_destroy()
{
    qDeleteAll(_curves);
    qDeleteAll(_events);
    _curves.clear();
    _events.clear();
}

bool Track::isHead(int m) const
{
    if(!_parent)
    {
        return false;
    }

    int c = _parent->measure()->at(0).up;
    for(; m > 0; m -= c)
    {
        c = _parent->measure()->at(m * stand::sequence::DefaultTicksPerBeat).up;
    }
    return (m == 0);
}

bool Track::isValid() const
{
    int previous = INT_MIN;
    for(int i = 0; i < _events.size(); i++)
    {
        NoteEvent *n = dynamic_cast<NoteEvent *>(_events[i]);
        if(!n)
        {
            continue;
        }
        if(previous > n->tick())
        {
            return false;
        }
        previous = n->tick() + n->length();
    }
    return true;
}

void Track::validate()
{
    int previous = INT_MIN;
    NoteEvent *previousNote;
    for(int i = 0; i < _events.size(); i++)
    {
        NoteEvent *n = dynamic_cast<NoteEvent *>(_events[i]);
        if(!n)
        {
            continue;
        }
        if(previous > n->tick())
        {
            previousNote->setLength(n->tick() - previousNote->tick());
        }
        previous = n->tick() + n->length();
        previousNote = n;
    }
}

int Track::length() const
{
    int retval = 0;
    const NoteEvent *last = NULL;
    if(_events.empty())
    {
        return 0;
    }
    for(int i = _events.size() - 1; i >= 0 && !last; i--)
    {
        last = dynamic_cast<const NoteEvent *>(_events[i]);
    }
    if(last)
    {
        retval = last->tick() + last->length();
    }
    return retval;
}

const QVector<NoteEvent *> Track::notes()
{
    QVector<NoteEvent *> events;
    for(int i = 0; i < _events.size(); i++)
    {
        NoteEvent *e = dynamic_cast<NoteEvent *>(_events[i]);
        if(e)
        {
            events.push_back(e);
        }
    }
    return events;
}

void Track::addNote(NoteEvent *n)
{
    if(!n)
    {
        return;
    }
    for(int i = 0; i < _events.size(); i++)
    {
        if(_events[i]->tick() < n->tick())
        {
            _events.insert(i, n);
            return;
        }
    }
    _events.push_back(n);
}

void Track::removeNote(NoteEvent *n)
{
    if(!n)
    {
        return;
    }
    int i = _events.indexOf(n);
    if(i < 0)
    {
        return;
    }
    _events.removeAt(i);
}

const QString Track::toString() const
{
    QString retval = "";

    for(int i = 0; i < _events.size(); i++)
    {
        retval += name() + " " + _events.at(i)->type() + " " + _events.at(i)->name() +  " " + QString::number(_events.at(i)->tick()) + "\n";
    }
    for(int i = 0; i < _events.size(); i++)
    {
        retval += _events[i]->toString();
    }

    for(int i = 0; i < _curves.size(); i++)
    {
        retval += name() + " Control " + _curves[i]->name() + "\n";
        retval += _curves[i]->name() + "\n";
    }

    return retval;
}

importer::Interpreter *Track::interpret(const QString &line)
{
    QStringList sl = line.split(" ", QString::SkipEmptyParts);
    if(sl.size() < 2)
    {
        return NULL;
    }

    if(Factories.contains(sl.at(0)))
    {
        return (this->*Factories[sl.at(0)])(line);
    }
    return NULL;
}

importer::Interpreter *Track::_createControl(const QString &name)
{
    QStringList sl = name.split(" ", QString::SkipEmptyParts);
    if(sl.size() != 2)
    {
        return NULL;
    }
    for(int i = 0; i < _curves.size(); i++)
    {
        if(name == _curves[i]->name())
        {
            return _curves[i];
        }
    }

    double val = ControlCurve::defaultValue(sl.at(1));
    ControlCurve *c = new ControlCurve(val);
    c->setName(sl.at(1));
    _curves.push_back(c);

    return c;
}

importer::Interpreter *Track::_createNote(const QString &name)
{
    // "Note tick #e0000" の形式で来る，来て．
    QStringList sl = name.split(" ", QString::SkipEmptyParts);
    if(sl.size() != 3)
    {
        return NULL;
    }
    int tick = sl.at(2).toInt();
    NoteEvent *e = new NoteEvent(this, tick);
    e->setType(sl.at(0));
    e->setName(sl.at(1));
    _events.push_back(e);

    return e;
}

importer::Interpreter *Track::_createSinger(const QString &name)
{
    QStringList sl = name.split(" ", QString::SkipEmptyParts);
    if(sl.size() != 3)
    {
        return NULL;
    }
    int tick = sl.at(2).toInt();
    SingerEvent *e = new SingerEvent(this, tick);
    e->setType(sl.at(0));
    e->setName(sl.at(1));
    _events.push_back(e);

    return e;
}

void Track::addControl(ControlCurve *c)
{
    _curves.push_back(c);
}
