#include "VsqImporter.h"

#include "../../io/midi/MidiFile.h"
#include "../../io/midi/MetaEvent.h"
#include "../../io/midi/MidiEvent.h"

#include <QTextStream>

using namespace stand::sequence::importer;
using namespace stand::io::midi;

VsqImporter::VsqImporter() :
    VsqTextImporter(QTextCodec::codecForName("Shift-JIS"))
{
}

VsqImporter::~VsqImporter()
{
}

bool VsqImporter::open(const QString &path)
{
    close();
    stand::io::midi::MidiFile m;
    if(!m.read(path.toLocal8Bit().data()))
    {
        qDebug("VsqImporter::open(%s); // File could not be opened.", path.toUtf8().data());
        return false;
    }

    if(m.empty())
    {
        qDebug("VsqImporter::open(%s); // File is empty!", path.toUtf8().data());
        return false;
    }

    QString text;
    _setConductor(text, m);
    for(int i = 1; i < m.size(); i++)
    {
        _setTrack(text, m, i);
    }

    QTextStream stream(&text);
    setQueue(stream);

    return true;
}

void VsqImporter::_setConductor(QString &text, MidiFile &m)
{
    QString tempo, beat;
    tempo += "[Tempo]\n";
    beat += "[Measure]\n";
    for(int i = 0; i < m.track(0)->size(); i++)
    {
        MetaEvent *meta = dynamic_cast<MetaEvent *>((*m.track(0))[i]);
        if(!meta)
        {
            continue;
        }
        if(meta->type() == 0x2f && meta->length() == 0)
        {
            break;
        }
        if(meta->type() == 0x51 && meta->length() >= 3)
        {
            int microsec = ((unsigned char)(meta->data()[0]) << 16) + ((unsigned char)(meta->data()[1]) << 8) + (unsigned char)(meta->data()[2]);
            double val = 60.0 / (microsec / 1000000.0);
            tempo += QString::number(meta->tick()) + "=" + QString::number(val) + "\n";
        }
        else if(meta->type() == 0x58 && meta->length() >= 4)
        {
            int up = meta->data()[0];
            int down = meta->data()[1];
            beat += QString::number(meta->tick()) + "=" + QString::number(up) + "," + QString::number(down) + "\n";
        }
    }
    text += tempo + beat;
}

void VsqImporter::_setTrack(QString &text, MidiFile &m, int index)
{
    for(int i = 0; i < m.track(index)->size(); i++)
    {
        MetaEvent *meta = dynamic_cast<MetaEvent *>((*m.track(index))[i]);
        if(!meta)
        {
            continue;
        }
        if(meta->type() == 0x2f && meta->length() == 0)
        {
            break;
        }
        if(meta->type() == 0x01 && meta->length() >= 9)
        {
            text += _codec->toUnicode(meta->data() + 8, meta->length() - 8);
        }
    }
}
