#include "VsqTextImporter.h"

#include <QStringList>
#include <QTextCodec>

#include <QFile>
#include <QTextStream>

using namespace stand::sequence::importer;

VsqTextImporter::VsqTextImporter(QTextCodec *codec)
{
    setCodec(codec);
}

VsqTextImporter::~VsqTextImporter()
{
}

void VsqTextImporter::setCodec(QTextCodec *codec)
{
    if(!codec)
    {
        codec = QTextCodec::codecForLocale();
    }
    _codec = codec;
}

bool VsqTextImporter::eof() const
{
    return _queue.empty();
}

bool VsqTextImporter::isOpen() const
{
    return !eof();
}

void VsqTextImporter::close()
{
    _queue.clear();
    _dictionary.clear();
}

bool VsqTextImporter::pop(QString &oper)
{
    if(eof())
    {
        return false;
    }

    oper = _queue.dequeue();
    return true;
}

bool VsqTextImporter::open(const QString &path)
{
    QFile file(path);
    if(!file.exists())
    {
        qDebug("VsqTextImporter::open(%s); // File not found.", path.toUtf8().data());
        return false;
    }
    if(!file.open(QIODevice::ReadOnly | QIODevice::Text))
    {
        qDebug("VsqTextImporter::open(%s); // File could not be opened.", path.toUtf8().data());
        return false;
    }

    QTextStream stream(&file);
    stream.setCodec(_codec);

    close();

    setQueue(stream);

    return true;
}

void VsqTextImporter::_setDefaultObjects()
{
    _factories.clear();
    _dictionary.clear();
    _eventTicks.clear();
    _trackId = -1;
    _eventId = 0;

    _factories["[EventList]"] = &VsqTextImporter::_readEventList;
    _factories["[Measure]"] = &VsqTextImporter::_readMeasure;

    _factories["[Tempo]"] = &VsqTextImporter::_readControl;
    _factories["[PitchBendBPList]"] = &VsqTextImporter::_readControl;
    _factories["[PitchBendSensBPList]"] = &VsqTextImporter::_readControl;
    _factories["[DynamicsBPList]"] = &VsqTextImporter::_readControl;
    _factories["[EpRResidualBPList]"] = &VsqTextImporter::_readControl;
//    _factories["[EpRESlopeBPList]"] = &VsqTextImporter::_readControl;
//    _factories["[EpRESlopeDepthBPList]"] = &VsqTextImporter::_readControl;
    _factories["[GenderFactorBPList]"] = &VsqTextImporter::_readControl;
//    _factories["[PortamentoTimingBPList]"] = &VsqTextImporter::_readControl;
//    _factories["[VibTremDepthBPList]"] = &VsqTextImporter::_readControl;
//    _factories["[OpeningBPList]"] = &VsqTextImporter::_readControl;

    _dictionary["[Tempo]"] = "Tempo";
    _dictionary["[Measure]"] = "Measure";
    _dictionary["[PitchBendBPList]"] = "PitchBend";
    _dictionary["[PitchBendSensBPList]"] = "PBS";
    _dictionary["[DynamicsBPList]"] = "Dynamics";
    _dictionary["[EpRResidualBPList]"] = "Residual";
//    _dictionary["[EpRESlopeBPList]"] = "Slope";
//    _dictionary["[EpRESlopeDepthBPList]"] = "SlopeDepth";
    _dictionary["[GenderFactorBPList]"] = "Gender";
//    _dictionary["[PortamentoTimingBPList]"] = "Portamento";
//    _dictionary["[VibTremDepthBPList]"] = "Tremolo";
//    _dictionary["[OpeningBPList]"] = "Opening";

    _queue.clear();
}

void VsqTextImporter::setQueue(QTextStream &stream)
{
    int c = 0;

    _setDefaultObjects();

    _queue.enqueue("Sequence Sequence sequence");

    while(!stream.atEnd())
    {
        c++;
        QString line = stream.readLine();
        if(_factories.contains(line))
        {
            _currentParse = line;
            if(_currentParse == "[EventList]")
            {
                _trackId++;
                _dictionary[_currentParse] = "track" + QString::number(_trackId);
                _queue.enqueue("sequence Track track" + QString::number(_trackId));
            }
            else if(_currentParse == "[Tempo]")
            {
                _queue.enqueue("sequence Tempo tempo");
                _currentObjectName = "tempo";
            }
            else if(_currentParse == "[Measure]")
            {
                _queue.enqueue("sequence Measure measure");
            }
            else if(!_currentParse.contains("#"))
            {
                _queue.enqueue("sequence " + _dictionary[line] + " " + _dictionary[line]);
            }
            continue;
        }
        if(_factories.contains(_currentParse))
        {
            if(_dictionary.contains(_currentParse) && _currentParse != "[Tempo]")
            {
                _currentObjectName = _dictionary[_currentParse].toLower() + QString::number(_trackId);
            }
            else if(_currentParse != "[Tempo]")
            {
                _currentObjectName = QString::null;
            }
            (this->*_factories[_currentParse])(line);
        }
        else
        {
            _currentParse = "";
            qDebug(" Undefined symbol at VsqTextImporter::setQueue();");
            qDebug("   \"%s\" is unknown in l. %d", _currentParse.toUtf8().data(), c);
        }
    }
}

void VsqTextImporter::_readEventList(const QString &line)
{
    QStringList sl = line.split("=", QString::KeepEmptyParts);
    if(sl.size() < 2)
    {
        return;
    }
    QStringList right = sl.at(1).split(",", QString::KeepEmptyParts);
    for(int i = 0; i < right.size(); i++)
    {
        QString label = "[" + right.at(i) + "]";
        _eventTicks[label] = sl.at(0).toInt();
        _factories[label] = &VsqTextImporter::_readEvent;
    }
}

void VsqTextImporter::_readMeasure(const QString &line)
{
    QStringList sl = line.split("=");
    if(sl.size() != 2)
    {
        return;
    }
    QStringList right = sl.at(1).split(",", QString::KeepEmptyParts);
    if(right.size() != 2)
    {
        return;
    }
    QString l = "measure " + sl.at(0) + " " + right.at(0) + " " + right.at(1);
    _queue.enqueue(l);
}

void VsqTextImporter::_readLyric(const QString &line)
{
    QStringList sl = line.split("=");
    if(sl.size() != 2)
    {
        return;
    }
    QStringList params = sl.at(1).split(",");
    if(params.size() != 5)
    {
        return;
    }
    QString content = params.at(0);
    content.remove("\"");
    content.replace(" ", "&");
    QString l = _dictionary[_currentParse] + " Lyric " + content;
    _queue.enqueue(l);

    content = params.at(1);
    content.remove("\"");
    content.replace(" ", "&");
    l = _dictionary[_currentParse] + " Pronounce " + content;
    _queue.enqueue(l);
}

void VsqTextImporter::_readEvent(const QString &line)
{
    QStringList sl = line.split("=");
    if(sl.size() < 2)
    {
        return;
    }
    if(sl.at(0) == "Type")
    {
        QString l = "track" + QString::number(_trackId), name;
        if(sl.at(1) == "Singer")
        {
            l += " Singer singer";
            name = "singer" + QString::number(_eventId);
        }
        else
        {
            l += " Note note";
            name = "note" + QString::number(_eventId);
        }
        l += QString::number(_eventId) + " " + QString::number(_eventTicks[_currentParse]);
        _dictionary[_currentParse] = name;
        _queue.enqueue(l);
        _eventId++;
        return;
    }

    if(!_dictionary.contains(_currentParse))
    {
        return;
    }

    QString l = "";

    if(sl.at(0) == "Length")
    {
        l = _dictionary[_currentParse] + " Length " + sl.at(1);
        _queue.enqueue(l);
    }
    else if(sl.at(0) == "Note#")
    {
        l = _dictionary[_currentParse] + " Note " + sl.at(1);
        _queue.enqueue(l);
    }
    else if(sl.at(0) == "LyricHandle")
    {
        l = _dictionary[_currentParse] + " Lyric lyric" + QString::number(_eventId);
        QString name = "[" + sl.at(1) + "]";
        _factories[name] = &VsqTextImporter::_readLyric;
        _dictionary[name] = "lyric" + QString::number(_eventId);
        _queue.enqueue(l);
        _eventId++;
    }
    else if(sl.at(0) == "Dynamics")
    {
        l = _dictionary[_currentParse] + " Velocity " + sl.at(1);
    }

    if(l != "")
    {
        _queue.enqueue(l);
    }
}

void VsqTextImporter::_readControl(const QString &line)
{
    if(_currentObjectName == QString::null)
    {
        return;
    }
    QStringList sl = line.split("=");
    if(sl.size() != 2)
    {
        return;
    }
    QString l = _currentObjectName + " " + sl.at(0) + " " + sl.at(1);
    _queue.enqueue(l);
}
