#include <QTest>
#include <QDir>
#include <fstream>
#include <QFileInfo>

#include "ControlItemTest.h"

#include "ControlItem.h"

using namespace stand::model;

ControlItemTest::ControlItemTest(QObject *parent) :
    QObject(parent)
{
}

void ControlItemTest::execTest()
{
    ControlItem control("test", 0.0);
    for(int i = 0; i < 100; i++)
    {
        control.setData(i * 100, i, Qt::EditRole);
    }

    for(int i = 0; i < 10000; i++)
    {
        QVERIFY2(control.data(i, Qt::EditRole).toDouble() == (double)(i / 100),
                 (QString::number(i / 100) + " expected, actual data is "
                  + QString::number(control.data(i, Qt::EditRole).toDouble())
                  + "; index at " + QString::number(i)).toUtf8().data());
    }
}

void ControlItemTest::contourTest()
{
    ControlItem control("test", 0.0);
    for(int i = 0; i < 100; i++)
    {
        control.setData(i * 100, i, Qt::EditRole);
    }
    const int begin = 5, end = 65;
    QList<stand::utility::ControlPoint> cl = control.contour(100 * begin - 50, 100 * end);

    for(int i = 0; i < cl.size(); i++)
    {
        QVERIFY(cl[i].tick / 100 - begin == i);
    }
}

void ControlItemTest::setControlItem(ControlItem *control, const QString &filePath)
{
    QString nativepath = QDir::toNativeSeparators(filePath);
    nativepath.replace("\\", "/");  //
    QFileInfo f(nativepath);
    nativepath = f.absoluteFilePath();
    std::ifstream ifs(nativepath.toLocal8Bit().data());
    if(!ifs.is_open())
    {
        qDebug("Test file not found.; %s", nativepath.toUtf8().data());
        return;
    }

    while(!ifs.eof())
    {
        int tick;
        double value;
        ifs >> tick >> value;
        control->setData(tick, value, Qt::EditRole);
    }
}

void ControlItemTest::insertTest_data()
{
    QTest::addColumn<QString>("input");
    QTest::addColumn<QString>("insert");
    QTest::addColumn<QString>("result");

    QTest::newRow("Read1")   << "test\\data\\ControlItemTest\\ReadInput1.txt"
                             << "test\\data\\ControlItemTest\\ReadInsert.txt"
                             << "test\\data\\ControlItemTest\\ReadResult.txt";

    QTest::newRow("Read2")   << "test\\data\\ControlItemTest\\ReadInput2.txt"
                             << "test\\data\\ControlItemTest\\ReadInsert.txt"
                             << "test\\data\\ControlItemTest\\ReadResult.txt";

    QTest::newRow("Simple")  << "test\\data\\ControlItemTest\\SimpleInput.txt"
                             << "test\\data\\ControlItemTest\\SimpleInsert.txt"
                             << "test\\data\\ControlItemTest\\SimpleResult.txt";

    QTest::newRow("Mid1")    << "test\\data\\ControlItemTest\\MidInput.txt"
                             << "test\\data\\ControlItemTest\\MidInsert1.txt"
                             << "test\\data\\ControlItemTest\\MidResult1.txt";

    QTest::newRow("Mid1")    << "test\\data\\ControlItemTest\\MidInput.txt"
                             << "test\\data\\ControlItemTest\\MidInsert2.txt"
                             << "test\\data\\ControlItemTest\\MidResult2.txt";
}

void ControlItemTest::insertAndReplaceTest_data()
{
    insertTest_data();
}

void ControlItemTest::insertTest()
{
    QFETCH(QString, input);
    QFETCH(QString, insert);
    QFETCH(QString, result);

    ControlItem inputControl("test", 0);
    setControlItem(&inputControl, input);

    ControlItem insertControl("test", 0);
    setControlItem(&insertControl, insert);

    ControlItem resultControl("test", 0);
    setControlItem(&resultControl, result);

    QList<stand::utility::ControlPoint> res, act, ins;
    ins = insertControl.contour(0, INT_MAX);
    res = resultControl.contour(0, INT_MAX);

    if(!ins.empty())
    {
        inputControl.replace(ins, ins.at(0).tick, ins.last().tick);
    }
    act = inputControl.contour(0, INT_MAX);

    for(int i = 0; i < act.size() || i < res.size(); i++)
    {
        qDebug(" index = %d; (%8d, %8f), (%8d, %8f)", i, (i < act.size()) ? act[i].tick : -1, (i < act.size()) ? act[i].value : -1, (i < res.size()) ? res[i].tick : -1, (i < res.size()) ?res[i].value : -1);
    }

    QVERIFY2(act.size() == res.size(), (QString("actual size = ") + QString::number(act.size()) + ", expected size = " + QString::number(res.size())).toUtf8().data());

    for(int i = 0; i < act.size(); i++)
    {
        QString err;
        if(!(act[i].tick == res[i].tick && act[i].value == res[i].value))
        {
            err = "index " + QString::number(i) + "\t; (" + QString::number(act[i].tick) + ", " + QString::number(act[i].value) + ")\t (" + QString::number(res[i].tick) + ", " + QString::number(res[i].value) + ")";
        }
        QVERIFY2(act[i].tick == res[i].tick && act[i].value == res[i].value, err.toUtf8().data());
    }
}

void ControlItemTest::insertAndReplaceTest()
{
    QFETCH(QString, input);
    QFETCH(QString, insert);
    QFETCH(QString, result);

    ControlItem inputControl("test", 0);
    setControlItem(&inputControl, input);

    ControlItem insertControl("test", 0);
    setControlItem(&insertControl, insert);

    QList<stand::utility::ControlPoint> src, res, ins, tmp;
    ins = insertControl.contour(0, INT_MAX);
    src = inputControl.contour(0, INT_MAX);

    if(ins.empty())
    {
        return;
    }

    tmp = inputControl.contour(ins.at(0).tick, ins.last().tick);
    inputControl.replace(ins, ins.at(0).tick, ins.last().tick);
    inputControl.replace(tmp, ins.at(0).tick, ins.last().tick);

    res = inputControl.contour(0, INT_MAX);

    QVERIFY2(res.size() == src.size(), (QString("src = ") + QString::number(src.size()) + " : dst = " + QString::number(res.size())).toUtf8().data());
    for(int i = 0; i < src.size(); i++)
    {
        QString err;
        if(!(src[i].tick == res[i].tick && src[i].value == res[i].value))
        {
            err = "index " + QString::number(i) + "\t; (" + QString::number(src[i].tick) + ", " + QString::number(src[i].value) + ")\t (" + QString::number(res[i].tick) + ", " + QString::number(res[i].value) + ")";
        }
        QVERIFY2(src[i].tick == res[i].tick && src[i].value == res[i].value, err.toUtf8().data());
    }
}

