(function() {
    'use strict';
    var eos = require('../../server/class/Eos').instance;
    var chai = require('chai');
    var chaiAsPromised = require('chai-as-promised');
    chai.use(chaiAsPromised);
    var expect = chai.expect;
    chai.should();

    var db = require('../../server/class/DB').instance;
    var testCommand = require('./testCommand.json');

    var test1 = `
    /**
    * コマンドとオプションのバリデーション
    * @param {string} Eosコマンドの名前
    * @param {Array<Object>} コマンドに渡すオプション
    * @returns {valid: boolean, message: string}
    * function validate(command, params) {
    */
    `;

    
    describe('Eos クラス', function() {
        before(function() {
        });

        describe(test1, function() {

            it('should return Array', function() {
                return expect(eos.getFiles()).to.eventually.be.length(2);
            });

            it('should be rejected(2)', function() {
                return eos.validate('hoge').should.be.rejectedWith(Error, 'Command name is invalid');
            });

            it('should be rejected(3)', function() {
                return eos.validate('mrcImageNoiseAdd').should.be.rejectedWith(Error, 'Options need to be Array');
            });

            it('should be rejected(4)', function() {
                return eos.validate('mrcImageNoiseAdd', []).should.be.rejectedWith(Error, 'At least one option is required.');
            });

            it('should return false when options is not Array', function() {
                return eos.validate('mrcImageNoiseAdd', {}).should.be.rejectedWith(Error, 'Options need to be Array');
            });

            it('should return false when options is invalid Object which have not "name" and "argumetns"', function() {
                return eos.validate('mrcImageInfo', [{hoge: 'hoge'}]).should.be.rejectedWith(Error, 'Options need to include Object which have properties "name" and "arguments"');
            });
            it('should return false when "argumetns" properties are not Array', function() {
                return eos.validate('mrcImageInfo', [{name: 'hoge', arguments: 'hoge'}]).should.be.rejectedWith(Error,'Each "arguments" properties needs to be Array');
            });

            it('should return false when required options do not exist', function() {
                return eos.validate('mrcImageInfo', [{name: 'hoge', arguments: []}]).should.be.rejectedWith('Option -i are required');
            });

            it('should return false when required options do not exist', function() {
                return eos.validate('dcdFilePrint', [{name: '-r', arguments: [1,2]}, { name: '-i', arguments: []}, { name: '-o', arguments: []} ]).should.be.rejectedWith(Error, 'Invalid arguments number');
            });

            it('should return false when input file does not exist', function() {
                return eos.validate('dcdFilePrint', [{name: '-r', arguments: [1,2,3]}, { name: '-i', arguments: ['hoge.txt']}, { name: '-o', arguments: ['hoge.txt']} ]).should.be.rejectedWith(Error, 'hoge.txt doesn\'t exist.');
            });

            it('should return false when output file is not string', function() {
                return eos.validate('dcdFilePrint', [{name: '-r', arguments: [1,2,3]}, { name: '-i', arguments: ['file1.txt']}, { name: '-o', arguments: [3]} ]).should.be.rejectedWith(Error, 'argType is invalid');
            });

            it('should return false when output file has already existed', function() {
                return eos.validate('dcdFilePrint', [{name: '-r', arguments: [1,2,3]}, { name: '-i', arguments: ['file1.txt']}, { name: '-o', arguments: ['file1.txt']} ]).should.be.rejectedWith(Error, 'Invalid outfile name.');
            });

        });

        var test2 = `
        /**
        * 実行時文字列への変換 
        *
        * @param command
        * @param options
        * @returns {string}
        * function toExecString(command, params) {
        */
        `;

        describe(test2, function() {
            it('should return true when all options is proper.', function() {
                var result = eos.toExecString('dcdFilePrint', [{name: '-r', arguments: [1,2,3]}, { name: '-i', arguments: ['file1.txt']}, { name: '-o', arguments: ['file3.txt']} ], "1f83f620-c1ed-11e5-9657-7942989daa00");
                expect(result).to.equal('dcdFilePrint -r 1 2 3 -s 10 -e 100 -d 10 -m 0 -i /Users/Takafumi/zephyr/server/class/../../user-specific-files/workspace/file1.txt -o /Users/Takafumi/zephyr/server/class/../../user-specific-files/workspace/file3.txt');
            });
        });

        var toExecArray = `
        /**
        * 実行時配列への変換 
        *
        * @param command
        * @returns {string}
        * function toExecString(command, params) {
        */
        `;

        describe(toExecArray, function() {
            before(function() {
                process.env.NODE_ENV = '';
                return db.init()
                .then(function() {
                    return Promise.all([db.test1(), db.test2(), db.testRest()])
                });
            });

            it('should be resolved with length 4', function() {
                var command = testCommand.command;
                var options = testCommand.options;
                var workspace = testCommand.workspace;
                return eos.toExecArray(command, options, '1f83f620-c1ed-11e5-9657-7942989daa00').should.be.eventually.length(16);
            });

        });
        
        describe('execute', function() {
            before(function() {
                process.env.NODE_ENV = '';
                return db.init()
                .then(function() {
                    return Promise.all([db.test1(), db.test2(), db.testRest()])
                });
            });

            it('should be resolved with length 4', function() {
                return expect(eos.getFiles('1f83f620-c1ed-11e5-9657-7942989daa00')).to.eventually.length(4);
            });

            it('should be resolved about testCommand', function() {
                var command = testCommand.command;
                var options = testCommand.options;
                var workspace = testCommand.workspace;
                return eos.validate(command, options, workspace).should.be.fulfilled;
            });

            after(function() {
                process.env.NODE_ENV = 'test';
            });
        });

        describe('getFiles', function() {
            before(function() {
                process.env.NODE_ENV = '';
                return db.init()
                .then(function() {
                    return Promise.all([db.test1(), db.test2(), db.testRest()])
                });
            });

            it('should be resolved with length 4', function() {
                return expect(eos.getFiles('1f83f620-c1ed-11e5-9657-7942989daa00')).to.eventually.length(4);
            });

            it('should be resolved about testCommand', function() {
                var command = testCommand.command;
                var options = testCommand.options;
                var workspace = testCommand.workspace;
                return eos.validate(command, options, workspace).should.be.fulfilled;
            });

            after(function() {
                process.env.NODE_ENV = 'test';
            });
        });

        var getUUIDs = `
        /**
        * function getUUIDs(fileId)
        * uuidとファイル名のkey-valueを取得 
        *
        * @param fileId
        * @returns {object} key: filename, value: uuid
        */
        `;

        describe(getUUIDs, function() {
            before(function() {
                process.env.NODE_ENV = '';
                return db.init()
                .then(function() {
                    return Promise.all([db.test1(), db.test2(), db.testRest()])
                });
            });

            it('should be resolved', function() {
                return eos.getUUIDs('1f83f620-c1ed-11e5-9657-7942989daa00').should.eventually.have.property("hoge.txt");
            });
        });
    });
})();
