/**
 * DBを操作するクラス
 *
 * @returns {object}
 */
'use strict';

var fs = require('fs');
var Sequelize = require('sequelize');
var co = require('co');
var uuid = require('node-uuid');

var sequelize;
var test = 'hello';
var Files; // Model object of sequelize

 var instance = {
    init: init,
    sayHello: function() {
        return test;
    },
    getDirectoryParentId: getDirectoryParentId,
    getDirectoryId: getDirectoryId,
    getDirectory: getDirectory,
    getFiles: getFiles,
    getDirectories: getDirectories,
    existFile: existFile,
    notExistFile: notExistFile,
    existDirectory: existDirectory,
    notExistDirectory: notExistDirectory,
    createFile: createFile,
    removeFile: removeFile,
    createDirectory: createDirectory,
    test1: test1,
    test2: test2,
    testRest: testRest
    /*
    removeDirectory: removeDirectory,
    getFilename: getFilename,
    moveFile: moveFile,
    moveDirectory: moveDirectory
    */
};

/**
 * init
 * @constructor
 * @returns {promise}
 */

/** 
 * productionモード時は、db.workspaceを永続化させる
 * DEBUGモードの時は、db.debugを実行毎に作成する。
 * DEBUGモードの時は、sync()の中で、/レコードがFilesテーブルに追加される。
 * モードの切り替えは環境変数NODE_ENVで行っている。
 * zephyr serveコマンドが実行されるとNODE_ENV = 'production'
 * zephyr debugコマンドが実行されるとNODE_ENV = 'debug'
 * となる。
 * それぞれの設定は$ZEOHYR_HOME/cli/zephyr-serve, $ZEPHYR_HOME/zephyre_debugを参照
 */

    function init() {
        var dbPath;
        var dbOption = {
            dialect: 'sqlite'
        };

        if(process.env['NODE_ENV'] === 'production') {
            dbPath = __dirname + '/../../user-specific-files/db/db.workspace';
            // if doesn't exist workspace.db, create.
            try {
                fs.accessSync(dbPath, fs.R_OK | fs.W_OK);
            } catch(e) {
                fs.writeFileSync(dbPath, '');
            }
            dbOption.storage = dbPath;

        } else if(process.env['NODE_ENV'] === 'debug') {
            dbPath = __dirname + '/../../user-specific-files/db/db.debug';
            try {
                fs.accessSync(dbPath, fs.R_OK | fs.W_OK);
                fs.unlinkSync(dbPath);
            } catch(e) {
            }

            fs.writeFileSync(dbPath, '');
            dbOption.storage = dbPath;
        }

        sequelize = new Sequelize('','','', dbOption);

        Files = sequelize.define('file', {
            fileId: {
                type: Sequelize.UUID,
                field: 'file_id',
                primaryKey: true
            },
            name: {
                type: Sequelize.STRING,
                field: 'name',
                allowNull: false
            },
            parentId: {
                type: Sequelize.UUID,
                field: 'parent_id',
                allowNull: false
            },
            fileType: {
                type: Sequelize.ENUM(0,1),
                field: 'file_type',
                allowNull: false
            }
        });

        var force = process.env.NODE_ENV !== 'production';
        return Files.sync({force: force})
        .then(function() {
            return insertRoot();
        });
    }


/**
 * sync
 *
 * @returns {promise}<DBオブジェクトの関数群のオブジェクトをresolveする>
 */
function insertRoot() {
    return new Promise(function(resolve, reject) {
        var q = {
            where: {
                name: '/'
            }
        };
        Files.findOne(q)
        .then(function(r) {
            if(r === null) {
                var root = {
                    fileId: '1f83f620-c1ed-11e5-9657-7942989daa00', // rootのuuidは固定値
                    name: '/',
                    parentId: '',
                    fileType: 0
                };
                return Files.create(root);
            }
        })
        .then(function() {
            resolve();
        });
    });
}

/**
 * getDirectoryParentId
 * ディレクトリのparentIdを取得する 
 * @param name
 * @returns {promise} ディレクトリが存在すればresolve(uuid), しなければreject
 */
function getDirectoryParentId(name) {
    return new Promise(function(resolve, reject) {
        var q = {
            where: {
                name: name,
                fileType: 0
            }
        };
        Files.findAll(q)
        .then(function(r) {
            if(r.length === 0) {
                reject(new Error('"' + name + '" directory doesn\'t exist.'));
            } else {
                var map = r.map(function(c) { return c.dataValues.parentId });
                resolve(map);
            }
        });
    });
}


/**
 * getDirectoryId
 * ディレクトリのfileIdを取得する 
 * @param name
 * @returns {promise} ディレクトリが存在すればresolve(uuid), しなければreject
 */

function getDirectoryId(name) {
    return new Promise(function(resolve, reject) {
        var q = {
            where: {
                name: name,
                fileType: 0
            }
        };
        Files.findAll(q)
        .then(function(r) {
            if(r.length === 0) {
                reject(new Error('"' + name + '" directory doesn\'t exist.'));
            } else {
                var map = r.map(function(c) { return c.dataValues.fileId });
                resolve(map);
            }
        });
    });
}

/**
 * getDirectory
 * ディレクトリのfileIdを取得する 
 * @param name
 * @returns {promise} ディレクトリが存在すればresolve(name), しなければreject
 */

function getDirectory(name) {
    return new Promise(function(resolve, reject) {
        var q = {
            where: {
                name: name,
                fileType: 0
            }
        };
        Files.findAll(q)
        .then(function(r) {
            if(r.length === 0) {
                reject(new Error('"' + name + '" directory doesn\'t exist.'));
            } else {
                var map = r.map(function(c) { return c.dataValues });
                resolve(map);
            }
        });
    });
}

/**
 * getFiles
 * ディレクトリのfileId一覧を取得する 
 * @param {string} fileId ディレクトリのfileId 
 * @returns {promise} resolve([Array]<string>fileId) 引数で与えられたディレクトリを直接の親に持つファイルのレコードの配列を返す。
 *                    与えられたfileIdがディレクトリでなければresolve
 */
function getFiles(fileId) {
    return new Promise(function(resolve, reject) {
        if(!fileId) {
            reject(new Error('"'+fileId+'" is invalid.'));
        }
        var q = {
            where: {
                parentId: fileId
            }
        };
        Files.findAll(q)
        .then(function(r) {
            if(r.length === 0) {
                reject(new Error('"'+fileId+'" is invalid.'));
            }
            var files = r.map(function(data) { return data; });
            resolve(files);
        });
    });
}

/**
 * getDirectories
 * ディレクトリの一覧を取得する 
 * @returns {promise} resolve([Array]<Object>) 
 */
function getDirectories() {
    return new Promise(function(resolve, reject) {
        var q = {
            where: {
                fileType: 0
            }
        };
        Files.findAll(q)
        .then(function(r) {
            var dirs = r.map(function(data) { 
                return data.dataValues; 
            });
            resolve(dirs);
        });
    });
}

/**
 * existFile
 * 同一ディレクトリに同名のファイルが存在することを確かめる
 * @param {string} fileName
 * @param {string} parentDirectory parentDirectoryの絶対パス
 * @returns {promise} ファイルが存在すればresolve、存在しなければreject
 */
function existFile(fileName, parentDirectory) {
    return new Promise(function(resolve, reject) {
        existDirectory(parentDirectory)
        .catch(function(error) {
            reject(error);
        })
        .then(function(fileId) {
            var q = {
                where: {
                    name: fileName,
                    parentId: fileId
                }
            };
            return Files.findOne(q)
        })
        .then(function(r) {
            if(r === null) {
                reject(new Error("\"" + fileName + "\" does not exist in " + '"' + parentDirectory + "\" directory."));
            } else {
                resolve(r.fileId);
            }
        });
    });
}


/**
 * notExistFile
 * 同一ディレクトリに同名のファイルが存在していないことを確かめる
 * @param {string}fileName
 * @param {string}parentDirectory
 * @returns {promise} ファイルが存在しなければresolve、存在すればreject
 */
function notExistFile(fileName, parentDirectory) {
    return new Promise(function(resolve, reject) {
        existDirectory(parentDirectory)
        .catch(function(error) {
            reject(error);
        })
        .then(function(fileId) {
            var q = {
                where: {
                    name: fileName,
                    parentId: fileId
                }
            };
            return Files.findOne(q)
        })
        .then(function(r) {
            if(r === null) {
                resolve();
            } else {
                reject(new Error("\"" + fileName + "\" has already existed in " + '"' + parentDirectory + "\" directory."));
            }
        });
    });
}

/**
 * existDirectory
 * ディレクトリが存在することを確認する
 * @param {string} directory
 * @returns {promise} ディレクトリが存在すればresolve{fileId)、存在しなければreject
 */
function existDirectory(directory) {
    return new Promise(function(resolve, reject) {
        if(!directory) {
            reject(new Error('parameter "directory" is undefined'));
        }

        var arrayDirectory; 
        var root = directory.substr(0,1);

        if(root !== '/') {
            reject(new Error('directory name should start "/" so that it is absolute path including root.'));
        }

        if(directory === '/') {
            resolve('1f83f620-c1ed-11e5-9657-7942989daa00'); // rootのuuid
        } else {
            arrayDirectory = directory.split('/');
            arrayDirectory.shift(); // root
            arrayDirectory.unshift('/');
        }

        var directoriesPromise = arrayDirectory.map(function(name) {
            return getDirectory(name);
        });

        Promise.all(directoriesPromise)
        .then(function(r) {
            var parentId = r[0][0].fileId;
            var index;
            for(var i=1;i<r.length;i++) {
                index = r[i].map(function(c) { return c.parentId }).indexOf(parentId);
                if(index > -1) {
                    parentId = r[i][index].fileId;
                } else {
                    reject(new Error('"' + directory + '" directory doesn\'t exist.'));
                }
            }
            resolve(parentId);
        })
        .catch(function(error) {
            reject(new Error('"' + directory + '" directory doesn\'t exist.'));
        });
    });
}

/**
 * notExistDirectory
 * ディレクトリが存在しないことを確認する
 * @param {string} directory
 * @returns {promise} ディレクトリが存在しなければresolve、存在すればreject
 */
function notExistDirectory(directory) {
    return new Promise(function(resolve, reject) {
        if(!directory) {
            resolve();
        } 

        var arrayDirectory; 
        var root = directory.substr(0,1);

        if(root !== '/') {
            resolve();
        }

        if(directory === '/') {
            reject(new Error('"' + directory + '" directory exists.'));
        } else {
            arrayDirectory = directory.split('/');
            arrayDirectory.shift(); // root
            arrayDirectory.unshift('/');
        }

        var directoriesPromise = arrayDirectory.map(function(name) {
            return getDirectory(name);
        });
        Promise.all(directoriesPromise)
        .then(function(r) {
            var parentId = r[0][0].fileId;
            var index;
            for(var i=1;i<r.length;i++) {
                index = r[i].map(function(c) { return c.parentId }).indexOf(parentId);
                if(index > -1) {
                    parentId = r[i][index].fileId;
                } else {
                    resolve();
                }
            }
            reject(new Error('"' + directory + '" directory exists.'));
        })
        .catch(function(error) {
            resolve();
        });
    });
}

/**
 * createFile
 *
 * @param fileName 
 * @param parentDirectory
 * @returns {promise}<sequelize.createの結果を格納したobject | Error>
 */
function createFile(fileName,parentDirectory) {
    return new Promise(function(resolve, reject) {
        if(!fileName) {
            reject(new Error('filename is required.'));
        }
        Promise.all([existDirectory(parentDirectory), notExistFile(fileName, parentDirectory) ])
        .catch(function(error) {
            reject(error);
        })
        .then(function(r) {
            var parentId = r[0]
            var q = {
                fileId: uuid.v1(),
                name: fileName,
                parentId: parentId,
                fileType: 1 
            }
            return Files.create(q)
        })
        .then(function(r) {
            resolve(r.dataValues.fileId);
        });
    });
}


/**
 * removeFile
 * ファイルを削除する
 * @param {string} fileName
 * @param {string} parentDirectory
 * @returns {promise} ファイル削除に成功すればresolve、失敗すればreject
 */
function removeFile(fileName, parentDirectory) {
    return new Promise(function(resolve, reject) {
        existFile(fileName, parentDirectory)
        .catch(function(error) {
            reject(error);
        })
        .then(function(fileId) {
            var q = {
                where: {
                    fileId: fileId
                }
            };
            return Files.destroy(q);
        })
        .then(function() {
            resolve();
        });
    });
}

/**
 * createDirectory
 * ディレクトリを作成
 * @param directory
 * @returns {promise} ディレクトリの作成に成功すればresolve、失敗すればreject
 */
function createDirectory(directory) {
    return new Promise(function(resolve, reject) {
        if(!directory) {
            reject(new Error('directory name should start "/" so that it is absolute path including root.'));
        }

        var leaf = directory.split('/').pop();
        var parentDirectory = directory.replace('/'+leaf, '');
        if(!parentDirectory) {
            parentDirectory = '/';
        }
        Promise.all([existDirectory(parentDirectory), notExistDirectory(directory), notExistFile(leaf, parentDirectory)])
        .catch(function(error) {
            reject(error);
        })
        .then(function(r) {
            var parentId = r[0];
            var q = {
                fileId: uuid.v1(),
                name: leaf,
                parentId: parentId,
                fileType: 0 
            }
            return Files.create(q)
        })
        .then(function(r) {
            resolve(r.dataValues.fileId);
        });
    });
}


/**
 * removeDirectory
 * ディレクトリを削除
 * @param directory
 * @returns {promise} ディレクトリの削除に成功すればresolve、失敗すればreject
 */
function removeDirectory(directory) {
    return new Promise(function(resolve, reject) {
        var leaf = directory.split('/').pop();
        var parentDirectory = directory.replace('/'+leaf, '');
        if(!parentDirectory) {
            parentDirectory = '/';
        }
        Promise.all([existDirectory(parentDirectory), notExistDirectory(directory), notExistFile(leaf, parentDirectory)])
        .catch(function(error) {
            reject(error);
        })
        .then(function(r) {
            var parentId = r[0];
            var q = {
                fileId: uuid.v1(),
                name: leaf,
                parentId: parentId,
                fileType: 0 
            }
            return Files.create(q)
        })
        .then(function(r) {
            resolve(r.dataValues.fileId);
        });
    });
}

/**
 * test1
 * test用にデータベースのレコードを追加する関数
 * @returns {promise}
 */
function test1() {
    var q = {
        fileId: uuid.v1(),
        name: 'hoge.txt',
        parentId: '1f83f620-c1ed-11e5-9657-7942989daa00', // rootのuuid
        fileType: 1 
    };
    return Files.create(q);
}

/**
 * test2
 * test用にデータベースのレコードを追加する関数
 * @returns {promise}
 */

function test2() {
    var q1 = {
        fileId: '1f83f620-c1ed-11e5-9657-7942989daa04',
        name: 'one',
        parentId: '1f83f620-c1ed-11e5-9657-7942989daa00', // rootのuuid
        fileType: 0 //directory
    };
    return Files.create(q1)
    .then(function() {
        var q2 = {
            fileId: '1f83f620-c1ed-11e5-9657-7942989daa01',
            name: 'two',
            parentId: q1.fileId,
            fileType: 0 

        };
        return Files.create(q2);
    })
    .then(function(r) {
        var q3 = {
            fileId: '1f83f620-c1ed-11e5-9657-7942989daa02',
            name: 'two',
            parentId: '1f83f620-c1ed-11e5-9657-7942989daa00', //rootのuuid
            fileType: 0 
        };
        return Files.create(q3);
    })
    .then(function() {
        var q4 = {
            fileId:'1f83f620-c1ed-11e5-9657-7942989daa03',
            name: 'hogehoge.txt',
            parentId: q1.fileId,
            fileType: 1
        };
        return Files.create(q4);
    });
}

function testRest() {
    var q = {
        fileId: 'dca94750-c44f-11e5-88c4-f31cdfe97f4f',
        name: 'debugRest.mrc',
        parentId: '1f83f620-c1ed-11e5-9657-7942989daa00', //rootのuuid
        fileType: 1 // file
    };
    return Files.create(q);
}

module.exports = { instance: instance };
