(function() {
    'use strict';
    var db = require('../../server/class/DB').instance;
    var chai = require('chai');
    var chaiAsPromised = require('chai-as-promised');
    chai.use(chaiAsPromised);
    var expect = chai.expect;
    chai.should();

    describe('DB クラス', function() {

        var constructor = `
        /**
        * createFile
        *
        * @param name
        * @param directory
        * @returns {promise}<保存したファイルのファイル名(uuid)をresolveする>
        */
        `;
        describe(constructor, function() {
            before(function() {
                return db.init()
            });
            it('should return hello when call sayHello', function() {
                expect(db.sayHello()).to.equal('hello');
            });
        });


        var getDirectoryParentId = `
        /**
        * getDirectoryParentId
        * ディレクトリのparentIdを取得する 
        * @param name
        * @returns {promise} ディレクトリが存在すればresolve(uuid), しなければreject
        */
        `;
        describe(getDirectoryParentId, function() {
            before(function() {
                return db.init()
                .then(function() {
                    return db.test2();
                });
            });

            it('should be rejected when a directory does not exist', function() {
                return db.getDirectoryParentId('hoge').should.be.rejectedWith(Error, '"hoge" directory doesn\'t exist.');
            });
            it('should be resolved when \'one\' exists', function() {
                return expect(db.getDirectoryParentId('one')).to.eventually.include.members(['1f83f620-c1ed-11e5-9657-7942989daa00']);
            });
            it('should be rejected when \'two\' exists', function() {
                return expect(db.getDirectoryParentId('two')).to.eventually.have.length(2);
            });
        });

        var getDirectoryId = `
        /**
        * getDirectoryId
        * ディレクトリのfileIdを取得する 
        * @param name
        * @returns {promise} ディレクトリが存在すればresolve(uuid), しなければreject
        */
        `;
        describe(getDirectoryId, function() {
            before(function() {
                return db.init()
                .then(function() {
                    return db.test2();
                });
            });
            it('should be rejected when a directory does not exist', function() {
                return db.getDirectoryId('hoge').should.be.rejectedWith(Error, '"hoge" directory doesn\'t exist.');
            });
            it('should be resolved when \'/\' exists', function() {
                return expect(db.getDirectoryId('/')).to.eventually.include.members(['1f83f620-c1ed-11e5-9657-7942989daa00']);
            });
            it('should be rejected when \'two\' exists', function() {
                return expect(db.getDirectoryId('two')).to.eventually.have.length(2);
            });
        });


        var getDirectory = `
        /**
        * getDirectory
        * ディレクトリの情報を取得する 
        * @param name
        * @returns {promise} ディレクトリが存在すればresolve(obj), しなければreject
        */
        `;
        describe(getDirectory, function() {
            before(function() {
                return db.init()
                .then(function() {
                    return db.test2();
                });
            });

            it('should be rejected when a directory does not exist', function() {
                return db.getDirectoryId('hoge').should.be.rejectedWith(Error, '"hoge" directory doesn\'t exist.');
            });
            it('should be resolved when \'/\' exists', function() {
                return expect(db.getDirectory('/')).to.eventually.have.length(1);
            });
            it('should be rejected when \'two\' exists', function() {
                return expect(db.getDirectory('two')).to.eventually.have.length(2);
            });
        });


        var existFile = `
        /**
        * existFile
        * 同一ディレクトリに同名のファイルが存在することを確かめる
        * @param {string}fileName
        * @param {string}parentDirectory
        * @returns {promise}
        */
        `;
        describe(existFile, function() {
            before(function() {
                return db.init()
                .then(function() {
                    return db.test1();
                })
                .then(function() {
                    return db.test2();
                });
            });

            it('should be rejected when does not exist same name file in a directory', function() {
                return db.existFile('hogehoge.txt', '/').should.be.rejectedWith(Error, '"hogehoge.txt" does not exist in "/" directory.');
            });
            it('should be rejected when a directory does not exist', function() {
                return db.existFile('hogehoge.txt', '/hoge').should.be.rejectedWith(Error, '"/hoge" directory doesn\'t exist.');
            });
            it('should be resolved when file exists in a directory', function() {
                return expect(db.existFile('hogehoge.txt', '/one')).to.eventually.be.a('string');
            });
        });



        var notExistFile = `
        /**
        * notExistFile
        * 同一ディレクトリに同名のファイルが存在しないことを確かめる
        * @param {string}fileName
        * @param {string}parentDirectory
        * @returns {promise}
        */
        `;
        describe(notExistFile, function() {
            before(function() {
                return db.init()
                .then(function() {
                    return db.test1();
                })
                .then(function() {
                    return db.test2();
                });
            });

            it('should be rejected when the same name file has already existed in directory(1)', function() {
                return db.notExistFile('hoge.txt', '/').should.be.rejectedWith(Error, '"hoge.txt" has already existed in "/" directory.');
            });
            it('should be rejected when the same name file has already existed in directory(2)', function() {
                return db.notExistFile('hogehoge.txt', '/one').should.be.rejectedWith(Error, '"hogehoge.txt" has already existed in "/one" directory.');
            });
            it('should be resolved when the same name file does not exist in directory', function() {
                return expect(db.notExistFile('hoge.txt', '/one')).to.eventually.equal();
            });
        });

        var existDirectory = `
        /**
        * existDirectory
        * ディレクトリが存在することを確認する
        * @param {string} directory ディレクトリの絶対パス
        * @returns {promise} ディレクトリが存在しなければresolve、すればreject
        */
        `;
        describe(existDirectory, function() {
            before(function() {
                return db.init()
                .then(function() {
                    return db.test2();
                })
            });

            it('should be rejected when does not exist directory(1)', function() {
                return db.existDirectory('/one/two/four').should.be.rejectedWith(Error, '"/one/two/four" directory doesn\'t exist.');
            });
            it('should be rejected when does not exist directory(2)', function() {
                return db.existDirectory('hoge').should.be.rejectedWith(Error, 'directory name should start "/" so that it is absolute path including root.');
            });
            it('should be resolved when exists directory(1)', function() {
                return expect(db.existDirectory('/')).to.eventually.equal('1f83f620-c1ed-11e5-9657-7942989daa00');
            });
            it('should be resolved when exists directory(2)', function() {
                return expect(db.existDirectory('/one')).to.eventually.be.a('string');
            });
            it('should be resolved when exists directory(3)', function() {
                return expect(db.existDirectory('/one/two')).to.eventually.be.a('string');
            });
            it('should be rejected when call with no parameter', function() {
                return db.existDirectory().should.be.rejectedWith(Error, 'parameter "directory" is undefined');
            });
        });


        var notExistDirectory = `
        /**
        * notExistDirectory
        * ディレクトリが存在しないことを確認する
        * @param {string} directory
        * @returns {promise} ディレクトリが存在すればresolve、存在しなければreject
        */
        `;
        describe(existDirectory, function() {
            before(function() {
                return db.init()
                .then(function() {
                    return db.test1();
                })
                .then(function() {
                    return db.test2();
                });
            });

            it('should be rejected when exists directory(1)', function() {
                return db.notExistDirectory('/').should.be.rejectedWith(Error,'"/" directory exists.');
            });
            it('should be rejected when exists directory(2)', function() {
                return db.notExistDirectory('/one').should.be.rejectedWith(Error,'"/one" directory exists.');
            });
            it('should be rejected when exists directory(3)', function() {
                return db.notExistDirectory('/one/two').should.be.rejectedWith(Error,'"/one/two" directory exists.');
            });
            it('should be resolved(1)', function() {
                return expect(db.notExistDirectory()).to.eventually.equal();
            });
            it('should be resolved(2)', function() {
                return expect(db.notExistDirectory('hoge')).to.eventually.equal();
            });
        });


        var createFile = `
        /**
        * createFile
        *
        * @param name
        * @param directory
        * @returns {promise}<保存したファイルのファイル名(uuid)をresolveする>
        */
        `;
        describe(createFile, function() {
            before(function() {
                return db.init()
                .then(function() {
                    return db.test1();
                })
                .then(function() {
                    return db.createFile('tarou.txt', '/');
                });
            });

            it('should be rejected when "parentDirectory" doesn\'t exist', function() {
                return db.createFile('hoge.txt', '/hoge').should.be.rejectedWith(Error, '"/hoge" directory doesn\'t exist.');
            });
            it('should be rejected when a file has already existed in a directory.', function() {
                return db.createFile('hoge.txt', '/').should.be.rejectedWith(Error, '"hoge.txt" has already existed in "/" directory.');
            });
            it('should be resolved when file creation successed.', function() {
                return expect(db.existFile('tarou.txt', '/')).to.eventually.be.a('string');
            });
            it('should be rejected(1)', function() {
                return db.createFile().should.be.rejectedWith(Error, 'filename is required.');
            });
            it('should be rejected(2)', function() {
                return db.createFile('hoge.txt', '').should.be.rejectedWith(Error, 'parameter "directory" is undefined');
            });
            it('should be rejected(3)', function() {
                return db.createFile(null, '/').should.be.rejectedWith(Error, 'filename is required.');
            });
        });


        var removeFile = `
        /**
        * removeFile
        * ファイルを削除する
        * @param {string} fileName
        * @param {string} parentDirectory
        * @returns {promise} ファイル削除に成功すればresolve、失敗すればreject
        */
        `;

        describe(removeFile, function() {
            before(function() {
                return db.init()
                .then(function() {
                    return db.test1();
                });
            });
            it('should be resolved when removeFile() successed', function() {
                return db.removeFile('hoge.txt', '/')
                .then(function() {
                    return db.existFile('hoge.txt', '/')
                })
                .should.be.rejectedWith(Error, '"hoge.txt" does not exist in "/" directory.');
            });
        });


        var createDirectory = `
        /**
        * createFile
        *
        * @param directory
        * @returns {promise}
        */
        `;
        describe(createDirectory, function() {
            before(function() {
                return db.init()
                .then(function() {
                    return db.test1();
                })
                .then(function() {
                    return db.test2();
                })
                .then(function() {
                    return db.createDirectory('/one/hoge');
                });
            });

            it('should be rejected when directory has already existed', function() {
                return db.createDirectory('/one/two').should.be.rejectedWith(Error, '"/one/two" directory exists.');
            });
            it('should be rejected when parent directory does not exist.', function() {
                return db.createDirectory('/four/hoge').should.be.rejectedWith(Error, '"/four" directory doesn\'t exist.');
            });
            it('should be rejected when the same name file has already existed.', function() {
                return db.createDirectory('/hoge.txt').should.be.rejectedWith(Error, '"hoge.txt" has already existed in "/" directory.');
            });
            it('should be rejected when the /one/hoge directory has already existed', function() {
                return db.notExistDirectory('/one/hoge').should.be.rejectedWith(Error, '"/one/hoge" directory exists.');
            });
            it('should be rejected(1)', function() {
                return db.createDirectory('hoge').should.be.rejectedWith(Error, 'directory name should start "/" so that it is absolute path including root.');
            });
            it('should be rejected(2)', function() {
                return db.createDirectory().should.be.rejectedWith(Error, 'directory name should start "/" so that it is absolute path including root.');
            });
        });

        var getFiles = `
        /**
        * getFiles
        * ディレクトリのfileIdを取得する 
        * @param {string} fileId ディレクトリのfileId 
        * @returns {promise} resolve([Array]<string>fileId) 引数で与えられたディレクトリを直接の親に持つファイルのレコードの配列を返す。
        */
        `;
        describe(createDirectory, function() {
            before(function() {
                return db.init()
                .then(function() {
                    return db.test1();
                })
                .then(function() {
                    return db.test2();
                });
            });

            it('should be rejected when calling with invalid fileId', function() {
                return db.getFiles('').should.be.rejected;
            });
            it('should be resolved(1)', function() {
                return expect(db.getFiles('1f83f620-c1ed-11e5-9657-7942989daa00')).to.eventually.be.an('array');
            });
            it('should be resolved(2)', function() {
                return expect(db.getFiles('1f83f620-c1ed-11e5-9657-7942989daa04')).to.eventually.be.length(2);
            });
        });

        var getDirectories = `
        /**
        * getDirectories
        * ディレクトリのfileIdを取得する 
        * @param {string} fileId ディレクトリのfileId 
        * @returns {promise} resolve([Array]<string>fileId) 引数で与えられたディレクトリを直接の親に持つファイルのレコードの配列を返す。
        */
        `;
        describe(createDirectory, function() {
            before(function() {
                return db.init()
                .then(function() {
                    return db.test1();
                })
                .then(function() {
                    return db.test2();
                });
            });

            it('should be resolved', function() {
                return expect(db.getDirectories()).to.eventually.be.length(4);
            });
        });

    });
})();
