package com.yuji.ef.utility;

import java.net.ConnectException;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.List;

import javax.net.ssl.SSLPeerUnverifiedException;

import org.apache.http.conn.ConnectTimeoutException;
import org.apache.thrift.TException;
import org.apache.thrift.transport.TTransportException;

import com.evernote.client.oauth.android.EvernoteSession;
import com.evernote.edam.error.EDAMNotFoundException;
import com.evernote.edam.error.EDAMSystemException;
import com.evernote.edam.error.EDAMUserException;
import com.evernote.edam.notestore.NoteFilter;
import com.evernote.edam.notestore.NoteList;
import com.evernote.edam.notestore.NoteStore;
import com.evernote.edam.type.Note;
import com.evernote.edam.type.NoteSortOrder;
import com.evernote.edam.type.Notebook;
import com.evernote.edam.type.Tag;

public class EvernoteUtil {
	private static final int N = 50;
	//private String username;
	//private String password;
	private EvernoteSession session;

	public enum Error {
		NONE, CONNECT, INVALID_AUTH, NOT_FOUND, OTHER
	};

	private Error errorCode;

	public static EvernoteUtil getInstance() {
		if (instance == null) {
			instance = new EvernoteUtil();
			instance.init();
		}
		return instance;
	}

	private static EvernoteUtil instance = null;

	private EvernoteUtil() {
		errorCode = Error.NONE;
	}

	private String authenticationToken = null;
	private NoteStore.Client noteStore = null;

	//public void setConfig(String username, String password) {
	//	this.username = username;
	//	this.password = password;
	//	authenticationToken = null;
	//	noteStore = null;
	//}

	private void init() {

	}

	public Note getNote(String guid) throws EDAMNotFoundException {
		boolean withContent = true;
		boolean withResourcesData = false;
		boolean withResourcesRecognition = false;
		boolean withResourcesAlternateData = false;
		Note note = null;
		errorCode = Error.OTHER;
		try {
			String token = getAuthenticationToken();
			if (token == null) {
				return null;
			}
			note = noteStore.getNote(token, guid, withContent,
					withResourcesData, withResourcesRecognition,
					withResourcesAlternateData);
			errorCode = Error.NONE;
		} catch (EDAMUserException e) {
			Debug.d(this, null, e);
		} catch (EDAMSystemException e) {
			Debug.d(this, null, e);
		} catch (EDAMNotFoundException e) {
			Debug.d(this, null, e);
			throw e;
		} catch (TException e) {
			Debug.d(this, null, e);

			if (isNetworkError(e)) {
				errorCode = Error.CONNECT;
			}
		}
		return note;
	}

	private NoteFilter getDefaultFilter() {
		// 検索条件として、検索語なし、更新日順ソートを指定
		NoteFilter filter = new NoteFilter();
		filter.setOrder(NoteSortOrder.UPDATED.getValue());
		filter.setAscending(false);

		return filter;
	}

	public List<NoteList> getNoteList() {
		NoteFilter filter = getDefaultFilter();
		return getNoteList(filter);
	}

	public List<NoteList> getNoteListByNotebook(String notebookGuid) {
		NoteFilter filter = getDefaultFilter();
		filter.setNotebookGuid(notebookGuid);
		return getNoteList(filter);
	}

	public List<NoteList> getNoteListByTag(String tagGuid) {
		NoteFilter filter = getDefaultFilter();
		List<String> tagGuids = new ArrayList<String>();
		tagGuids.add(tagGuid);
		filter.setTagGuids(tagGuids);
		return getNoteList(filter);
	}

	public List<NoteList> getNoteList(NoteFilter filter) {
		errorCode = Error.OTHER;

		List<NoteList> list = new ArrayList<NoteList>();
		try {
			String token = getAuthenticationToken();
			if (token == null) {
				return null;
			}
			int index = 0;
			while (true) {
				NoteList noteList = noteStore
						.findNotes(token, filter, index, N);
				int size = noteList.getNotesSize();

				if (size <= 0) {
					break;
				}
				list.add(noteList);
				if (size < N) {
					break;
				}
				index += size;
			}

			errorCode = Error.NONE;
		} catch (EDAMUserException e) {
			Debug.d(this, null, e);
			list = null;
		} catch (EDAMSystemException e) {
			Debug.d(this, null, e);
			list = null;
		} catch (EDAMNotFoundException e) {
			Debug.d(this, null, e);
			list = null;
		} catch (TException e) {
			Debug.d(this, null, e);

			if (isNetworkError(e)) {
				errorCode = Error.CONNECT;
			}
			list = null;
		}
		return list;
	}

	public List<Notebook> getNoteBookList() {
		errorCode = Error.OTHER;

		List<Notebook> noteBookList = null;
		try {
			String token = getAuthenticationToken();
			if (token == null) {
				return null;
			}
			noteBookList = noteStore.listNotebooks(token);
			errorCode = Error.NONE;
		} catch (EDAMUserException e) {
			Debug.d(this, null, e);
		} catch (EDAMSystemException e) {
			Debug.d(this, null, e);
		} catch (TException e) {
			Debug.d(this, null, e);

			if (isNetworkError(e)) {
				errorCode = Error.CONNECT;
			}
		}
		return noteBookList;
	}

	public List<Tag> getTagList() {
		errorCode = Error.OTHER;

		List<Tag> tagList = null;
		try {
			String token = getAuthenticationToken();
			if (token == null) {
				return null;
			}
			tagList = noteStore.listTags(token);
			errorCode = Error.NONE;
		} catch (EDAMUserException e) {
			Debug.d(this, null, e);
		} catch (EDAMSystemException e) {
			Debug.d(this, null, e);
		} catch (TException e) {
			Debug.d(this, null, e);

			if (isNetworkError(e)) {
				errorCode = Error.CONNECT;
			}
		}
		return tagList;
	}

	public Note updateNoteContext(String guid, String title, String text) {
		errorCode = Error.OTHER;

		Note note = null;
		try {
			note = getNote(guid);
		} catch (EDAMNotFoundException e) {
			errorCode = Error.NOT_FOUND;
		}
		if (note == null) {
			return null;
		}
		String content = note.getContent();

		// #27970
		// 見つからない場合
		String endTag = "</en-note>";
		int pos = content.indexOf(endTag);
		if (pos < 0) {
			return null;
		}
		StringBuffer sb = new StringBuffer();
		sb.append(content.substring(0, pos));
		if (title != null) {
			sb.append(title);
		}
		sb.append("<div>");
		sb.append(text);
		sb.append("</div>");
		sb.append(content.substring(pos));
		content = sb.toString();
		note.setContent(content);

		Note n = updateNote(note);

		errorCode = Error.NONE;
		return n;
	}

	public Note updateNote(Note note) {
		errorCode = Error.OTHER;
		Note n = null;
		try {
			String token = getAuthenticationToken();
			if (token == null) {
				return null;
			}
			n = noteStore.updateNote(token, note);
			errorCode = Error.NONE;
		} catch (EDAMUserException e) {
			Debug.d(this, null, e);
		} catch (EDAMSystemException e) {
			Debug.d(this, null, e);
		} catch (EDAMNotFoundException e) {
			e.printStackTrace();
		} catch (TException e) {
			Debug.d(this, null, e);

			if (isNetworkError(e)) {
				errorCode = Error.CONNECT;
			}
		}
		return n;
	}

	public boolean isLoggedIn() {
		if (session == null){
			return false;
		}
		return session.isLoggedIn();
	}

	private String getAuthenticationToken() {
		if (authenticationToken == null) {
			authenticationToken = session.getAuthToken();
			if (authenticationToken == null) {
				return null;
			}
			try {
				noteStore = session.createNoteStore();
			} catch (TTransportException e) {
				authenticationToken = null;
			}
		}
		return authenticationToken;
	}

	public Error getErrorCode() {
		return errorCode;
	}

	public void setSession(EvernoteSession session) {
		this.session = session;
	}

	public void createNotes(String guid, String title, int size){
		Note note;
		String token = getAuthenticationToken();
		if (token == null) {
			return;
		}
		Notebook notebook;
		try {
			notebook = noteStore.getNotebook(token, guid);
			for (int i = 0; i < size; i++){
				note = createNote(notebook, title + i);
				if (note == null){
					return;
				}
			}
		} catch (EDAMUserException e) {
			Debug.d(this,  null, e);
		} catch (EDAMSystemException e) {
			Debug.d(this,  null, e);
		} catch (EDAMNotFoundException e) {
			Debug.d(this,  null, e);
		} catch (TException e) {
			Debug.d(this,  null, e);
		}
	}
	
	public Note createNote(Notebook notebook, String title){
		String authToken = getAuthenticationToken();
		if (authToken == null){
			return null;
		}
		Note note = new Note();  
		note.setTitle(title);  
		String content = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>"  
				+ "<!DOCTYPE en-note SYSTEM \"http://xml.evernote.com/pub/enml.dtd\">"  
				+ "<en-note>Hello Evernote.</en-note>";  
		note.setContent(content);  
		note.setNotebookGuid(notebook.getGuid());  
		Note newNote = null;
		try {
			newNote = noteStore.createNote(authToken, note);
		} catch (EDAMUserException e) {
			Debug.d(this,  null, e);
		} catch (EDAMSystemException e) {
			Debug.d(this,  null, e);
		} catch (EDAMNotFoundException e) {
			Debug.d(this,  null, e);
		} catch (TException e) {
			Debug.d(this,  null, e);
		}
		return newNote;
	}

	public void createHelpData(){
		String token = getAuthenticationToken();

		try {
			Notebook food = createHelpBook(token, "食べ歩き");
			createHelpNote(token, food, "ラーメン");
			createHelpNote(token, food, "寿司");
			createHelpNote(token, food, "カレー");
			
			Notebook  travelJapan = createHelpBook(token, "日本");

			Notebook  travelForeign = createHelpBook(token, "海外");
			createHelpNote(token, travelForeign, "ハワイ");
			createHelpNote(token, travelForeign, "タイ");
			createHelpNote(token, travelForeign, "バンコク");

			Notebook  computer = createHelpBook(token, "コンピュータ");
			createHelpNote(token, computer, "Mac");

			Notebook  jobProblem = createHelpBook(token, "懸案");
			Notebook  jobOrder = createHelpBook(token, "発注");
			Notebook  memoTodo = createHelpBook(token, "ToDo");
			Notebook  memoSchedule = createHelpBook(token, "スケジュール");
			
		} catch (EDAMUserException e) {
			Debug.d(this,  null, e);
		} catch (EDAMSystemException e) {
			Debug.d(this,  null, e);
		} catch (TException e) {
			Debug.d(this,  null, e);
		} catch (EDAMNotFoundException e) {
			Debug.d(this,  null, e);
		}
	}

	public Notebook createHelpBook(String token, String name) throws EDAMUserException, EDAMSystemException, TException{
		Notebook notebook = new Notebook();
		notebook.setName(name);
		return noteStore.createNotebook(token, notebook);
	}
	
	public Note createHelpNote(String toke, Notebook notebook, String name) throws EDAMUserException, EDAMSystemException, EDAMNotFoundException, TException{
		return createNote(notebook, name);
	}	
	
	private boolean isNetworkError(Exception e){
//		Throwable t = e.getCause();
//		if (t instanceof ConnectException){
//			return true;
//		}
//		if (t instanceof ConnectTimeoutException){
//			return true;
//		}
//		if (t instanceof SocketTimeoutException){
//			return true;
//		}
//		if (t instanceof UnknownHostException){
//			return true;
//		}
//		if (t instanceof SSLPeerUnverifiedException){
//			return true;
//		}
//		return false;
		return true;
	}
}
